/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.ametys.plugins.repository.dom;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Locale;
import java.util.Map;

import org.w3c.dom.Element;
import org.w3c.dom.Node;

import org.ametys.core.util.dom.AbstractWrappingAmetysElement;
import org.ametys.core.util.dom.AmetysAttribute;
import org.ametys.core.util.dom.AmetysText;
import org.ametys.plugins.repository.metadata.MultilingualString;

/**
 * DOM layer for a locale value of a {@link MultilingualString} metadata
 */
public class LocaleStringElement extends AbstractWrappingAmetysElement<Locale>
{
    private MultilingualString _multilingualString;
    private Locale _locale;
    
    /**
     * Constructor.
     * @param locale the underlying {@link Locale}.
     * @param multilingualString the parent {@link MultilingualString} holding the locale values
     */
    public LocaleStringElement(Locale locale, MultilingualString multilingualString)
    {
        this(locale, multilingualString, null);
    }

    /**
     * Constructor.
     * @param locale the locale of underlying metadata
     * @param multilingualString the parent {@link MultilingualString} holding the languages values
     * @param parent the parent {@link Element}.
     */
    public LocaleStringElement(Locale locale, MultilingualString multilingualString, MultilingualStringElement parent)
    {
        super(locale, parent);
        _locale = locale;
        _multilingualString = multilingualString;
    }
    
    @Override
    public String getTagName()
    {
        return _object.getLanguage();
    }
    
    @Override
    protected Map<String, AmetysAttribute> _lookupAttributes()
    {
        return new HashMap<>();
    }
    
    @Override
    public boolean hasChildNodes()
    {
        return _multilingualString.hasLocale(_locale);
    }
    
    @Override
    public Node getFirstChild()
    {
        return new AmetysText(_multilingualString.getValue(_locale), this);
    }
    
    @Override
    public Node getNextSibling()
    {
        if (_parent == null)
        {
            return null;
        }
        
        boolean isNext = false;
        Locale nextSibling = null;
        
        Iterator<Locale> brothers = _multilingualString.getLocales().iterator();
        while (nextSibling == null && brothers.hasNext())
        {
            Locale brother = brothers.next();
            
            if (isNext)
            {
                nextSibling = brother;
            }
            else if (_object.getLanguage().equals(brother.getLanguage()))
            {
                isNext = true;
            }
        }
        
        if (nextSibling == null)
        {
            return null;
        }
        
        return new LocaleStringElement(nextSibling, _multilingualString, (MultilingualStringElement) getParentNode());
    }
}
