/*
 *  Copyright 2011 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.ametys.plugins.repository.dom;

import java.io.File;
import java.util.HashMap;
import java.util.Map;

import org.w3c.dom.Element;
import org.w3c.dom.Node;

import org.ametys.core.util.dom.AbstractWrappingAmetysElement;
import org.ametys.core.util.dom.AmetysAttribute;
import org.ametys.core.util.dom.AmetysText;
import org.ametys.plugins.repository.metadata.CompositeMetadata;
import org.ametys.plugins.repository.metadata.CompositeMetadata.MetadataType;

/**
 * DOM layer on top if a {@link File} hierarchy.
 */
public class MetadataElement extends AbstractWrappingAmetysElement<String>
{
    CompositeMetadata _metadataHolder;
    private String _defaultValue;
    
    /**
     * Constructor.
     * @param metadataName the name of underlying metadata
     * @param metadataHolder the parent {@link CompositeMetadata}
     */
    public MetadataElement(String metadataName, CompositeMetadata metadataHolder)
    {
        this(metadataName, metadataHolder, null, "");
    }
    
    /**
     * Constructor.
     * @param metadataName the name of underlying metadata
     * @param metadataHolder the parent {@link CompositeMetadata}
     * @param defaultValue the default value
     */
    public MetadataElement(String metadataName, CompositeMetadata metadataHolder, String defaultValue)
    {
        this(metadataName, metadataHolder, null, defaultValue);
    }
    
    /**
     * Constructor.
     * @param metadataName the name of underlying metadata
     * @param metadataHolder the parent {@link CompositeMetadata}
     * @param parent the parent {@link Element}.
     */
    public MetadataElement(String metadataName, CompositeMetadata metadataHolder, CompositeMetadataElement parent)
    {
        this(metadataName, metadataHolder, parent, "");
    }

    /**
     * Constructor.
     * @param metadataName the name of underlying metadata
     * @param metadataHolder the parent {@link CompositeMetadata}
     * @param parent the parent {@link Element}.
     * @param defaultValue the default value
     */
    public MetadataElement(String metadataName, CompositeMetadata metadataHolder, CompositeMetadataElement parent, String defaultValue)
    {
        super(metadataName, parent);
        _metadataHolder = metadataHolder;
        _defaultValue = defaultValue;
    }
    
    @Override
    public String getTagName()
    {
        return _object;
    }
    
    @Override
    protected Map<String, AmetysAttribute> _lookupAttributes()
    {
        Map<String, AmetysAttribute> result = new HashMap<>();
        MetadataType type = _metadataHolder.getType(_object);
        result.put("type", new AmetysAttribute("type", "type", null, type.toString(), this));
        return result;
    }
    
    @Override
    public boolean hasChildNodes()
    {
        return _object.length() > 0;
    }
    
    @Override
    public Node getFirstChild()
    {
        return new AmetysText(_metadataHolder.getString(_object, _defaultValue), this);
    }
    
    @Override
    public Node getNextSibling()
    {
        if (_parent == null)
        {
            return null;
        }
        
        boolean isNext = false;
        String nextSibling = null;
        int i = 0;
        
        String[] brothers = _metadataHolder.getMetadataNames();
        while (nextSibling == null && i < brothers.length)
        {
            String brother = brothers[i++];
            
            if (isNext)
            {
                nextSibling = brother;
            }
            else if (_object.equals(brother))
            {
                isNext = true;
            }
        }
        
        if (nextSibling == null)
        {
            return null;
        }
        
        MetadataType type = _metadataHolder.getType(nextSibling);
        switch (type)
        {
            case COMPOSITE:
                return new CompositeMetadataElement(_metadataHolder.getCompositeMetadata(nextSibling), nextSibling, (CompositeMetadataElement) getParentNode());
            case USER:
                return new UserMetadataElement(_metadataHolder.getCompositeMetadata(nextSibling), nextSibling, (CompositeMetadataElement) getParentNode());
            case MULTILINGUAL_STRING:
                return new MultilingualStringElement(_metadataHolder.getMultilingualString(nextSibling), nextSibling, (CompositeMetadataElement) getParentNode());
            default:
                return new MetadataElement(nextSibling, _metadataHolder, (CompositeMetadataElement) getParentNode());
        }
    }
}
