/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.jcr;

import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.jcr.AccessDeniedException;
import javax.jcr.ItemNotFoundException;
import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.PathNotFoundException;
import javax.jcr.Repository;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.Value;
import javax.jcr.lock.Lock;
import javax.jcr.lock.LockManager;
import javax.jcr.query.Query;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.jackrabbit.JcrConstants;
import org.apache.jackrabbit.util.ISO9075;
import org.apache.jackrabbit.util.Text;
import org.slf4j.Logger;

import org.ametys.core.group.GroupIdentity;
import org.ametys.core.right.ProfileAssignmentStorage.AnonymousOrAnyConnectedKeys;
import org.ametys.core.right.ProfileAssignmentStorage.UserOrGroup;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.util.LambdaUtils;
import org.ametys.plugins.repository.ACLAmetysObject;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.ModifiableACLAmetysObject;
import org.ametys.plugins.repository.ModifiableACLAmetysObjectProfileAssignmentStorage;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.provider.AbstractRepository;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.LogicalOperator;
import org.ametys.plugins.repository.query.expression.OrExpression;
import org.ametys.runtime.plugin.component.LogEnabled;

/**
 * Helper for implementing {@link ModifiableACLAmetysObject} in JCR under its node.
 */
public class ACLJCRAmetysObjectHelper implements Component, Serviceable, LogEnabled
{
    /** The AmetysObject resolver */
    protected static AmetysObjectResolver _resolver;
    /** The repository */
    protected static Repository _repository;
    
    private static final String __NODE_NAME_ROOT_ACL = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":acl";
    private static final String __NODETYPE_ROOT_ACL = RepositoryConstants.NAMESPACE_PREFIX + ":acl";
    
    private static final String __NODE_NAME_ACL_USERS = "users";
    private static final String __NODE_NAME_ACL_GROUPS = "groups";
    private static final String __NODETYPE_ACL_USER = RepositoryConstants.NAMESPACE_PREFIX + ":acl-user";
    private static final String __NODETYPE_ACL_GROUP = RepositoryConstants.NAMESPACE_PREFIX + ":acl-group";
    
    private static final String __PROPERTY_NAME_ALLOWED_ANY_CONNECTED_PROFILES = RepositoryConstants.NAMESPACE_PREFIX + ":allowed-any-connected-profiles";
    private static final String __PROPERTY_NAME_DENIED_ANY_CONNECTED_PROFILES = RepositoryConstants.NAMESPACE_PREFIX + ":denied-any-connected-profiles";
    private static final String __PROPERTY_NAME_ALLOWED_ANONYMOUS_PROFILES = RepositoryConstants.NAMESPACE_PREFIX + ":allowed-anonymous-profiles";
    private static final String __PROPERTY_NAME_DENIED_ANONYMOUS_PROFILES = RepositoryConstants.NAMESPACE_PREFIX + ":denied-anonymous-profiles";
    
    private static final String __PROPERTY_NAME_ALLOWED_PROFILES = RepositoryConstants.NAMESPACE_PREFIX + ":allowed-profiles";
    private static final String __PROPERTY_NAME_DENIED_PROFILES = RepositoryConstants.NAMESPACE_PREFIX + ":denied-profiles";
    
    private static final String __PROPERTY_NAME_DISALLOW_INHERITANCE = RepositoryConstants.NAMESPACE_PREFIX + ":disallow-inheritance";
    
    private static final Map<AnonymousOrAnyConnectedKeys, Set<String>> __ANONYMOUS_OR_ANYCONNECTEDUSER_NORIGHT = Map.of(
            AnonymousOrAnyConnectedKeys.ANONYMOUS_ALLOWED, Set.of(),
            AnonymousOrAnyConnectedKeys.ANONYMOUS_DENIED, Set.of(),
            AnonymousOrAnyConnectedKeys.ANYCONNECTEDUSER_ALLOWED, Set.of(),
            AnonymousOrAnyConnectedKeys.ANYCONNECTEDUSER_DENIED, Set.of());
    private static final Map<UserOrGroup, Set<String>> __USER_OR_GROUP_NORIGHT = Map.of(
            UserOrGroup.ALLOWED, Set.of(),
            UserOrGroup.DENIED, Set.of());
 
    
    private static Logger _logger;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _repository = (Repository) manager.lookup(AbstractRepository.ROLE);
    }
    
    public void setLogger(Logger logger)
    {
        _logger = logger;
    }
    
    
    /* -------------- */
    /* HAS PERMISSION */
    /* -------------- */
    
    private static Set<String> _convertNodeToPath(Set<? extends Object> rootNodes)
    {
        return rootNodes.stream().filter(JCRAmetysObject.class::isInstance).map(JCRAmetysObject.class::cast).map(LambdaUtils.wrap(ao -> ISO9075.encodePath(ao.getNode().getPath()))).collect(Collectors.toSet());
    }
    
    
    /**
     * Returns some profiles that are matching if any ACL Ametys object has one of the given profiles as allowed for the user
     * @param user The user
     * @param profileIds The ids of the profiles to check
     * @param rootNodes The JCR root nodes where starts the query search (must be something like "//element(myNode, ametys:collection)"), it will be the beginning of the JCR query. Can be null to not restrict the search.
     * @return If the Set is empty, it means the user has no matching profile.<br>
     *         If the Set is non empty, it contains at least one of the given profile BUT it may not contains all the matching profiles for the user AND it can contains some other profiles that were not in the given profiles
     */
    public static Set<String> hasUserAnyAllowedProfile(Set<? extends Object> rootNodes, UserIdentity user, Set<String> profileIds)
    {
        Expression expr = new AllowedProfileExpression(profileIds.toArray(new String[profileIds.size()]));
        for (String rootPath : _convertNodeToPath(rootNodes))
        {
            NodeIterator nodes = getACLUsers(user, rootPath, expr);
            
            if (nodes.hasNext())
            {
                // To be complete we could loop on all results, but we only want to answer the question and return additional data if we can
                Node userNode = nodes.nextNode();
                return _getProperty(userNode, __PROPERTY_NAME_ALLOWED_PROFILES);
            }
        }
        return Set.of();
    }
    
    /**
     * Returns some profiles that are matching if any ACL Ametys object has one of the given profiles as allowed for the group
     * @param groups The groups
     * @param profileIds The ids of the profiles
     * @param rootNodes The JCR root nodes where starts the query search (must be something like "//element(myNode, ametys:collection)"), it will be the beginning of the JCR query. Can be null to not restrict the search.
     * @return If the Set is empty, it means the group has no matching profile.<br>
     *         If the Set is non empty, it contains at least one of the given profile BUT it may not contains all the matching profiles for the group AND it can contains some other profiles that were not in the given profiles
     */
    public static Set<String> hasGroupAnyAllowedProfile(Set<? extends Object> rootNodes, Set<GroupIdentity> groups, Set<String> profileIds)
    {
        if (!groups.isEmpty())
        {
            Expression expr = new AllowedProfileExpression(profileIds.toArray(new String[profileIds.size()]));
            for (String rootPath : _convertNodeToPath(rootNodes))
            {
                // Approximative query (to be fast)
                NodeIterator nodes = _getApprochingACLGroups(groups, rootPath, expr);
                
                while (nodes.hasNext())
                {
                    Node groupNode = nodes.nextNode();
    
                    // As the query was a fast approximative request, we now check if the result is fine
                    String groupId;
                    String directoryId;
                    try
                    {
                        groupId = Text.unescapeIllegalJcrChars(groupNode.getName());
                        directoryId = groupNode.getParent().getName();
                    }
                    catch (RepositoryException ex)
                    {
                        throw new AmetysRepositoryException("An error occured getting group information", ex);
                    }
                    
                    if (groups.contains(new GroupIdentity(groupId, directoryId)))
                    {
                        // To be complete we could loop on all results, but we only want to answer the question and return additional data if we can
                        return _getProperty(groupNode, __PROPERTY_NAME_ALLOWED_PROFILES);
                    }
                }
            }
        }
        
        return Set.of();
    }
    
    /**
     * Returns some profiles that are matching if any ACL Ametys object has one of the given profiles as allowed for any connected user
     * @param profileIds The ids of the profiles
     * @param rootNodes The JCR root nodes where starts the query search (must be something like "//element(myNode, ametys:collection)"), it will be the beginning of the JCR query. Can be null to not restrict the search.
     * @return If the Set is empty, it means any connected user has no matching profile.<br>
     *         If the Set is non empty, it contains at least one of the given profile BUT it may not contains all the matching profiles for anyconnected user AND it can contains some other profiles that were not in the given profiles
     */
    public static Set<String> hasAnyConnectedAnyAllowedProfile(Set<? extends Object> rootNodes, Set<String> profileIds)
    {
        Expression expr = new AnyConnectedAllowedProfileExpression(profileIds.toArray(new String[profileIds.size()]));
        for (String rootPath : _convertNodeToPath(rootNodes))
        {
            NodeIterator nodes = getACLRoots(rootPath, expr);
            
            if (nodes.hasNext())
            {
                // To be complete we could loop on all results, but we only want to answer the question and return additional data if we can
                Node aclNode = nodes.nextNode();
                return _getProperty(aclNode, __PROPERTY_NAME_ALLOWED_ANY_CONNECTED_PROFILES);
            }
        }
        return Set.of();
    }
    
    
    /**
     * Returns some profiles that are matching if any ACL Ametys object has one of the given profiles as allowed for anonymous
     * @param profileIds The ids of the profiles
     * @param rootNodes The JCR root nodes where starts the query search (must be something like "//element(myNode, ametys:collection)"), it will be the beginning of the JCR query. Can be null to not restrict the search.
     * @return If the Set is empty, it means anonymous has no matching profile.<br>
     *         If the Set is non empty, it contains at least one of the given profile BUT it may not contains all the matching profiles for anonymous AND it can contains some other profiles that were not in the given profiles
     */
    public static Set<String> hasAnonymousAnyAllowedProfile(Set<? extends Object> rootNodes, Set<String> profileIds)
    {
        Expression expr = new AnonymousAllowedProfileExpression(profileIds.toArray(new String[profileIds.size()]));
        for (String rootPath : _convertNodeToPath(rootNodes))
        {
            NodeIterator nodes = getACLRoots(rootPath, expr);
            
            if (nodes.hasNext())
            {
                // To be complete we could loop on all results, but we only want to answer the question and return additional data if we can
                Node aclNode = nodes.nextNode();
                return _getProperty(aclNode, __PROPERTY_NAME_ALLOWED_ANONYMOUS_PROFILES);
            }
        }
        return Set.of();
    }
    
    /**
     * Gets all contexts with stored profiles (allowed or denied) for anonymous or any connected user and for each, a description of the permission
     * @param rootNodes The JCR root nodes where starts the query search (must be something like "//element(myNode, ametys:collection)"), it will be the beginning of the JCR query
     * @return a map associating a context object to the stored profile for each permission
     */
    public static Map<Object, Map<AnonymousOrAnyConnectedKeys, Set<String>>> getAllProfilesForAnonymousAndAnyConnectedUser(Set< ? extends Object> rootNodes)
    {
        Map<Object, Map<AnonymousOrAnyConnectedKeys, Set<String>>> result = new HashMap<>();
        // Only retrieve node with assignments to anonymous or any connected
        Expression predicate = new OrExpression(
                () -> "@" + __PROPERTY_NAME_ALLOWED_ANONYMOUS_PROFILES,
                () -> "@" + __PROPERTY_NAME_DENIED_ANONYMOUS_PROFILES,
                () -> "@" + __PROPERTY_NAME_ALLOWED_ANY_CONNECTED_PROFILES,
                () -> "@" + __PROPERTY_NAME_DENIED_ANY_CONNECTED_PROFILES
                );
        
        for (String rootPath : _convertNodeToPath(rootNodes))
        {
            NodeIterator nodes = getACLRoots(rootPath, predicate);
            
            while (nodes.hasNext())
            {
                Node aclNode = nodes.nextNode();
                try
                {
                    Map<AnonymousOrAnyConnectedKeys, Set<String>> aoResult = new HashMap<>();
                    Set<String> allowedAnonymous = _getProperty(aclNode, __PROPERTY_NAME_ALLOWED_ANONYMOUS_PROFILES);
                    if (!allowedAnonymous.isEmpty())
                    {
                        aoResult.put(AnonymousOrAnyConnectedKeys.ANONYMOUS_ALLOWED, allowedAnonymous);
                    }
                    
                    Set<String> deniedAnonymous = _getProperty(aclNode, __PROPERTY_NAME_DENIED_ANONYMOUS_PROFILES);
                    if (!deniedAnonymous.isEmpty())
                    {
                        aoResult.put(AnonymousOrAnyConnectedKeys.ANONYMOUS_DENIED, deniedAnonymous);
                    }
                    Set<String> allowedAny = _getProperty(aclNode, __PROPERTY_NAME_ALLOWED_ANY_CONNECTED_PROFILES);
                    if (!allowedAny.isEmpty())
                    {
                        aoResult.put(AnonymousOrAnyConnectedKeys.ANYCONNECTEDUSER_ALLOWED, allowedAny);
                    }
                    Set<String> deniedAny = _getProperty(aclNode, __PROPERTY_NAME_DENIED_ANY_CONNECTED_PROFILES);
                    if (!deniedAny.isEmpty())
                    {
                        aoResult.put(AnonymousOrAnyConnectedKeys.ANYCONNECTEDUSER_DENIED, deniedAny);
                    }
                    
                    if (!aoResult.isEmpty())
                    {
                        AmetysObject ao = _getAmetysObjectFromACLNode(aclNode);
                        result.put(ao, aoResult);
                    }
                }
                catch (RepositoryException e)
                {
                    _logger.error("Failed to retrieve object for acl node " + aclNode.toString() + ". The node will be ignored.");
                }
            }
        }
        return result;
    }
    
    /**
     * Gets all context with stored profiles (allowed or denied) for the groups and for each, a description of the permission
     * Gets the groups that have allowed profiles assigned on the given object
     * @param rootNodes The JCR root nodes where starts the query search (must be something like "//element(myNode, ametys:collection)"), it will be the beginning of the JCR query
     * @param groups The groups to get profiles for.
     * @return The map of context with their assigned permissions
     */
    public static Map<Object, Map<GroupIdentity, Map<UserOrGroup, Set<String>>>> getAllProfilesForGroups(Set< ? extends Object> rootNodes, Set<GroupIdentity> groups)
    {
        Map<Object, Map<GroupIdentity, Map<UserOrGroup, Set<String>>>> result = new HashMap<>();
        if (!groups.isEmpty())
        {
            for (String rootPath : _convertNodeToPath(rootNodes))
            {
                // Approximative query (to be fast)
                NodeIterator nodes = _getApprochingACLGroups(groups, rootPath, null);
                
                while (nodes.hasNext())
                {
                    Node groupNode = nodes.nextNode();
                    
                    // As the query was a fast approximative request, we now check if the result is fine
                    String groupId;
                    String directoryId;
                    try
                    {
                        groupId = Text.unescapeIllegalJcrChars(groupNode.getName());
                        directoryId = groupNode.getParent().getName();
                    }
                    catch (RepositoryException ex)
                    {
                        throw new AmetysRepositoryException("An error occured getting group information", ex);
                    }
                    
                    GroupIdentity currentGroup = new GroupIdentity(groupId, directoryId);
                    if (groups.contains(currentGroup))
                    {
                        try
                        {
                            
                            // Determine the group permissions
                            Map<UserOrGroup, Set<String>> groupPermissions = new HashMap<>();
                            
                            Set<String> allowedProfiles = _getProperty(groupNode, __PROPERTY_NAME_ALLOWED_PROFILES);
                            if (!allowedProfiles.isEmpty())
                            {
                                groupPermissions.put(UserOrGroup.ALLOWED, allowedProfiles);
                            }
                            Set<String> deniedProfiles = _getProperty(groupNode, __PROPERTY_NAME_DENIED_PROFILES);
                            if (!deniedProfiles.isEmpty())
                            {
                                groupPermissions.put(UserOrGroup.DENIED, deniedProfiles);
                            }
                            
                            // Only add actual permissions to the result
                            if (!groupPermissions.isEmpty())
                            {
                                AmetysObject ao = _getAmetysObjectFromACLNode(groupNode);
                                // The ametys object could already be in the result map having permissions from an other group
                                Map<GroupIdentity, Map<UserOrGroup, Set<String>>> objectPermissions = result.computeIfAbsent(ao, k -> new HashMap<>());
                                // There can only be one node per group, so we don't need to retrieve existing value
                                objectPermissions.put(currentGroup, groupPermissions);
                            }
                        }
                        catch (RepositoryException e)
                        {
                            _logger.error("Failed to retrieve object for group acl node " + groupNode.toString() + ". The node will be ignored.");
                        }
                    }
                }
            }
        }
        return result;
    }
    
    /**
     * Gets all context with stored profiles (allowed or denied) for the user and for each, a description of the permission
     * @param rootNodes The JCR root nodes where starts the query search (must be something like "//element(myNode, ametys:collection)"), it will be the beginning of the JCR query
     * @param user The user to get profiles for.
     * @return The map of context with their assigned allowed/denied profiles
     */
    public static Map<Object, Map<UserOrGroup, Set<String>>> getAllProfilesForUser(Set< ? extends Object> rootNodes, UserIdentity user)
    {
        Map<Object, Map<UserOrGroup, Set<String>>> result = new HashMap<>();
        
        for (String rootPath : _convertNodeToPath(rootNodes))
        {
            NodeIterator nodes = getACLUsers(user, rootPath, null);
            
            while (nodes.hasNext())
            {
                Node userNode = nodes.nextNode();
                try
                {
                    Map<UserOrGroup, Set<String>> aoResult = new HashMap<>();
                    Set<String> allowedProfiles = _getProperty(userNode, __PROPERTY_NAME_ALLOWED_PROFILES);
                    if (!allowedProfiles.isEmpty())
                    {
                        aoResult.put(UserOrGroup.ALLOWED, allowedProfiles);
                    }
                    Set<String> deniedProfiles = _getProperty(userNode, __PROPERTY_NAME_DENIED_PROFILES);
                    if (!deniedProfiles.isEmpty())
                    {
                        aoResult.put(UserOrGroup.DENIED, deniedProfiles);
                    }
                    
                    if (!aoResult.isEmpty())
                    {
                        AmetysObject ao = _getAmetysObjectFromACLNode(userNode);
                        result.put(ao, aoResult);
                    }
                }
                catch (RepositoryException e)
                {
                    _logger.error("Failed to retrieve object for user acl node " + userNode.toString() + ". The node will be ignored.");
                }
            }
        }
        return result;
    }
    
    private static AmetysObject _getAmetysObjectFromACLNode(Node node) throws RepositoryException, ItemNotFoundException, AccessDeniedException
    {
        switch (node.getPrimaryNodeType().getName())
        {
            case __NODETYPE_ROOT_ACL:
                return _resolver.resolve(node.getParent(), false);
            case __NODETYPE_ACL_USER:
            case __NODETYPE_ACL_GROUP:
                return _resolver.resolve(node.getParent().getParent().getParent().getParent(), false);
            default:
                return null;
        }
    }
    
    /**
     * Returns all ACL root objects (ametys:acl nodes)
     * @param rootPath The root path to restrict the search. Can be null.
     * @param predicat The predicat expression. Can be null.
     * @return The ACL root objects
     */
    public static NodeIterator getACLRoots (String rootPath, Expression predicat)
    {
        StringBuilder sb = new StringBuilder("/jcr:root");
        
        if (rootPath != null)
        {
            sb.append(rootPath);
        }
        
        sb.append("//element(*, ").append(__NODETYPE_ROOT_ACL).append(")");
        
        if (predicat != null)
        {
            sb.append("[").append(predicat.build()).append("]");
        }
        
        return _query(sb.toString());
    }
    
    /**
     * Returns all ACL objects for a given user (ametys:acl-user nodes)
     * @param user The user
     * @param rootPath The root path to restrict the search. Can be null.
     * @param predicat The predicat expression. Can be null.
     * @return The ACL user objects for user
     */
    public static NodeIterator getACLUsers (UserIdentity user, String rootPath, Expression predicat)
    {
        StringBuilder sb = new StringBuilder("/jcr:root");
        
        if (rootPath != null)
        {
            sb.append(rootPath);
        }
        
        sb.append("//element(*, ").append(__NODETYPE_ROOT_ACL).append(")")
            .append("/").append(__NODE_NAME_ACL_USERS)
            .append("/").append(user.getPopulationId())
            .append("/").append(ISO9075.encode(user.getLogin()));
        
        if (predicat != null)
        {
            sb.append("[").append(predicat.build()).append("]");
        }
        
        String jcrQuery = sb.toString();
        return _query(jcrQuery);
    }
    
    /**
     * Returns all ACL objects for users user (ametys:acl-user nodes)
     * @param rootPath The root path to restrict the search. Can be null.
     * @param predicate The predicate expression. Can be null.
     * @return The ACL user objects for users
     */
    public static NodeIterator getACLUsers (String rootPath, Expression predicate)
    {
        StringBuilder sb = new StringBuilder("/jcr:root");
        
        if (rootPath != null)
        {
            sb.append(rootPath);
        }
        
        sb.append("//element(*, ").append(__NODETYPE_ACL_USER).append(")");
        
        if (predicate != null)
        {
            sb.append("[").append(predicate.build()).append("]");
        }
        
        String jcrQuery = sb.toString();
        return _query(jcrQuery);
    }
    
    /**
     * Returns all ACL objects for users (ametys:acl-user nodes)
     * @param predicat The predicat expression. Can be null.
     * @return The ACL user objects for users
     */
    public static NodeIterator getACLUsers (Expression predicat)
    {
        StringBuilder sb = new StringBuilder();
        
        sb.append("//element(*, ").append(__NODETYPE_ACL_USER).append(")");
        
        if (predicat != null)
        {
            sb.append("[").append(predicat.build()).append("]");
        }
        
        return _query(sb.toString());
    }
    
    /**
     * Returns all ACL objects for groups (ametys:acl-group nodes)
     * @param predicat The predicat expression. Can be null.
     * @return The ACL group objects for groups
     */
    public static NodeIterator getACLGroups (Expression predicat)
    {
        StringBuilder sb = new StringBuilder();
        
        sb.append("//element(*, ").append(__NODETYPE_ACL_GROUP).append(")");
        
        if (predicat != null)
        {
            sb.append("[").append(predicat.build()).append("]");
        }
        
        return _query(sb.toString());
    }
    
    /**
     * Returns all ACL objects for groups (ametys:acl-group nodes)
     * @param rootPath The root path to restrict the search. Can be null.
     * @param predicate The predicate expression. Can be null.
     * @return The ACL group objects for groups
     */
    public static NodeIterator getACLGroups (String rootPath, Expression predicate)
    {
        StringBuilder sb = new StringBuilder("/jcr:root");
        
        if (rootPath != null)
        {
            sb.append(rootPath);
        }
        
        sb.append("//element(*, ").append(__NODETYPE_ACL_GROUP).append(")");
        
        if (predicate != null)
        {
            sb.append("[").append(predicate.build()).append("]");
        }
        
        return _query(sb.toString());
    }
    
    private static NodeIterator _getApprochingACLGroups (Set<GroupIdentity> groups, String rootPath, Expression predicat)
    {
        StringBuilder sb = new StringBuilder("/jcr:root");
        
        if (rootPath != null)
        {
            sb.append(rootPath);
        }
        
        sb.append("//element(*, ametys:acl-group)[(");
        
        sb.append(groups.stream()
            .map(GroupIdentity::getId)
            .map(Text::escapeIllegalJcrChars)
            .map(ISO9075::encode) // used to support nodeName with number (id of SQL Group)
            .map(nodeName -> "fn:name()='" + nodeName + "'")
            .collect(Collectors.joining(LogicalOperator.OR.toString())));
        sb.append(")");
        
        if (predicat != null)
        {
            sb.append(LogicalOperator.AND.toString()).append(predicat.build());
        }
        
        sb.append("]");
        
        return _query(sb.toString());
    }
    
    /**
     * Returns all ACL objects for a given group (ametys:acl-group nodes)
     * @param group The group
     * @param rootPath The root path to restrict the search. Can be null.
     * @param predicat The predicat expression. Can be null.
     * @return The ACL user objects for groups
     */
    public static NodeIterator getACLGroups (GroupIdentity group, String rootPath, Expression predicat)
    {
        StringBuilder sb = new StringBuilder("/jcr:root");
        
        if (rootPath != null)
        {
            sb.append(rootPath);
        }
        
        sb.append("//element(*, ").append(__NODETYPE_ROOT_ACL).append(")")
            .append("/").append(__NODE_NAME_ACL_GROUPS)
            .append("/").append(group.getDirectoryId())
            .append("/").append(ISO9075.encode(Text.escapeIllegalJcrChars(group.getId())));
        
        if (predicat != null)
        {
            sb.append("[").append(predicat.build()).append("]");
        }
        
        return _query(sb.toString());
    }
    
    private static NodeIterator _query (String jcrQuery)
    {
        Session session = null;
        try
        {
            session = _repository.login();
            long t1 = System.currentTimeMillis();
            @SuppressWarnings("deprecation")
            Query query = session.getWorkspace().getQueryManager().createQuery(jcrQuery, Query.XPATH);
            if (_logger.isInfoEnabled())
            {
                _logger.info("ACLJCR query '" + jcrQuery + "' executed in " + (System.currentTimeMillis() - t1) + " ms");
            }
            return query.execute().getNodes();
        }
        catch (RepositoryException ex)
        {
            if (session != null)
            {
                session.logout();
            }
            throw new AmetysRepositoryException("An error occured executing the JCR query : " + jcrQuery, ex);
        }
    }
    
    /* ------------------------------------------- */
    /* PROFILES FOR ANY CONNECTED USER / ANONYMOUS */
    /* ------------------------------------------- */
    
    /**
     * Helper for {@link ACLAmetysObject#getProfilesForAnonymousAndAnyConnectedUser}
     * @param node The JCR node for the Ametys object
     * @return a map containing allowed/denied profiles that anonymous and any connected user has on the given object
     */
    public static Map<AnonymousOrAnyConnectedKeys, Set<String>> getProfilesForAnonymousAndAnyConnectedUser(Node node)
    {
        Node aclNode = _getACLNode(node);
        if (aclNode == null)
        {
            return __ANONYMOUS_OR_ANYCONNECTEDUSER_NORIGHT;
        }
        else
        {
            return Map.of(AnonymousOrAnyConnectedKeys.ANONYMOUS_ALLOWED,         _getProperty(aclNode, __PROPERTY_NAME_ALLOWED_ANONYMOUS_PROFILES),
                          AnonymousOrAnyConnectedKeys.ANONYMOUS_DENIED,          _getProperty(aclNode, __PROPERTY_NAME_DENIED_ANONYMOUS_PROFILES),
                          AnonymousOrAnyConnectedKeys.ANYCONNECTEDUSER_ALLOWED,  _getProperty(aclNode, __PROPERTY_NAME_ALLOWED_ANY_CONNECTED_PROFILES),
                          AnonymousOrAnyConnectedKeys.ANYCONNECTEDUSER_DENIED,   _getProperty(aclNode, __PROPERTY_NAME_DENIED_ANY_CONNECTED_PROFILES));
        }
    }

    /**
     * Helper for {@link ModifiableACLAmetysObject#addAllowedProfilesForAnyConnectedUser(Set)}
     * @param node The JCR node for the Ametys object
     * @param profileIds The profiles to add
     */
    public static void addAllowedProfilesForAnyConnectedUser(Node node, Set<String> profileIds)
    {
        Node aclNode = _getOrCreateACLNode(node);
        for (String profile : profileIds)
        {
            _addProperty(aclNode, __PROPERTY_NAME_ALLOWED_ANY_CONNECTED_PROFILES, profile);
        }
        _save(node);
    }

    /**
     * Helper for {@link ModifiableACLAmetysObject#removeAllowedProfilesForAnyConnectedUser(Set)}
     * @param node The JCR node for the Ametys object
     * @param profileIds The profiles to remove
     */
    public static void removeAllowedProfilesForAnyConnectedUser(Node node, Set<String> profileIds)
    {
        Node aclNode = _getOrCreateACLNode(node);
        for (String profile : profileIds)
        {
            _removeProperty(aclNode, __PROPERTY_NAME_ALLOWED_ANY_CONNECTED_PROFILES, profile);
        }
        _save(node);
    }
    
    /**
     * Helper for {@link ModifiableACLAmetysObject#addDeniedProfilesForAnyConnectedUser(Set)}
     * @param node The JCR node for the Ametys object
     * @param profileIds The profiles to add
     */
    public static void addDeniedProfilesForAnyConnectedUser(Node node, Set<String> profileIds)
    {
        Node aclNode = _getOrCreateACLNode(node);
        for (String profile : profileIds)
        {
            _addProperty(aclNode, __PROPERTY_NAME_DENIED_ANY_CONNECTED_PROFILES, profile);
        }
        _save(node);
    }
    
    /**
     * Helper for {@link ModifiableACLAmetysObject#removeDeniedProfilesForAnyConnectedUser(Set)}
     * @param node The JCR node for the Ametys object
     * @param profileIds The profiles to remove
     */
    public static void removeDeniedProfilesForAnyConnectedUser(Node node, Set<String> profileIds)
    {
        Node aclNode = _getOrCreateACLNode(node);
        for (String profile : profileIds)
        {
            _removeProperty(aclNode, __PROPERTY_NAME_DENIED_ANY_CONNECTED_PROFILES, profile);
        }
        _save(node);
    }
    
    /**
     * Helper for {@link ModifiableACLAmetysObject#addAllowedProfilesForAnonymous(Set)}
     * @param node The JCR node for the Ametys object
     * @param profileIds The profiles to add
     */
    public static void addAllowedProfilesForAnonymous(Node node, Set<String> profileIds)
    {
        Node aclNode = _getOrCreateACLNode(node);
        for (String profile : profileIds)
        {
            _addProperty(aclNode, __PROPERTY_NAME_ALLOWED_ANONYMOUS_PROFILES, profile);
        }
        _save(node);
    }
    
    /**
     * Helper for {@link ModifiableACLAmetysObject#removeAllowedProfilesForAnonymous(Set)}
     * @param node The JCR node for the Ametys object
     * @param profileIds The profiles to remove
     */
    public static void removeAllowedProfilesForAnonymous(Node node, Set<String> profileIds)
    {
        Node aclNode = _getOrCreateACLNode(node);
        for (String profile : profileIds)
        {
            _removeProperty(aclNode, __PROPERTY_NAME_ALLOWED_ANONYMOUS_PROFILES, profile);
        }
        _save(node);
    }
    
    /**
     * Helper for {@link ModifiableACLAmetysObject#addDeniedProfilesForAnyConnectedUser(Set)}
     * @param node The JCR node for the Ametys object
     * @param profileIds The profiles to add
     */
    public static void addDeniedProfilesForAnonymous(Node node, Set<String> profileIds)
    {
        Node aclNode = _getOrCreateACLNode(node);
        for (String profile : profileIds)
        {
            _addProperty(aclNode, __PROPERTY_NAME_DENIED_ANONYMOUS_PROFILES, profile);
        }
        _save(node);
    }
    
    /**
     * Helper for {@link ModifiableACLAmetysObject#removeDeniedProfilesForAnyConnectedUser(Set)}
     * @param node The JCR node for the Ametys object
     * @param profileIds The profiles to remove
     */
    public static void removeDeniedProfilesForAnonymous(Node node, Set<String> profileIds)
    {
        Node aclNode = _getOrCreateACLNode(node);
        for (String profile : profileIds)
        {
            _removeProperty(aclNode, __PROPERTY_NAME_DENIED_ANONYMOUS_PROFILES, profile);
        }
        _save(node);
    }
    
    
    /* ------------------- */
    /* MANAGEMENT OF USERS */
    /* ------------------- */
    /**
     * Helper for {@link ACLAmetysObject#getProfilesForUsers}
     * @param node The JCR node for the Ametys object
     * @param user The user to get profiles for. Can be null to get profiles for all users that have rights
     * @return The map of allowed users with their assigned allowed/denied profiles
     */
    public static Map<UserIdentity, Map<UserOrGroup, Set<String>>> getProfilesForUsers(Node node, UserIdentity user)
    {
        if (user == null)
        {
            try
            {
                Node usersNode = _getUsersNode(node);
                if (usersNode == null)
                {
                    return Map.of();
                }
                
                Map<UserIdentity, Map<UserOrGroup, Set<String>>> result = new HashMap<>();
                
                NodeIterator populationsIterator = usersNode.getNodes();
                while (populationsIterator.hasNext())
                {
                    Node populationNode = populationsIterator.nextNode();
                    NodeIterator usersIterator = populationNode.getNodes();
                    while (usersIterator.hasNext())
                    {
                        Node userNode = usersIterator.nextNode();
                        Set<String> allowedProfiles = _getProperty(userNode, __PROPERTY_NAME_ALLOWED_PROFILES);
                        Set<String> deniedProfiles = _getProperty(userNode, __PROPERTY_NAME_DENIED_PROFILES);
                        if (!allowedProfiles.isEmpty() || !deniedProfiles.isEmpty())
                        {
                            result.put(new UserIdentity(userNode.getName(), populationNode.getName()),
                                        Map.of(UserOrGroup.ALLOWED, allowedProfiles,
                                               UserOrGroup.DENIED, deniedProfiles));
                        }
                    }
                }
                
                return result;
            }
            catch (RepositoryException e)
            {
                throw new AmetysRepositoryException("Unable to get allowed/denied users", e);
            }
        }
        else
        {
            Node userNode = _getUserNode(node, user);
            if (userNode == null)
            {
                return Map.of(user, __USER_OR_GROUP_NORIGHT);
            }
            else
            {
                return Map.of(user,
                              Map.of(UserOrGroup.ALLOWED, _getProperty(userNode, __PROPERTY_NAME_ALLOWED_PROFILES),
                                     UserOrGroup.DENIED,  _getProperty(userNode, __PROPERTY_NAME_DENIED_PROFILES)));
            }
        }
    }

    /**
     * Helper for {@link ModifiableACLAmetysObject#addAllowedUsers(Set, String)}
     * @param users The users to add
     * @param node The JCR node for the Ametys object
     * @param profileId The id of the profile
     */
    public static void addAllowedUsers(Set<UserIdentity> users, Node node, String profileId)
    {
        for (UserIdentity userIdentity : users)
        {
            Node userNode = _getOrCreateUserNode(node, userIdentity);
            _addProperty(userNode, __PROPERTY_NAME_ALLOWED_PROFILES, profileId);
        }
        _save(node);
    }

    /**
     * Helper for {@link ModifiableACLAmetysObject#removeAllowedUsers(Set, String)}
     * @param users The users to remove
     * @param node The JCR node for the Ametys object
     * @param profileId The id of the profile
     */
    public static void removeAllowedUsers(Set<UserIdentity> users, Node node, String profileId)
    {
        for (UserIdentity userIdentity : users)
        {
            Node userNode = _getOrCreateUserNode(node, userIdentity);
            _removeProperty(userNode, __PROPERTY_NAME_ALLOWED_PROFILES, profileId);
        }
        _save(node);
    }

    /**
     * Helper for {@link ModifiableACLAmetysObject#removeAllowedGroups(Set)}
     * @param users The users to remove
     * @param node The JCR node for the Ametys object
     */
    public static void removeAllowedUsers(Set<UserIdentity> users, Node node)
    {
        for (UserIdentity userIdentity : users)
        {
            Node userNode = _getOrCreateUserNode(node, userIdentity);
            _setProperty(userNode, __PROPERTY_NAME_ALLOWED_PROFILES, Collections.EMPTY_SET);
        }
        _save(node);
    }
    
    /**
     * Helper for {@link ModifiableACLAmetysObject#addDeniedUsers(Set, String)}
     * @param users The users to add
     * @param node The JCR node for the Ametys object
     * @param profileId The id of the profile
     */
    public static void addDeniedUsers(Set<UserIdentity> users, Node node, String profileId)
    {
        for (UserIdentity userIdentity : users)
        {
            Node userNode = _getOrCreateUserNode(node, userIdentity);
            _addProperty(userNode, __PROPERTY_NAME_DENIED_PROFILES, profileId);
        }
        _save(node);
    }

    /**
     * Helper for {@link ModifiableACLAmetysObject#removeDeniedUsers(Set, String)}
     * @param users The users to remove
     * @param node The JCR node for the Ametys object
     * @param profileId The id of the profile
     */
    public static void removeDeniedUsers(Set<UserIdentity> users, Node node, String profileId)
    {
        for (UserIdentity userIdentity : users)
        {
            Node userNode = _getOrCreateUserNode(node, userIdentity);
            _removeProperty(userNode, __PROPERTY_NAME_DENIED_PROFILES, profileId);
        }
        _save(node);
    }

    /**
     * Helper for {@link ModifiableACLAmetysObject#removeDeniedUsers(Set)}
     * @param users The users to remove
     * @param node The JCR node for the Ametys object
     */
    public static void removeDeniedUsers(Set<UserIdentity> users, Node node)
    {
        for (UserIdentity userIdentity : users)
        {
            Node userNode = _getOrCreateUserNode(node, userIdentity);
            _setProperty(userNode, __PROPERTY_NAME_DENIED_PROFILES, Collections.EMPTY_SET);
        }
        _save(node);
    }

    /* -------------------- */
    /* MANAGEMENT OF GROUPS */
    /* -------------------- */

    /**
     * Helper for {@link ACLAmetysObject#getProfilesForGroups}
     * @param node The JCR node for the Ametys object
     * @param groups The group to get profiles for. Can be null to get profiles for all groups that have rights
     * @return The map of allowed/denied groups with their assigned profiles
     */
    public static Map<GroupIdentity, Map<UserOrGroup, Set<String>>> getProfilesForGroups(Node node, Set<GroupIdentity> groups)
    {
        try
        {
            if (groups != null && groups.isEmpty())
            {
                return Map.of();
            }
            
            Node groupsNode = _getGroupsNode(node);
            if (groupsNode == null)
            {
                return Map.of();
            }
            
            Map<GroupIdentity, Map<UserOrGroup, Set<String>>> result = new HashMap<>();
            
            if (groups == null)
            {
                NodeIterator groupDirectoriesIterator = groupsNode.getNodes();
                while (groupDirectoriesIterator.hasNext())
                {
                    Node groupDirectoryNode = groupDirectoriesIterator.nextNode();
                    NodeIterator groupsIterator = groupDirectoryNode.getNodes();
                    while (groupsIterator.hasNext())
                    {
                        Node groupNode = groupsIterator.nextNode();
                        Set<String> allowedProfiles = _getProperty(groupNode, __PROPERTY_NAME_ALLOWED_PROFILES);
                        Set<String> deniedProfiles = _getProperty(groupNode, __PROPERTY_NAME_DENIED_PROFILES);
                        if (!allowedProfiles.isEmpty() || !deniedProfiles.isEmpty())
                        {
                            result.put(new GroupIdentity(Text.unescapeIllegalJcrChars(groupNode.getName()), groupDirectoryNode.getName()),
                                        Map.of(UserOrGroup.ALLOWED, allowedProfiles,
                                               UserOrGroup.DENIED, deniedProfiles));
                        }
                    }
                }
            }
            else
            {
                Map<String, Node> groupsNodeByDirectoryIdCache = new HashMap<>();
                
                for (GroupIdentity group : groups)
                {
                    Node directoryNode = groupsNodeByDirectoryIdCache.computeIfAbsent(group.getDirectoryId(), LambdaUtils.wrap(directoryId -> groupsNode.hasNode(directoryId) ? groupsNode.getNode(directoryId) : null));
                    
                    String groupNodeName = Text.escapeIllegalJcrChars(group.getId());
                    if (directoryNode != null && directoryNode.hasNode(groupNodeName))
                    {
                        Node groupNode = directoryNode.getNode(groupNodeName);
                        
                        result.put(group,
                                Map.of(UserOrGroup.ALLOWED, _getProperty(groupNode, __PROPERTY_NAME_ALLOWED_PROFILES),
                                       UserOrGroup.DENIED, _getProperty(groupNode, __PROPERTY_NAME_DENIED_PROFILES)));
                    }
                }
            }
            
            return result;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get allowed/denied groups", e);
        }
    }
    
    /**
     * Helper for {@link ModifiableACLAmetysObject#addAllowedGroups(Set, String)}
     * @param groups The groups to add
     * @param node The JCR node for the Ametys object
     * @param profileId The id of the profile
     */
    public static void addAllowedGroups(Set<GroupIdentity> groups, Node node, String profileId)
    {
        for (GroupIdentity groupIdentity : groups)
        {
            Node groupNode = _getOrCreateGroupNode(node, groupIdentity);
            _addProperty(groupNode, __PROPERTY_NAME_ALLOWED_PROFILES, profileId);
        }
        _save(node);
    }

    /**
     * Helper for {@link ModifiableACLAmetysObject#removeAllowedGroups(Set, String)}
     * @param groups The groups to remove
     * @param node The JCR node for the Ametys object
     * @param profileId The id of the profile
     */
    public static void removeAllowedGroups(Set<GroupIdentity> groups, Node node, String profileId)
    {
        for (GroupIdentity groupIdentity : groups)
        {
            Node groupNode = _getOrCreateGroupNode(node, groupIdentity);
            _removeProperty(groupNode, __PROPERTY_NAME_ALLOWED_PROFILES, profileId);
        }
        _save(node);
    }

    /**
     * Helper for {@link ModifiableACLAmetysObject#removeAllowedGroups(Set)}
     * @param groups The groups to remove
     * @param node The JCR node for the Ametys object
     */
    public static void removeAllowedGroups(Set<GroupIdentity> groups, Node node)
    {
        for (GroupIdentity groupIdentity : groups)
        {
            Node groupNode = _getOrCreateGroupNode(node, groupIdentity);
            _setProperty(groupNode, __PROPERTY_NAME_ALLOWED_PROFILES, Collections.EMPTY_SET);
        }
        _save(node);
    }

    /**
     * Helper for {@link ModifiableACLAmetysObject#addDeniedGroups(Set, String)}
     * @param groups The groups to add
     * @param node The JCR node for the Ametys object
     * @param profileId The id of the profile
     */
    public static void addDeniedGroups(Set<GroupIdentity> groups, Node node, String profileId)
    {
        for (GroupIdentity groupIdentity : groups)
        {
            Node groupNode = _getOrCreateGroupNode(node, groupIdentity);
            _addProperty(groupNode, __PROPERTY_NAME_DENIED_PROFILES, profileId);
        }
        _save(node);
    }

    /**
     * Helper for {@link ModifiableACLAmetysObject#removeDeniedGroups(Set, String)}
     * @param groups The groups to remove
     * @param node The JCR node for the Ametys object
     * @param profileId The id of the profile
     */
    public static void removeDeniedGroups(Set<GroupIdentity> groups, Node node, String profileId)
    {
        for (GroupIdentity groupIdentity : groups)
        {
            Node groupNode = _getOrCreateGroupNode(node, groupIdentity);
            _removeProperty(groupNode, __PROPERTY_NAME_DENIED_PROFILES, profileId);
        }
        _save(node);
    }

    /**
     * Helper for {@link ModifiableACLAmetysObject#removeDeniedGroups(Set)}
     * @param groups The groups to remove
     * @param node The JCR node for the Ametys object
     */
    public static void removeDeniedGroups(Set<GroupIdentity> groups, Node node)
    {
        for (GroupIdentity groupIdentity : groups)
        {
            Node groupNode = _getOrCreateGroupNode(node, groupIdentity);
            _setProperty(groupNode, __PROPERTY_NAME_DENIED_PROFILES, Collections.EMPTY_SET);
        }
        _save(node);
    }
    
    
    /* ------ */
    /* REMOVE */
    /* ------ */
    
    /**
     * Helper for {@link ModifiableACLAmetysObjectProfileAssignmentStorage#removeProfile(String)}
     * @param profileId The id of the profile
     */
    public static void removeProfile(String profileId)
    {
        // Remove this profile set as allowed or denied in users
        Expression expr = new OrExpression(new AllowedProfileExpression(profileId), new DeniedProfileExpression(profileId));
        NodeIterator users = getACLUsers(expr);
        while (users.hasNext())
        {
            Node userNode = (Node) users.next();
            _removeProperty(userNode, __PROPERTY_NAME_ALLOWED_PROFILES, profileId);
            _removeProperty(userNode, __PROPERTY_NAME_DENIED_PROFILES, profileId);
            _save(userNode);
        }
        
        // Remove this profile set as allowed or denied in groups
        NodeIterator groups = getACLGroups(expr);
        while (groups.hasNext())
        {
            Node groupNode = (Node) groups.next();
            _removeProperty(groupNode, __PROPERTY_NAME_ALLOWED_PROFILES, profileId);
            _removeProperty(groupNode, __PROPERTY_NAME_DENIED_PROFILES, profileId);
            _save(groupNode);
        }
        
        // Remove this profile set as allowed or denied for anonymous and any connected
        expr = new OrExpression(new AnonymousAllowedProfileExpression(profileId), new AnonymousDeniedProfileExpression(profileId), new AnyConnectedAllowedProfileExpression(profileId), new AnyConnectedDeniedProfileExpression(profileId));
        NodeIterator nodes = getACLRoots(null, expr);
        while (nodes.hasNext())
        {
            Node node = (Node) nodes.next();
            _removeProperty(node, __PROPERTY_NAME_ALLOWED_ANY_CONNECTED_PROFILES, profileId);
            _removeProperty(node, __PROPERTY_NAME_DENIED_ANY_CONNECTED_PROFILES, profileId);
            _removeProperty(node, __PROPERTY_NAME_ALLOWED_ANONYMOUS_PROFILES, profileId);
            _removeProperty(node, __PROPERTY_NAME_DENIED_ANONYMOUS_PROFILES, profileId);
            _save(node);
        }
    }
    
    /**
     * Helper for {@link ModifiableACLAmetysObjectProfileAssignmentStorage#removeUser(UserIdentity)}
     * @param user The user
     */
    public static void removeUser(UserIdentity user)
    {
        NodeIterator users = getACLUsers(user, null, null);
        
        while (users.hasNext())
        {
            Node userNode = (Node) users.next();
            try
            {
                userNode.remove();
                _save(userNode);
            }
            catch (RepositoryException e)
            {
                throw new AmetysRepositoryException(e);
            }
        }
    }
    
    /**
     * Helper for {@link ModifiableACLAmetysObjectProfileAssignmentStorage#removeGroup(GroupIdentity)}
     * @param group The group
     */
    public static void removeGroup(GroupIdentity group)
    {
        NodeIterator groups = getACLGroups(group, null, null);
        while (groups.hasNext())
        {
            Node gpNode = (Node) groups.next();
            try
            {
                gpNode.remove();
                _save(gpNode);
            }
            catch (RepositoryException e)
            {
                throw new AmetysRepositoryException(e);
            }
        }
    }
    
    /* --------------- */
    /* INHERITANCE     */
    /* --------------- */
    /**
     * Helper for {@link ACLAmetysObject#isInheritanceDisallowed()}
     * @param node The JCR node for the Ametys object
     * @return true if the inheritance is disallow of the given node
     */
    public static boolean isInheritanceDisallowed(Node node)
    {
        try
        {
            Node aclNode = _getACLNode(node);
            if (aclNode != null && aclNode.hasProperty(__PROPERTY_NAME_DISALLOW_INHERITANCE))
            {
                return aclNode.getProperty(__PROPERTY_NAME_DISALLOW_INHERITANCE).getBoolean();
            }
            return false;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get " + __PROPERTY_NAME_DISALLOW_INHERITANCE + " property", e);
        }
    }
    
    /**
     * Helper for {@link ModifiableACLAmetysObject#disallowInheritance(boolean)}
     * @param node The JCR node for the Ametys object
     * @param disallow true to disallow the inheritance, false otherwise
     */
    public static void disallowInheritance(Node node, boolean disallow)
    {
        Node aclNode = _getOrCreateACLNode(node);
        try
        {
            aclNode.setProperty(__PROPERTY_NAME_DISALLOW_INHERITANCE, disallow);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to set " + __PROPERTY_NAME_DISALLOW_INHERITANCE + " property", e);
        }
        _save(node);
    }
    
    
    /* --------------- */
    /* PRIVATE METHODS */
    /* --------------- */
    
    private static void _checkLock(Node node) throws AmetysRepositoryException
    {
        try
        {
            if (node.isLocked())
            {
                LockManager lockManager = node.getSession().getWorkspace().getLockManager();
                
                Lock lock = lockManager.getLock(node.getPath());
                Node lockHolder = lock.getNode();
                
                lockManager.addLockToken(lockHolder.getProperty(RepositoryConstants.METADATA_LOCKTOKEN).getString());
            }
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to add lock token on ACL node", e);
        }
    }
    
    private static Node _getOrCreateACLNode(Node node)
    {
        try
        {
            if (node.hasNode(__NODE_NAME_ROOT_ACL))
            {
                return node.getNode(__NODE_NAME_ROOT_ACL);
            }
            else
            {
                _checkLock(node);
                return node.addNode(__NODE_NAME_ROOT_ACL, __NODETYPE_ROOT_ACL);
            }
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error while getting root ACL node.", e);
        }
    }
    
    private static Node _getACLNode(Node node)
    {
        try
        {
            if (node.hasNode(__NODE_NAME_ROOT_ACL))
            {
                return node.getNode(__NODE_NAME_ROOT_ACL);
            }
            else
            {
                return null;
            }
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error while getting root ACL node.", e);
        }
    }
    
    private static Node _getOrCreateUsersNode(Node node)
    {
        try
        {
            Node aclNode = _getOrCreateACLNode(node);
            if (aclNode.hasNode(__NODE_NAME_ACL_USERS))
            {
                return aclNode.getNode(__NODE_NAME_ACL_USERS);
            }
            else
            {
                return aclNode.addNode(__NODE_NAME_ACL_USERS, JcrConstants.NT_UNSTRUCTURED);
            }
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error while getting 'users' ACL node.", e);
        }
    }
    
    private static Node _getUserNode(Node node, UserIdentity user)
    {
        try
        {
            Node aclNode = _getACLNode(node);
            if (aclNode != null && aclNode.hasNode(__NODE_NAME_ACL_USERS))
            {
                Node aclUsersNode = aclNode.getNode(__NODE_NAME_ACL_USERS);
                if (aclUsersNode.hasNode(user.getPopulationId()))
                {
                    Node popNode = aclUsersNode.getNode(user.getPopulationId());
                    if (popNode.hasNode(user.getLogin()))
                    {
                        return popNode.getNode(user.getLogin());
                    }
                }
            }
            
            return null;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error while getting 'users' ACL node.", e);
        }
    }
    
    private static Node _getUsersNode(Node node)
    {
        try
        {
            Node aclNode = _getACLNode(node);
            if (aclNode != null && aclNode.hasNode(__NODE_NAME_ACL_USERS))
            {
                return aclNode.getNode(__NODE_NAME_ACL_USERS);
            }
            else
            {
                return null;
            }
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error while getting 'users' ACL node.", e);
        }
    }
    
    private static Node _getOrCreateGroupsNode(Node node)
    {
        try
        {
            Node aclNode = _getOrCreateACLNode(node);
            if (aclNode.hasNode(__NODE_NAME_ACL_GROUPS))
            {
                return aclNode.getNode(__NODE_NAME_ACL_GROUPS);
            }
            else
            {
                return aclNode.addNode(__NODE_NAME_ACL_GROUPS, JcrConstants.NT_UNSTRUCTURED);
            }
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error while getting 'groups' ACL node.", e);
        }
    }
    
    private static Node _getGroupsNode(Node node)
    {
        try
        {
            Node aclNode = _getACLNode(node);
            if (aclNode != null && aclNode.hasNode(__NODE_NAME_ACL_GROUPS))
            {
                return aclNode.getNode(__NODE_NAME_ACL_GROUPS);
            }
            else
            {
                return null;
            }
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error while getting 'groups' ACL node.", e);
        }
    }
    
    private static Node _getOrCreateUserNode(Node node, UserIdentity userIdentity)
    {
        try
        {
            Node usersNode = _getOrCreateUsersNode(node);
            String population = userIdentity.getPopulationId();
            String login = userIdentity.getLogin();
            
            if (usersNode.hasNode(population))
            {
                Node populationNode = usersNode.getNode(population);
                if (populationNode.hasNode(login))
                {
                    return populationNode.getNode(login);
                }
                else
                {
                    return populationNode.addNode(login, __NODETYPE_ACL_USER);
                }
            }
            else
            {
                return usersNode.addNode(population, JcrConstants.NT_UNSTRUCTURED).addNode(login, __NODETYPE_ACL_USER);
            }
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException(String.format("Error while getting 'user' ACL node for %s.", userIdentity.toString()), e);
        }
    }
    
    private static Node _getOrCreateGroupNode(Node node, GroupIdentity groupIdentity)
    {
        try
        {
            Node groupsNode = _getOrCreateGroupsNode(node);
            String directoryId = groupIdentity.getDirectoryId();
            String id = Text.escapeIllegalJcrChars(groupIdentity.getId());
            
            if (groupsNode.hasNode(directoryId))
            {
                Node populationNode = groupsNode.getNode(directoryId);
                if (populationNode.hasNode(id))
                {
                    return populationNode.getNode(id);
                }
                else
                {
                    return populationNode.addNode(id, __NODETYPE_ACL_GROUP);
                }
            }
            else
            {
                return groupsNode.addNode(directoryId, JcrConstants.NT_UNSTRUCTURED).addNode(id, __NODETYPE_ACL_GROUP);
            }
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException(String.format("Error while getting 'group' ACL node for %s.", groupIdentity.toString()), e);
        }
    }
    
    private static Set<String> _getProperty(Node node, String propertyName)
    {
        try
        {
            Value[] values = node.getProperty(propertyName).getValues();
            Set<String> result = new HashSet<>();
            for (Value value : values)
            {
                result.add(value.getString());
            }
            return result;
        }
        catch (PathNotFoundException e)
        {
            return new HashSet<>();
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get " + propertyName + " property", e);
        }
    }
    
    private static void _setProperty(Node node, String propertyName, Set<String> profiles)
    {
        try
        {
            node.setProperty(propertyName, profiles.toArray(new String[profiles.size()]));
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to set " + propertyName + " property", e);
        }
    }
    
    private static void _addProperty(Node node, String propertyName, String profileToAdd)
    {
        Set<String> profiles = _getProperty(node, propertyName);
        if (!profiles.contains(profileToAdd))
        {
            profiles.add(profileToAdd);
            _setProperty(node, propertyName, profiles);
        }
    }
    
    private static void _removeProperty(Node node, String propertyName, String profileToRemove)
    {
        Set<String> profiles = _getProperty(node, propertyName);
        if (profiles.contains(profileToRemove))
        {
            profiles.remove(profileToRemove);
            _setProperty(node, propertyName, profiles);
        }
    }
    
    private static void _save(Node node)
    {
        try
        {
            node.getSession().save();
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to save changes", e);
        }
    }
    
    /* ---------------------------------------*/
    /*      JCR EXPRESSIONS FOR PROFILES      */
    /* ---------------------------------------*/
    
    static class AllowedProfileExpression extends ACLProfileExpression
    {
        public AllowedProfileExpression (String ... profileIds)
        {
            super(__PROPERTY_NAME_ALLOWED_PROFILES, profileIds);
        }
    }
    
    static class DeniedProfileExpression extends ACLProfileExpression
    {
        public DeniedProfileExpression (String ... profileIds)
        {
            super(__PROPERTY_NAME_DENIED_PROFILES, profileIds);
        }
    }
    
    static class AnyConnectedDeniedProfileExpression extends ACLProfileExpression
    {
        public AnyConnectedDeniedProfileExpression (String ... profileIds)
        {
            super(__PROPERTY_NAME_DENIED_ANY_CONNECTED_PROFILES, profileIds);
        }
    }
    
    static class AnyConnectedAllowedProfileExpression extends ACLProfileExpression
    {
        public AnyConnectedAllowedProfileExpression (String ... profileIds)
        {
            super(__PROPERTY_NAME_ALLOWED_ANY_CONNECTED_PROFILES, profileIds);
        }
    }
    
    static class AnonymousDeniedProfileExpression extends ACLProfileExpression
    {
        public AnonymousDeniedProfileExpression (String ... profileIds)
        {
            super(__PROPERTY_NAME_DENIED_ANONYMOUS_PROFILES, profileIds);
        }
    }
    
    static class AnonymousAllowedProfileExpression extends ACLProfileExpression
    {
        public AnonymousAllowedProfileExpression (String ... profileIds)
        {
            super(__PROPERTY_NAME_ALLOWED_ANONYMOUS_PROFILES, profileIds);
        }
    }
    
    static class ACLProfileExpression implements Expression
    {
        private String[] _profileIds;
        private String _propertyName;
        
        public ACLProfileExpression (String propertyName, String ... profileIds)
        {
            _propertyName = propertyName;
            _profileIds = profileIds;
        }
        
        @Override
        public String build()
        {
            boolean isFirst = true;
            StringBuilder sb = new StringBuilder("(");
            
            for (String profileId : _profileIds)
            {
                if (isFirst)
                {
                    isFirst = false;
                }
                else
                {
                    sb.append(LogicalOperator.OR.toString());
                }
                
                sb.append("@")
                    .append(_propertyName)
                    .append(Operator.EQ)
                    .append("'").append(profileId).append("'");
            }
            
            if (isFirst)
            {
                return "";
            }
            else
            {
                return sb.append(")").toString();
            }
        }
    }

    /**
     * Get all contexts with a permission for the profile for anonymous or any connected user
     * @param rootNodes The JCR root nodes where starts the query search (must be something like "//element(myNode, ametys:collection)"), it will be the beginning of the JCR query
     * @param profileId the profile id
     * @return a map of the permissions for each context
     */
    public static Map<Object, Set<AnonymousOrAnyConnectedKeys>> getProfilePermissionsForAnonymousAndAnyConnectedUser(Set< ? extends Object> rootNodes, String profileId)
    {
        Expression expr = new OrExpression(
            new AnyConnectedAllowedProfileExpression(profileId),
            new AnyConnectedDeniedProfileExpression(profileId),
            new AnonymousAllowedProfileExpression(profileId),
            new AnonymousDeniedProfileExpression(profileId)
        );
        
        Map<Object, Set<AnonymousOrAnyConnectedKeys>> result = new HashMap<>();
        for (String rootPath : _convertNodeToPath(rootNodes))
        {
            NodeIterator nodes = getACLRoots(rootPath, expr);
            
            // Only retrieve node with assignments to anonymous or any connected
            while (nodes.hasNext())
            {
                Node aclNode = nodes.nextNode();
                try
                {
                    Set<AnonymousOrAnyConnectedKeys> aoResult = new HashSet<>();
                    Set<String> allowedAnonymous = _getProperty(aclNode, __PROPERTY_NAME_ALLOWED_ANONYMOUS_PROFILES);
                    if (allowedAnonymous.contains(profileId))
                    {
                        aoResult.add(AnonymousOrAnyConnectedKeys.ANONYMOUS_ALLOWED);
                    }
                    
                    Set<String> deniedAnonymous = _getProperty(aclNode, __PROPERTY_NAME_DENIED_ANONYMOUS_PROFILES);
                    if (deniedAnonymous.contains(profileId))
                    {
                        aoResult.add(AnonymousOrAnyConnectedKeys.ANONYMOUS_DENIED);
                    }
                    Set<String> allowedAny = _getProperty(aclNode, __PROPERTY_NAME_ALLOWED_ANY_CONNECTED_PROFILES);
                    if (allowedAny.contains(profileId))
                    {
                        aoResult.add(AnonymousOrAnyConnectedKeys.ANYCONNECTEDUSER_ALLOWED);
                    }
                    Set<String> deniedAny = _getProperty(aclNode, __PROPERTY_NAME_DENIED_ANY_CONNECTED_PROFILES);
                    if (deniedAny.contains(profileId))
                    {
                        aoResult.add(AnonymousOrAnyConnectedKeys.ANYCONNECTEDUSER_DENIED);
                    }
                    
                    if (!aoResult.isEmpty())
                    {
                        AmetysObject ao = _getAmetysObjectFromACLNode(aclNode);
                        result.put(ao, aoResult);
                    }
                }
                catch (RepositoryException e)
                {
                    _logger.error("Failed to retrieve object for acl node " + aclNode.toString() + ". The node will be ignored.");
                }
            }
        }

        return result;
    }
    
    /**
     * Get all context with a permission for the profile for a group
     * @param rootNodes The JCR root nodes where starts the query search (must be something like "//element(myNode, ametys:collection)"), it will be the beginning of the JCR query
     * @param profileId the profile id
     * @return a map of the group's permission for each context
     */
    public static Map<Object, Map<GroupIdentity, Set<UserOrGroup>>> getProfilePermissionsForGroups(Set< ? extends Object> rootNodes, String profileId)
    {
        Expression expr = new OrExpression(
                new AllowedProfileExpression(profileId),
                new DeniedProfileExpression(profileId)
        );
        
        Map<Object, Map<GroupIdentity, Set<UserOrGroup>>> result = new HashMap<>();
        for (String rootPath : _convertNodeToPath(rootNodes))
        {
            NodeIterator nodes = getACLGroups(rootPath, expr);
            
            while (nodes.hasNext())
            {
                Node groupNode = nodes.nextNode();
                try
                {
                    String groupId = Text.unescapeIllegalJcrChars(groupNode.getName());
                    String directoryId = groupNode.getParent().getName();
                    
                    GroupIdentity currentGroup = new GroupIdentity(groupId, directoryId);
                    
                    // Determine the group permissions
                    Set<UserOrGroup> permission = new HashSet<>();
                    
                    if (_getProperty(groupNode, __PROPERTY_NAME_DENIED_PROFILES).contains(profileId))
                    {
                        permission.add(UserOrGroup.DENIED);
                    }
                    else if (_getProperty(groupNode, __PROPERTY_NAME_ALLOWED_PROFILES).contains(profileId))
                    {
                        permission.add(UserOrGroup.ALLOWED);
                    }
                    
                    // Only add actual permissions to the result
                    if (!permission.isEmpty())
                    {
                        AmetysObject ao = _getAmetysObjectFromACLNode(groupNode);
                        // The ametys object could already be in the result map having permissions from an other group
                        Map<GroupIdentity, Set<UserOrGroup>> objectPermissions = result.computeIfAbsent(ao, k -> new HashMap<>());
                        // There can only be one node per group, so we don't need to retrieve existing value
                        objectPermissions.put(currentGroup, permission);
                    }
                }
                catch (RepositoryException e)
                {
                    _logger.error("Failed to retrieve object for group acl node " + groupNode.toString() + ". The node will be ignored.");
                }
            }
        }
        return result;
    }
    
    /**
     * Get all context with a permission for the profile for a user
     * @param rootNodes The JCR root nodes where starts the query search (must be something like "//element(myNode, ametys:collection)"), it will be the beginning of the JCR query
     * @param profileId the profile id
     * @return a map of the user's permission for each context
     */
    public static Map<Object, Map<UserIdentity, Set<UserOrGroup>>> getProfilePermissionsForUsers(Set< ? extends Object> rootNodes, String profileId)
    {
        Expression expr = new OrExpression(
                new AllowedProfileExpression(profileId),
                new DeniedProfileExpression(profileId)
        );
        
        Map<Object, Map<UserIdentity, Set<UserOrGroup>>> result = new HashMap<>();
        for (String rootPath : _convertNodeToPath(rootNodes))
        {
            NodeIterator nodes = getACLUsers(rootPath, expr);
            
            while (nodes.hasNext())
            {
                Node userNode = nodes.nextNode();
                try
                {
                    String login = ISO9075.decode(userNode.getName());
                    String populationId = userNode.getParent().getName();
                    
                    UserIdentity user = new UserIdentity(login, populationId);
                    
                    // Determine the user permissions
                    Set<UserOrGroup> permission = new HashSet<>();
                    
                    if (_getProperty(userNode, __PROPERTY_NAME_DENIED_PROFILES).contains(profileId))
                    {
                        permission.add(UserOrGroup.DENIED);
                    }
                    else if (_getProperty(userNode, __PROPERTY_NAME_ALLOWED_PROFILES).contains(profileId))
                    {
                        permission.add(UserOrGroup.ALLOWED);
                    }
                    
                    // Only add actual permissions to the result
                    if (!permission.isEmpty())
                    {
                        AmetysObject ao = _getAmetysObjectFromACLNode(userNode);
                        // The ametys object could already be in the result map having permissions from an other user
                        Map<UserIdentity, Set<UserOrGroup>> objectPermissions = result.computeIfAbsent(ao, k -> new HashMap<>());
                        // There can only be one node per user, so we don't need to retrieve existing value
                        objectPermissions.put(user, permission);
                    }
                }
                catch (RepositoryException e)
                {
                    _logger.error("Failed to retrieve object for group acl node " + userNode.toString() + ". The node will be ignored.");
                }
            }
        }
        return result;
    }
}
