/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.jcr;

import java.util.Arrays;
import java.util.Calendar;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.Set;

import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.Property;
import javax.jcr.PropertyIterator;
import javax.jcr.RepositoryException;
import javax.jcr.version.OnParentVersionAction;
import javax.jcr.version.Version;
import javax.jcr.version.VersionException;
import javax.jcr.version.VersionHistory;
import javax.jcr.version.VersionIterator;

import org.apache.jackrabbit.JcrConstants;
import org.apache.jackrabbit.core.NodeImpl;

import org.ametys.core.group.GroupIdentity;
import org.ametys.core.right.ProfileAssignmentStorage.AnonymousOrAnyConnectedKeys;
import org.ametys.core.right.ProfileAssignmentStorage.UserOrGroup;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.ModifiableACLAmetysObject;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.data.holder.ModifiableModelLessDataHolder;
import org.ametys.plugins.repository.data.holder.impl.DefaultModifiableModelLessDataHolder;
import org.ametys.plugins.repository.data.repositorydata.ModifiableRepositoryData;
import org.ametys.plugins.repository.data.repositorydata.impl.JCRRepositoryData;
import org.ametys.plugins.repository.lock.LockableAmetysObject;
import org.ametys.plugins.repository.version.ModifiableDataAwareVersionableAmetysObject;
import org.ametys.plugins.repository.version.VersionableAmetysObject;

/**
 * Default implementation of a {@link JCRAmetysObject}, which is also a {@link VersionableAmetysObject}.
 * @param <F> the actual type of factory.
 */
public class DefaultAmetysObject<F extends DefaultAmetysObjectFactory> extends SimpleAmetysObject<F> implements ModifiableDataAwareVersionableAmetysObject, ModifiableACLAmetysObject
{
    /** Properties that are auto-created or protected, which mustn't be copied when copying a node. */
    protected static final List<String> PROTECTED_PROPERTIES = Arrays.asList(
            JcrConstants.JCR_UUID,
            JcrConstants.JCR_PRIMARYTYPE,
            JcrConstants.JCR_PREDECESSORS,
            JcrConstants.JCR_VERSIONHISTORY,
            JcrConstants.JCR_BASEVERSION
    );
    
    /** Compare a pair of JCR version name and creation date with an other **/
    protected static final Comparator<Entry<String, Calendar>> __VERSION_COMPARATOR = Comparator.comparing(Entry<String, Calendar>::getValue).thenComparing(Entry::getKey);

    // Root JCR Node of this content
    private Node _baseNode;

    // pointed version, or null if HEAD
    private Version _versionNode;
    
    // Current version, either HEAD or a given version
    private Node _currentNode;
    
    // The version history of the base node
    private VersionHistory _versionHistory;

    /**
     * Creates an {@link DefaultAmetysObject}.
     * @param node the node backing this {@link AmetysObject}
     * @param parentPath the parentPath in the Ametys hierarchy
     * @param factory the DefaultAmetysObjectFactory which created the AmetysObject
     */
    public DefaultAmetysObject(Node node, String parentPath, F factory)
    {
        super(node, parentPath, factory);
        _baseNode = node;
        _currentNode = node;
    }
    
    @Override
    public Node getNode()
    {
        return _currentNode;
    }
    
    /**
     * Returns the JCR node backing this {@link AmetysObject} in the default JCR workspace
     * @return the JCR node backing this {@link AmetysObject} in the default JCR workspace
     */
    public Node getBaseNode ()
    {
        return _baseNode;
    }

    // Versioning capabilities
    
    /**
     * Returns the JCR {@link VersionHistory} of the base node.
     * @return the JCR {@link VersionHistory} of the base node.
     * @throws RepositoryException if something wrong occurs retrieving the VersionHistory.
     */
    protected VersionHistory getVersionHistory() throws RepositoryException
    {
        if (_versionHistory == null)
        {
            _versionHistory = _baseNode.getSession().getWorkspace().getVersionManager().getVersionHistory(_baseNode.getPath());
        }
        
        return _versionHistory;
    }
    
    /**
     * Returns the JCR base version of the node.
     * @return the JCR base version of the node.
     * @throws RepositoryException if something wrong occurs retrieving the base version.
     */
    protected Version getBaseVersion() throws RepositoryException
    {
        return _baseNode.getSession().getWorkspace().getVersionManager().getBaseVersion(_baseNode.getPath());
    }

    public void checkpoint() throws AmetysRepositoryException
    {
        try
        {
            getNode().getSession().getWorkspace().getVersionManager().checkpoint(getNode().getPath());
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to checkpoint", e);
        }
    }
    
    public void switchToLabel(String label) throws UnknownAmetysObjectException, AmetysRepositoryException
    {
        if (label == null)
        {
            // back to current version
            _versionNode = null;
            _currentNode = _baseNode;
        }
        else
        {
            try
            {
                VersionHistory history = getVersionHistory();
                _versionNode = history.getVersionByLabel(label);
                _currentNode = _versionNode.getFrozenNode();
            }
            catch (VersionException e)
            {
                throw new UnknownAmetysObjectException("There's no label : " + label, e);
            }
            catch (RepositoryException e)
            {
                throw new AmetysRepositoryException("Unable to switch to label : " + label, e);
            }
        }
    }
    
    public void switchToRevision(String revision) throws UnknownAmetysObjectException, AmetysRepositoryException
    {
        if (revision == null)
        {
            // back to current version
            _versionNode = null;
            _currentNode = _baseNode;
        }
        else
        {
            try
            {
                VersionHistory history = getVersionHistory();
                _versionNode = history.getVersion(revision);
                _currentNode = _versionNode.getNode(JcrConstants.JCR_FROZENNODE);
            }
            catch (VersionException e)
            {
                throw new UnknownAmetysObjectException("There's no revision : " + revision, e);
            }
            catch (RepositoryException e)
            {
                throw new AmetysRepositoryException("Unable to switch to revision : " + revision, e);
            }
        }
    }
    
    public void restoreFromLabel(String label) throws UnknownAmetysObjectException, AmetysRepositoryException
    {
        try
        {
            VersionHistory history = getVersionHistory();
            Node versionNode = history.getVersionByLabel(label);
            restoreFromNode(versionNode.getNode(JcrConstants.JCR_FROZENNODE));
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to restore from label: " + label, e);
        }
    }
    
    public void restoreFromRevision(String revision) throws UnknownAmetysObjectException, AmetysRepositoryException
    {
        try
        {
            VersionHistory history = getVersionHistory();
            Node versionNode = history.getVersion(revision);
            restoreFromNode(versionNode.getNode(JcrConstants.JCR_FROZENNODE));
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to restore from revision: " + revision, e);
        }
    }
    
    /**
     * Restore from a node
     * @param node The node to restore
     * @throws RepositoryException If error occurs
     */
    protected void restoreFromNode(Node node) throws RepositoryException
    {
        // Remove all properties and nodes of the current node (except jcr and OnParentVersion=IGNORE).
        PropertyIterator propIt = _baseNode.getProperties();
        while (propIt.hasNext())
        {
            Property prop = propIt.nextProperty();
            String propName = prop.getName();
            if (!propName.startsWith("jcr:") && prop.getDefinition().getOnParentVersion() != OnParentVersionAction.IGNORE)
            {
                prop.remove();
            }
        }
        
        NodeIterator nodeIt = _baseNode.getNodes();
        while (nodeIt.hasNext())
        {
            Node childNode = nodeIt.nextNode();
            String nodeName = childNode.getName();
            if (!nodeName.startsWith("jcr:") && childNode.getDefinition().getOnParentVersion() != OnParentVersionAction.IGNORE)
            {
                childNode.remove();
            }
        }
        
        // Copy all properties and nodes from the given node (except jcr).
        PropertyIterator newPropIt = node.getProperties();
        while (newPropIt.hasNext())
        {
            Property newProp = newPropIt.nextProperty();
            
            if (!newProp.getName().startsWith("jcr:"))
            {
                if (newProp.getDefinition().isMultiple())
                {
                    _baseNode.setProperty(newProp.getName(), newProp.getValues(), newProp.getType());
                }
                else
                {
                    _baseNode.setProperty(newProp.getName(), newProp.getValue(), newProp.getType());
                }
            }
        }
        
        NodeIterator newNodeIt = node.getNodes();
        while (newNodeIt.hasNext())
        {
            Node newNode = newNodeIt.nextNode();
            
            if (!newNode.getName().startsWith("jcr:"))
            {
                copyNode(_baseNode, newNode);
            }
        }
    }
    
    /**
     * Copy the source node in parent node
     * @param parentDest The dest node
     * @param src The source node to copy
     * @throws RepositoryException If error occurs
     */
    protected void copyNode(Node parentDest, Node src) throws RepositoryException
    {
        Node dest;
        if (parentDest.hasNode(src.getName()))
        {
            // case of auto created child
            dest = parentDest.getNode(src.getName());
        }
        else
        {
            String uuid = null;
            if (src.hasProperty(JcrConstants.JCR_FROZENUUID))
            {
                uuid = src.getProperty(JcrConstants.JCR_FROZENUUID).getString();
            }
            
            if (uuid == null)
            {
                dest = parentDest.addNode(src.getName(), src.getProperty(JcrConstants.JCR_FROZENPRIMARYTYPE).getString());
            }
            else
            {
                dest = ((NodeImpl) parentDest).addNodeWithUuid(src.getName(), src.getProperty(JcrConstants.JCR_FROZENPRIMARYTYPE).getString(), uuid);
            }
        }
        
        PropertyIterator pit = src.getProperties();
        while (pit.hasNext())
        {
            Property p = pit.nextProperty();
            String name = p.getName();
            
            // Tests for protected and/or autocreated properties
            if (!PROTECTED_PROPERTIES.contains(name) && !name.startsWith("jcr:frozen") && !dest.hasProperty(name))
            {
                if (p.getDefinition().isMultiple())
                {
                    dest.setProperty(name, p.getValues());
                }
                else
                {
                    dest.setProperty(name, p.getValue());
                }
            }
        }
        
        NodeIterator nit = src.getNodes();
        while (nit.hasNext())
        {
            copyNode(dest, nit.nextNode());
        }
    }

    public void addLabel(String label, boolean moveIfPresent) throws AmetysRepositoryException
    {
        try
        {
            VersionHistory history = getVersionHistory();
            String versionName;

            if (_versionNode == null)
            {
                // not sticked to a particular version
                versionName = getBaseVersion().getName();
            }
            else
            {
                // sticked to label
                versionName = _versionNode.getName();
            }

            history.addVersionLabel(versionName, label, moveIfPresent);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to add label : " + label, e);
        }
    }

    public void removeLabel(String label) throws AmetysRepositoryException
    {
        try
        {
            VersionHistory history = getVersionHistory();
            history.removeVersionLabel(label);
        }
        catch (RepositoryException ex)
        {
            throw new AmetysRepositoryException("Unable to remove label : " + label, ex);
        }
    }

    public String[] getAllLabels() throws AmetysRepositoryException
    {
        try
        {
            return getVersionHistory().getVersionLabels();
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to retrieve list of labels", e);
        }
    }

    public String[] getLabels() throws AmetysRepositoryException
    {
        try
        {
            Version version = _versionNode;

            if (version == null)
            {
                // not sticked to a particular version
                version = getBaseVersion();
            }

            return version.getContainingHistory().getVersionLabels(version);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to retrieve list of labels for current version", e);
        }
    }


    public String[] getLabels(String revision) throws UnknownAmetysObjectException, AmetysRepositoryException
    {
        try
        {
            VersionHistory history = getVersionHistory();
            Version version = history.getVersion(revision);
            return history.getVersionLabels(version);
        }
        catch (VersionException e)
        {
            throw new UnknownAmetysObjectException("There's no revision " + revision, e);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to retrieve list of labels for current version", e);
        }
    }

    public String getRevision() throws AmetysRepositoryException
    {
        if (_versionNode == null)
        {
            // Current version
            return null;
        }

        try
        {
            return _versionNode.getName();
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get revision", e);
        }
    }
    
    @Override
    public Date getRevisionTimestamp() throws AmetysRepositoryException
    {
        if (_versionNode == null)
        {
            // Current version
            return null;
        }

        try
        {
            return _versionNode.getCreated().getTime();
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get revision date", e);
        }
    }
    
    @Override
    public Date getRevisionTimestamp(String revision) throws UnknownAmetysObjectException, AmetysRepositoryException
    {
        try
        {
            VersionHistory history = getVersionHistory();
            Version version = history.getVersion(revision);
            
            return version.getCreated().getTime();
        }
        catch (VersionException e)
        {
            throw new UnknownAmetysObjectException("There's no revision " + revision, e);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get revision date", e);
        }
    }

    public String[] getAllRevisions() throws AmetysRepositoryException
    {
        try
        {
            Map<String, Calendar> revisions = new HashMap<>();
            VersionIterator iterator = getVersionHistory().getAllVersions();
            
            while (iterator.hasNext())
            {
                Version version = iterator.nextVersion();
                String name = version.getName();
                if (!JcrConstants.JCR_ROOTVERSION.equals(name))
                {
                    revisions.put(name, version.getCreated());
                }
            }
            
            return revisions.entrySet().stream()
                .sorted(__VERSION_COMPARATOR)
                .map(Entry::getKey)
                .toArray(String[]::new);
        }
        catch (RepositoryException ex)
        {
            throw new AmetysRepositoryException("Unable to get revisions list", ex);
        }
    }

    @Override
    public ModifiableModelLessDataHolder getUnversionedDataHolder()
    {
        try
        {
            ModifiableRepositoryData repositoryData = new JCRRepositoryData(_baseNode.getNode("ametys-internal:unversioned"));
            Optional<LockableAmetysObject> lockableAmetysObject = Optional.of(this)
                                                                          .filter(LockableAmetysObject.class::isInstance)
                                                                          .map(LockableAmetysObject.class:: cast);
            return new DefaultModifiableModelLessDataHolder(_getFactory().getUnversionedDataTypeExtensionPoint(), repositoryData, lockableAmetysObject);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException(e);
        }
    }
    
    public Map<AnonymousOrAnyConnectedKeys, Set<String>> getProfilesForAnonymousAndAnyConnectedUser()
    {
        return ACLJCRAmetysObjectHelper.getProfilesForAnonymousAndAnyConnectedUser(getNode());
    }
    
    public Map<GroupIdentity, Map<UserOrGroup, Set<String>>> getProfilesForGroups(Set<GroupIdentity> groups)
    {
        return ACLJCRAmetysObjectHelper.getProfilesForGroups(getNode(), groups);
    }
    
    public Map<UserIdentity, Map<UserOrGroup, Set<String>>> getProfilesForUsers(UserIdentity user)
    {
        return ACLJCRAmetysObjectHelper.getProfilesForUsers(getNode(), user);
    }
    
    public void addAllowedProfilesForAnyConnectedUser(Set<String> profileIds)
    {
        ACLJCRAmetysObjectHelper.addAllowedProfilesForAnyConnectedUser(getNode(), profileIds);
    }
    
    public void removeAllowedProfilesForAnyConnectedUser(Set<String> profileIds)
    {
        ACLJCRAmetysObjectHelper.removeAllowedProfilesForAnyConnectedUser(getNode(), profileIds);
    }
    
    public void addDeniedProfilesForAnyConnectedUser(Set<String> profileIds)
    {
        ACLJCRAmetysObjectHelper.addDeniedProfilesForAnyConnectedUser(getNode(), profileIds);
    }
    
    public void removeDeniedProfilesForAnyConnectedUser(Set<String> profileIds)
    {
        ACLJCRAmetysObjectHelper.removeDeniedProfilesForAnyConnectedUser(getNode(), profileIds);
    }
    
    public void addAllowedProfilesForAnonymous(Set<String> profileIds)
    {
        ACLJCRAmetysObjectHelper.addAllowedProfilesForAnonymous(getNode(), profileIds);
    }
    
    public void removeAllowedProfilesForAnonymous(Set<String> profileIds)
    {
        ACLJCRAmetysObjectHelper.removeAllowedProfilesForAnonymous(getNode(), profileIds);
    }
    
    public void addDeniedProfilesForAnonymous(Set<String> profileIds)
    {
        ACLJCRAmetysObjectHelper.addDeniedProfilesForAnonymous(getNode(), profileIds);
    }
    
    public void removeDeniedProfilesForAnonymous(Set<String> profileIds)
    {
        ACLJCRAmetysObjectHelper.removeDeniedProfilesForAnonymous(getNode(), profileIds);
    }
    
    public void addAllowedUsers(Set<UserIdentity> users, String profileId)
    {
        ACLJCRAmetysObjectHelper.addAllowedUsers(users, getNode(), profileId);
    }
    
    public void removeAllowedUsers(Set<UserIdentity> users, String profileId)
    {
        ACLJCRAmetysObjectHelper.removeAllowedUsers(users, getNode(), profileId);
    }
    
    public void removeAllowedUsers(Set<UserIdentity> users)
    {
        ACLJCRAmetysObjectHelper.removeAllowedUsers(users, getNode());
    }
    
    public void addAllowedGroups(Set<GroupIdentity> groups, String profileId)
    {
        ACLJCRAmetysObjectHelper.addAllowedGroups(groups, getNode(), profileId);
    }
    
    public void removeAllowedGroups(Set<GroupIdentity> groups, String profileId)
    {
        ACLJCRAmetysObjectHelper.removeAllowedGroups(groups, getNode(), profileId);
    }
    
    public void removeAllowedGroups(Set<GroupIdentity> groups)
    {
        ACLJCRAmetysObjectHelper.removeAllowedGroups(groups, getNode());
    }
    
    public void addDeniedUsers(Set<UserIdentity> users, String profileId)
    {
        ACLJCRAmetysObjectHelper.addDeniedUsers(users, getNode(), profileId);
    }
    
    public void removeDeniedUsers(Set<UserIdentity> users, String profileId)
    {
        ACLJCRAmetysObjectHelper.removeDeniedUsers(users, getNode(), profileId);
    }
    
    public void removeDeniedUsers(Set<UserIdentity> users)
    {
        ACLJCRAmetysObjectHelper.removeDeniedUsers(users, getNode());
    }
    
    public void addDeniedGroups(Set<GroupIdentity> groups, String profileId)
    {
        ACLJCRAmetysObjectHelper.addDeniedGroups(groups, getNode(), profileId);
    }
    
    public void removeDeniedGroups(Set<GroupIdentity> groups, String profileId)
    {
        ACLJCRAmetysObjectHelper.removeDeniedGroups(groups, getNode(), profileId);
    }
    
    public void removeDeniedGroups(Set<GroupIdentity> groups)
    {
        ACLJCRAmetysObjectHelper.removeDeniedGroups(groups, getNode());
    }
    
    public boolean isInheritanceDisallowed()
    {
        return ACLJCRAmetysObjectHelper.isInheritanceDisallowed(getNode());
    }
    
    public void disallowInheritance(boolean disallow)
    {
        ACLJCRAmetysObjectHelper.disallowInheritance(getNode(), disallow);
    }
}
