/*
 *  Copyright 2012 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.jcr;

import javax.jcr.Node;

import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.lock.LockableAmetysObject;

/**
 * {@link DefaultAmetysObject} which is also a {@link LockableAmetysObject}.
 * @param <F> the actual type of factory.
 */
public class DefaultLockableAmetysObject<F extends DefaultLockableAmetysObjectFactory> extends DefaultAmetysObject<F> implements LockableAmetysObject
{
    private boolean _lockAlreadyChecked;
    
    /**
     * Creates an {@link DefaultLockableAmetysObject}.
     * @param node the node backing this {@link AmetysObject}
     * @param parentPath the parentPath in the Ametys hierarchy
     * @param factory the DefaultAmetysObjectFactory which created the AmetysObject
     */
    public DefaultLockableAmetysObject(Node node, String parentPath, F factory)
    {
        super(node, parentPath, factory);
    }
    
    @Override
    public void lock() throws AmetysRepositoryException
    {
        _getFactory().getLockComponent().lock(this);
        
        // the lock component immediately detached the lock token, so we have to check it again at next usage 
        _lockAlreadyChecked = false;
    }
    
    @Override
    public void unlock() throws AmetysRepositoryException
    {
        _getFactory().getLockComponent().unlock(this);
        
        // the lock component removed the lock token on unlock
        _lockAlreadyChecked = true;
    }
    
    @Override
    public void setLockInfoOnCurrentContext() throws AmetysRepositoryException
    {
        if (!_lockAlreadyChecked)
        {
            _getFactory().getLockComponent().setLockTokenOnCurrentSession(this);
            _lockAlreadyChecked = true;
        }
    }
    
    @Override
    public boolean isLocked() throws AmetysRepositoryException
    {
        return _getFactory().getLockComponent().isLocked(this);
    }
    
    @Override
    public UserIdentity getLockOwner() throws AmetysRepositoryException
    {
        return _getFactory().getLockComponent().getLockOwner(this);
    }
}
