/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.ametys.plugins.repository.jcr;

import java.util.Map;
import java.util.Set;

import javax.jcr.Node;

import org.ametys.core.group.GroupIdentity;
import org.ametys.core.right.ProfileAssignmentStorage.AnonymousOrAnyConnectedKeys;
import org.ametys.core.right.ProfileAssignmentStorage.UserOrGroup;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.ModifiableACLAmetysObject;
import org.ametys.plugins.repository.RepositoryIntegrityViolationException;
import org.ametys.plugins.repository.TraversableAmetysObject;
import org.ametys.plugins.repository.UnknownAmetysObjectException;

/**
 * Default implementation of a {@link TraversableAmetysObject}, backed by a JCR node.<br>
 * This implementation heavily relies on its {@link DefaultTraversableAmetysObjectFactory} counterpart.
 * @param <F> the actual type of factory.
 */
public class DefaultTraversableAmetysObject<F extends DefaultTraversableAmetysObjectFactory> extends SimpleAmetysObject<F> implements JCRTraversableAmetysObject, ModifiableACLAmetysObject
{
    private DefaultTraversableAmetysObjectFactory _traversableFactory;
    
    /**
     * Creates a DefaultTraversableAmetysObject.
     * @param node the node backing this {@link AmetysObject}.
     * @param parentPath the parent path in the Ametys hierarchy.
     * @param factory the {@link DefaultTraversableAmetysObjectFactory} which creates the AmetysObject.
     */
    public DefaultTraversableAmetysObject(Node node, String parentPath, F factory)
    {
        super(node, parentPath, factory);
        _traversableFactory = factory;
    }

    public boolean hasChild(String name) throws AmetysRepositoryException
    {
        return _traversableFactory.hasChild(this, name);
    }

    @SuppressWarnings("unchecked")
    public <A extends AmetysObject> A createChild(String name, String type) throws AmetysRepositoryException, RepositoryIntegrityViolationException
    {
        return (A) _traversableFactory.createChild(this, name, type);
    }
    
    @SuppressWarnings("unchecked")
    public <A extends AmetysObject> A getChild(String path) throws AmetysRepositoryException, UnknownAmetysObjectException
    {
        return (A) _traversableFactory.getChild(this, path);
    }

    public <A extends AmetysObject> AmetysObjectIterable<A> getChildren() throws AmetysRepositoryException
    {
        return _traversableFactory.getChildren(this);
    }

    public Map<AnonymousOrAnyConnectedKeys, Set<String>> getProfilesForAnonymousAndAnyConnectedUser()
    {
        return ACLJCRAmetysObjectHelper.getProfilesForAnonymousAndAnyConnectedUser(getNode());
    }
    
    public Map<GroupIdentity, Map<UserOrGroup, Set<String>>> getProfilesForGroups(Set<GroupIdentity> groups)
    {
        return ACLJCRAmetysObjectHelper.getProfilesForGroups(getNode(), groups);
    }
    
    public Map<UserIdentity, Map<UserOrGroup, Set<String>>> getProfilesForUsers(UserIdentity user)
    {
        return ACLJCRAmetysObjectHelper.getProfilesForUsers(getNode(), user);
    }
    
    public void addAllowedProfilesForAnyConnectedUser(Set<String> profileIds)
    {
        ACLJCRAmetysObjectHelper.addAllowedProfilesForAnyConnectedUser(getNode(), profileIds);
    }
    
    public void removeAllowedProfilesForAnyConnectedUser(Set<String> profileIds)
    {
        ACLJCRAmetysObjectHelper.removeAllowedProfilesForAnyConnectedUser(getNode(), profileIds);
    }
    
    public void addDeniedProfilesForAnyConnectedUser(Set<String> profileIds)
    {
        ACLJCRAmetysObjectHelper.addDeniedProfilesForAnyConnectedUser(getNode(), profileIds);
    }
    
    public void removeDeniedProfilesForAnyConnectedUser(Set<String> profileIds)
    {
        ACLJCRAmetysObjectHelper.removeDeniedProfilesForAnyConnectedUser(getNode(), profileIds);
    }
    
    public void addAllowedProfilesForAnonymous(Set<String> profileIds)
    {
        ACLJCRAmetysObjectHelper.addAllowedProfilesForAnonymous(getNode(), profileIds);
    }
    
    public void removeAllowedProfilesForAnonymous(Set<String> profileIds)
    {
        ACLJCRAmetysObjectHelper.removeAllowedProfilesForAnonymous(getNode(), profileIds);
    }
    
    public void addDeniedProfilesForAnonymous(Set<String> profileIds)
    {
        ACLJCRAmetysObjectHelper.addDeniedProfilesForAnonymous(getNode(), profileIds);
    }
    
    public void removeDeniedProfilesForAnonymous(Set<String> profileIds)
    {
        ACLJCRAmetysObjectHelper.removeDeniedProfilesForAnonymous(getNode(), profileIds);
    }
    
    public void addAllowedUsers(Set<UserIdentity> users, String profileId)
    {
        ACLJCRAmetysObjectHelper.addAllowedUsers(users, getNode(), profileId);
    }
    
    public void removeAllowedUsers(Set<UserIdentity> users, String profileId)
    {
        ACLJCRAmetysObjectHelper.removeAllowedUsers(users, getNode(), profileId);
    }
    
    public void removeAllowedUsers(Set<UserIdentity> users)
    {
        ACLJCRAmetysObjectHelper.removeAllowedUsers(users, getNode());
    }
    
    public void addAllowedGroups(Set<GroupIdentity> groups, String profileId)
    {
        ACLJCRAmetysObjectHelper.addAllowedGroups(groups, getNode(), profileId);
    }
    
    public void removeAllowedGroups(Set<GroupIdentity> groups, String profileId)
    {
        ACLJCRAmetysObjectHelper.removeAllowedGroups(groups, getNode(), profileId);
    }
    
    public void removeAllowedGroups(Set<GroupIdentity> groups)
    {
        ACLJCRAmetysObjectHelper.removeAllowedGroups(groups, getNode());
    }
    
    public void addDeniedUsers(Set<UserIdentity> users, String profileId)
    {
        ACLJCRAmetysObjectHelper.addDeniedUsers(users, getNode(), profileId);
    }
    
    public void removeDeniedUsers(Set<UserIdentity> users, String profileId)
    {
        ACLJCRAmetysObjectHelper.removeDeniedUsers(users, getNode(), profileId);
    }
    
    public void removeDeniedUsers(Set<UserIdentity> users)
    {
        ACLJCRAmetysObjectHelper.removeDeniedUsers(users, getNode());
    }
    
    public void addDeniedGroups(Set<GroupIdentity> groups, String profileId)
    {
        ACLJCRAmetysObjectHelper.addDeniedGroups(groups, getNode(), profileId);
    }
    
    public void removeDeniedGroups(Set<GroupIdentity> groups, String profileId)
    {
        ACLJCRAmetysObjectHelper.removeDeniedGroups(groups, getNode(), profileId);
    }
    
    public void removeDeniedGroups(Set<GroupIdentity> groups)
    {
        ACLJCRAmetysObjectHelper.removeDeniedGroups(groups, getNode());
    }
    
    public boolean isInheritanceDisallowed()
    {
        return ACLJCRAmetysObjectHelper.isInheritanceDisallowed(getNode());
    }
    
    public void disallowInheritance(boolean disallow)
    {
        ACLJCRAmetysObjectHelper.disallowInheritance(getNode(), disallow);
    }
}
