/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.jcr;

import javax.jcr.Node;

import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.UnknownAmetysObjectException;

/**
 * Default implementation of an {@link JCRAmetysObjectFactory},
 * handling {@link DefaultTraversableAmetysObject}.<br>
 */
public class DefaultTraversableAmetysObjectFactory extends SimpleAmetysObjectFactory
{
    @Override
    @SuppressWarnings("unchecked")
    public DefaultTraversableAmetysObject getAmetysObject(Node node, String parentPath) throws AmetysRepositoryException
    {
        return new DefaultTraversableAmetysObject(node, parentPath, this);
    }
    
    /**
     * Returns the {@link AmetysObject} at the given subPath,
     * relative to the given {@link DefaultTraversableAmetysObject}.
     * @param <A> the actual type of {@link AmetysObject}.
     * @param object the context {@link DefaultTraversableAmetysObject}.
     * @param path the sub path. Cannot be <code>null</code>, empty or absolute.
     * @return the {@link AmetysObject} at the given subPath,
     *         relative to the given {@link DefaultTraversableAmetysObject}.
     * @throws AmetysRepositoryException if an error occurs.
     * @throws UnknownAmetysObjectException if no such object exists.
     */
    public <A extends AmetysObject> A getChild(DefaultTraversableAmetysObject object, String path) throws AmetysRepositoryException, UnknownAmetysObjectException
    {
        return TraversableAmetysObjectHelper.<A>getChild(object, this, path, _resolver, getLogger());
    }
    
    /**
     * Returns all children of the given {@link DefaultTraversableAmetysObject}.
     * @param <A> the actual type of {@link AmetysObject}s
     * @param object a {@link DefaultTraversableAmetysObject}.
     * @return a List containing all children object in the Ametys hierarchy.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public <A extends AmetysObject> AmetysObjectIterable<A> getChildren(DefaultTraversableAmetysObject object) throws AmetysRepositoryException
    {
        return TraversableAmetysObjectHelper.getChildren(object, this, _resolver, getLogger());
    }
    
    /**
     * Tests if a given object has a child with a given name.
     * @param object the context object.
     * @param name the name to test.
     * @return <code>true</code> is the given object has a child with the given name,
     *         <code>false</code> otherwise.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public boolean hasChild(DefaultTraversableAmetysObject object, String name) throws AmetysRepositoryException
    {
        return TraversableAmetysObjectHelper.hasChild(object, name, _ametysFactoryExtensionPoint, getLogger());
    }
    
    /**
     * Creates a child to the given object.
     * @param <A> the actual type of {@link AmetysObject}.
     * @param object the parent {@link AmetysObject}.
     * @param name the new object's name.
     * @param type the new object's type.
     * @return the newly created {@link AmetysObject}.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public <A extends AmetysObject> A createChild(DefaultTraversableAmetysObject object, String name, String type) throws AmetysRepositoryException
    {
        return TraversableAmetysObjectHelper.<A>createChild(object, this, name, type, _ametysFactoryExtensionPoint, _resolver, getLogger());
    }
}
