/*
 *  Copyright 2012 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.jcr;

import java.util.Date;

import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.metadata.CompositeMetadata;
import org.ametys.plugins.repository.metadata.MetadataAwareAmetysObject;
import org.ametys.plugins.repository.metadata.ModifiableCompositeMetadata;
import org.ametys.plugins.repository.metadata.ModifiableMetadataAwareAmetysObject;
import org.ametys.plugins.repository.metadata.UnknownMetadataException;

/**
 * Helper class providing methods to get and set Dublin Core metadata on {@link MetadataAwareAmetysObject}s.
 */
public final class DublinCoreHelper
{
    
    /** Constant for Dublin Core metadata container name. */
    public static final String METADATA_DC_CONTAINER = "dc";
    
    /** Constant for Dublin Core title metadata. */
    public static final String METADATA_DC_TITLE = "dc_title";
    
    /** Constant for Dublin Core creator metadata. */
    public static final String METADATA_DC_CREATOR = "dc_creator";
    
    /** Constant for Dublin Core subject metadata. */
    public static final String METADATA_DC_SUBJECT = "dc_subject";
    
    /** Constant for Dublin Core description metadata. */
    public static final String METADATA_DC_DESCRIPTION = "dc_description";
    
    /** Constant for Dublin Core publisher metadata. */
    public static final String METADATA_DC_PUBLISHER = "dc_publisher";
    
    /** Constant for Dublin Core contributor metadata. */
    public static final String METADATA_DC_CONTRIBUTOR = "dc_contributor";
    
    /** Constant for Dublin Core date metadata. */
    public static final String METADATA_DC_DATE = "dc_date";
    
    /** Constant for Dublin Core type metadata. */
    public static final String METADATA_DC_TYPE = "dc_type";
    
    /** Constant for Dublin Core title Format. */
    public static final String METADATA_DC_FORMAT = "dc_format";
    
    /** Constant for Dublin Core identifier metadata. */
    public static final String METADATA_DC_IDENTIFIER = "dc_identifier";
    
    /** Constant for Dublin Core source metadata. */
    public static final String METADATA_DC_SOURCE = "dc_source";
    
    /** Constant for Dublin Core language metadata. */
    public static final String METADATA_DC_LANGUAGE = "dc_language";
    
    /** Constant for Dublin Core relation metadata. */
    public static final String METADATA_DC_RELATION = "dc_relation";
    
    /** Constant for Dublin Core coverage metadata. */
    public static final String METADATA_DC_COVERAGE = "dc_coverage";
    
    /** Constant for Dublin Core rights metadata. */
    public static final String METADATA_DC_RIGHTS = "dc_rights";
    
    private DublinCoreHelper()
    {
        // Hides default constructor.
    }
    
    ///////
    // DublinCoreAwareAmetysObject methods.
    ///////
    
    /**
     * Get the Dublin Core title.
     * @param object the object.
     * @return the Dublin Core title. Can be null.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public static String getDCTitle(MetadataAwareAmetysObject object) throws AmetysRepositoryException
    {
        return getDCTitle(object, null);
    }
    
    /**
     * Get the Dublin Core title.
     * @param object the object.
     * @param defaultValue the default title.
     * @return the Dublin Core title.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public static String getDCTitle(MetadataAwareAmetysObject object, String defaultValue) throws AmetysRepositoryException
    {
        try
        {
            return getDCCompositeMetadata(object).getString(METADATA_DC_TITLE, defaultValue);
        }
        catch (UnknownMetadataException e)
        {
            return defaultValue;
        }
    }
    
    /**
     * Get the Dublin Core creator.
     * @param object the object.
     * @return the Dublin Core creator. Can be null.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public static String getDCCreator(MetadataAwareAmetysObject object) throws AmetysRepositoryException
    {
        return getDCCreator(object, null);
    }
    
    /**
     * Get the Dublin Core creator.
     * @param object the object.
     * @param defaultValue the default creator.
     * @return the Dublin Core creator.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public static String getDCCreator(MetadataAwareAmetysObject object, String defaultValue) throws AmetysRepositoryException
    {
        try
        {
            return getDCCompositeMetadata(object).getString(METADATA_DC_CREATOR, defaultValue);
        }
        catch (UnknownMetadataException e)
        {
            return defaultValue;
        }
    }
    
    /**
     * Get the Dublin Core subject.
     * @param object the object.
     * @return the Dublin Core subject. Can be null.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public static String[] getDCSubject(MetadataAwareAmetysObject object) throws AmetysRepositoryException
    {
        return getDCSubject(object, null);
    }
    
    /**
     * Get the Dublin Core subject.
     * @param object the object.
     * @param defaultValue the default subject.
     * @return the Dublin Core subject.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public static String[] getDCSubject(MetadataAwareAmetysObject object, String[] defaultValue) throws AmetysRepositoryException
    {
        try
        {
            return getDCCompositeMetadata(object).getStringArray(METADATA_DC_SUBJECT, defaultValue);
        }
        catch (UnknownMetadataException e)
        {
            return defaultValue;
        }
    }
    
    /**
     * Get the Dublin Core description.
     * @param object the object.
     * @return the Dublin Core description. Can be null.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public static String getDCDescription(MetadataAwareAmetysObject object) throws AmetysRepositoryException
    {
        return getDCDescription(object, null);
    }
    
    /**
     * Get the Dublin Core description.
     * @param object the object.
     * @param defaultValue the default description.
     * @return the Dublin Core description.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public static String getDCDescription(MetadataAwareAmetysObject object, String defaultValue) throws AmetysRepositoryException
    {
        try
        {
            return getDCCompositeMetadata(object).getString(METADATA_DC_DESCRIPTION, defaultValue);
        }
        catch (UnknownMetadataException e)
        {
            return defaultValue;
        }
    }
    
    /**
     * Get the Dublin Core publisher.
     * @param object the object.
     * @return the Dublin Core publisher. Can be null.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public static String getDCPublisher(MetadataAwareAmetysObject object) throws AmetysRepositoryException
    {
        return getDCPublisher(object, null);
    }
    
    /**
     * Get the Dublin Core publisher.
     * @param object the object.
     * @param defaultValue the default publisher.
     * @return the Dublin Core publisher.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public static String getDCPublisher(MetadataAwareAmetysObject object, String defaultValue) throws AmetysRepositoryException
    {
        try
        {
            return getDCCompositeMetadata(object).getString(METADATA_DC_PUBLISHER, defaultValue);
        }
        catch (UnknownMetadataException e)
        {
            return defaultValue;
        }
    }
    
    /**
     * Get the Dublin Core contributor.
     * @param object the object.
     * @return the Dublin Core contributor. Can be null.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public static String getDCContributor(MetadataAwareAmetysObject object) throws AmetysRepositoryException
    {
        return getDCContributor(object, null);
    }
    
    /**
     * Get the Dublin Core contributor.
     * @param object the object.
     * @param defaultValue the default contributor.
     * @return the Dublin Core contributor.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public static String getDCContributor(MetadataAwareAmetysObject object, String defaultValue) throws AmetysRepositoryException
    {
        try
        {
            return getDCCompositeMetadata(object).getString(METADATA_DC_CONTRIBUTOR, defaultValue);
        }
        catch (UnknownMetadataException e)
        {
            return defaultValue;
        }
    }
    
    /**
     * Get the Dublin Core date.
     * @param object the object.
     * @return the Dublin Core date. Can be null.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public static Date getDCDate(MetadataAwareAmetysObject object) throws AmetysRepositoryException
    {
        return getDCDate(object, null);
    }
    
    /**
     * Get the Dublin Core date.
     * @param object the object.
     * @param defaultValue the default date.
     * @return the Dublin Core date.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public static Date getDCDate(MetadataAwareAmetysObject object, Date defaultValue) throws AmetysRepositoryException
    {
        try
        {
            return getDCCompositeMetadata(object).getDate(METADATA_DC_DATE, defaultValue);
        }
        catch (UnknownMetadataException e)
        {
            return defaultValue;
        }
    }
    
    /**
     * Get the Dublin Core type.
     * @param object the object.
     * @return the Dublin Core type. Can be null.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public static String getDCType(MetadataAwareAmetysObject object) throws AmetysRepositoryException
    {
        return getDCType(object, null);
    }
    
    /**
     * Get the Dublin Core type.
     * @param object the object.
     * @param defaultValue the default type.
     * @return the Dublin Core type.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public static String getDCType(MetadataAwareAmetysObject object, String defaultValue) throws AmetysRepositoryException
    {
        try
        {
            return getDCCompositeMetadata(object).getString(METADATA_DC_TYPE, defaultValue);
        }
        catch (UnknownMetadataException e)
        {
            return defaultValue;
        }
    }
    
    /**
     * Get the Dublin Core format.
     * @param object the object.
     * @return the Dublin Core format. Can be null.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public static String getDCFormat(MetadataAwareAmetysObject object) throws AmetysRepositoryException
    {
        return getDCFormat(object, null);
    }
    
    /**
     * Get the Dublin Core format.
     * @param object the object.
     * @param defaultValue the default format.
     * @return the Dublin Core format.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public static String getDCFormat(MetadataAwareAmetysObject object, String defaultValue) throws AmetysRepositoryException
    {
        try
        {
            return getDCCompositeMetadata(object).getString(METADATA_DC_FORMAT, defaultValue);
        }
        catch (UnknownMetadataException e)
        {
            return defaultValue;
        }
    }
    
    /**
     * Get the Dublin Core identifier.
     * @param object the object.
     * @return the Dublin Core identifier. Can be null.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public static String getDCIdentifier(MetadataAwareAmetysObject object) throws AmetysRepositoryException
    {
        return getDCIdentifier(object, null);
    }
    
    /**
     * Get the Dublin Core identifier.
     * @param object the object.
     * @param defaultValue the default identifier.
     * @return the Dublin Core identifier.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public static String getDCIdentifier(MetadataAwareAmetysObject object, String defaultValue) throws AmetysRepositoryException
    {
        try
        {
            return getDCCompositeMetadata(object).getString(METADATA_DC_IDENTIFIER, defaultValue);
        }
        catch (UnknownMetadataException e)
        {
            return defaultValue;
        }
    }
    
    /**
     * Get the Dublin Core source.
     * @param object the object.
     * @return the Dublin Core source. Can be null.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public static String getDCSource(MetadataAwareAmetysObject object) throws AmetysRepositoryException
    {
        return getDCSource(object, null);
    }
    
    /**
     * Get the Dublin Core source.
     * @param object the object.
     * @param defaultValue the default source.
     * @return the Dublin Core source.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public static String getDCSource(MetadataAwareAmetysObject object, String defaultValue) throws AmetysRepositoryException
    {
        try
        {
            return getDCCompositeMetadata(object).getString(METADATA_DC_SOURCE, defaultValue);
        }
        catch (UnknownMetadataException e)
        {
            return defaultValue;
        }
    }
    
    /**
     * Get the Dublin Core language.
     * @param object the object.
     * @return the Dublin Core language. Can be null.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public static String getDCLanguage(MetadataAwareAmetysObject object) throws AmetysRepositoryException
    {
        return getDCLanguage(object, null);
    }
    
    /**
     * Get the Dublin Core language.
     * @param object the object.
     * @param defaultValue the default language.
     * @return the Dublin Core language.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public static String getDCLanguage(MetadataAwareAmetysObject object, String defaultValue) throws AmetysRepositoryException
    {
        try
        {
            return getDCCompositeMetadata(object).getString(METADATA_DC_LANGUAGE, defaultValue);
        }
        catch (UnknownMetadataException e)
        {
            return defaultValue;
        }
    }
    
    /**
     * Get the Dublin Core relation.
     * @param object the object.
     * @return the Dublin Core relation. Can be null.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public static String getDCRelation(MetadataAwareAmetysObject object) throws AmetysRepositoryException
    {
        return getDCRelation(object, null);
    }
    
    /**
     * Get the Dublin Core relation.
     * @param object the object.
     * @param defaultValue the default relation.
     * @return the Dublin Core relation.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public static String getDCRelation(MetadataAwareAmetysObject object, String defaultValue) throws AmetysRepositoryException
    {
        try
        {
            return getDCCompositeMetadata(object).getString(METADATA_DC_RELATION, defaultValue);
        }
        catch (UnknownMetadataException e)
        {
            return defaultValue;
        }
    }
    
    /**
     * Get the Dublin Core coverage.
     * @param object the object.
     * @return the Dublin Core coverage. Can be null.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public static String getDCCoverage(MetadataAwareAmetysObject object) throws AmetysRepositoryException
    {
        return getDCCoverage(object, null);
    }
    
    /**
     * Get the Dublin Core coverage.
     * @param object the object.
     * @param defaultValue the default coverage.
     * @return the Dublin Core coverage.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public static String getDCCoverage(MetadataAwareAmetysObject object, String defaultValue) throws AmetysRepositoryException
    {
        try
        {
            return getDCCompositeMetadata(object).getString(METADATA_DC_COVERAGE, defaultValue);
        }
        catch (UnknownMetadataException e)
        {
            return defaultValue;
        }
    }
    
    /**
     * Get the Dublin Core rights.
     * @param object the object.
     * @return the Dublin Core rights. Can be null.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public static String getDCRights(MetadataAwareAmetysObject object) throws AmetysRepositoryException
    {
        return getDCRights(object, null);
    }
    
    /**
     * Get the Dublin Core rights.
     * @param object the object.
     * @param defaultValue the default rights.
     * @return the Dublin Core rights.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public static String getDCRights(MetadataAwareAmetysObject object, String defaultValue) throws AmetysRepositoryException
    {
        try
        {
            return getDCCompositeMetadata(object).getString(METADATA_DC_RIGHTS, defaultValue);
        }
        catch (UnknownMetadataException e)
        {
            return defaultValue;
        }
    }
    
    ///////
    // ModifiableDublinCoreAwareAmetysObject methods.
    ///////
    
    /**
     * Set the Dublin Core title.
     * @param object the ametys object.
     * @param title the Dublin Core title to set. Can be null.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public static void setDCTitle(ModifiableMetadataAwareAmetysObject object, String title) throws AmetysRepositoryException
    {
        if (title != null)
        {
            getModifiableDCCompositeMetadata(object).setMetadata(METADATA_DC_TITLE, title);
        }
        else
        {
            if (getModifiableDCCompositeMetadata(object).hasMetadata(METADATA_DC_TITLE))
            {
                getModifiableDCCompositeMetadata(object).removeMetadata(METADATA_DC_TITLE);
            }
        }
    }
    
    /**
     * Set the Dublin Core creator.
     * @param object the ametys object.
     * @param creator the Dublin Core creator to set. Can be null.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public static void setDCCreator(ModifiableMetadataAwareAmetysObject object, String creator) throws AmetysRepositoryException
    {
        if (creator != null)
        {
            getModifiableDCCompositeMetadata(object).setMetadata(METADATA_DC_CREATOR, creator);
        }
        else
        {
            if (getModifiableDCCompositeMetadata(object).hasMetadata(METADATA_DC_CREATOR))
            {
                getModifiableDCCompositeMetadata(object).removeMetadata(METADATA_DC_CREATOR);
            }
        }
    }
    
    /**
     * Set the Dublin Core subject.
     * @param object the ametys object.
     * @param subject the Dublin Core subject to set. Can be null.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public static void setDCSubject(ModifiableMetadataAwareAmetysObject object, String[] subject) throws AmetysRepositoryException
    {
        if (subject != null)
        {
            getModifiableDCCompositeMetadata(object).setMetadata(METADATA_DC_SUBJECT, subject);
        }
        else
        {
            if (getModifiableDCCompositeMetadata(object).hasMetadata(METADATA_DC_SUBJECT))
            {
                getModifiableDCCompositeMetadata(object).removeMetadata(METADATA_DC_SUBJECT);
            }
        }
    }
    
    /**
     * Set the Dublin Core description.
     * @param object the ametys object.
     * @param description the Dublin Core description to set. Can be null.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public static void setDCDescription(ModifiableMetadataAwareAmetysObject object, String description) throws AmetysRepositoryException
    {
        if (description != null)
        {
            getModifiableDCCompositeMetadata(object).setMetadata(METADATA_DC_DESCRIPTION, description);
        }
        else
        {
            if (getModifiableDCCompositeMetadata(object).hasMetadata(METADATA_DC_DESCRIPTION))
            {
                getModifiableDCCompositeMetadata(object).removeMetadata(METADATA_DC_DESCRIPTION);
            }
        }
    }
    
    /**
     * Set the Dublin Core publisher.
     * @param object the ametys object.
     * @param publisher the Dublin Core publisher to set. Can be null.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public static void setDCPublisher(ModifiableMetadataAwareAmetysObject object, String publisher) throws AmetysRepositoryException
    {
        if (publisher != null)
        {
            getModifiableDCCompositeMetadata(object).setMetadata(METADATA_DC_PUBLISHER, publisher);
        }
        else
        {
            if (getModifiableDCCompositeMetadata(object).hasMetadata(METADATA_DC_PUBLISHER))
            {
                getModifiableDCCompositeMetadata(object).removeMetadata(METADATA_DC_PUBLISHER);
            }
        }
    }
    
    /**
     * Set the Dublin Core contributor.
     * @param object the ametys object.
     * @param contributor the Dublin Core contributor to set. Can be null.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public static void setDCContributor(ModifiableMetadataAwareAmetysObject object, String contributor) throws AmetysRepositoryException
    {
        if (contributor != null)
        {
            getModifiableDCCompositeMetadata(object).setMetadata(METADATA_DC_CONTRIBUTOR, contributor);
        }
        else
        {
            if (getModifiableDCCompositeMetadata(object).hasMetadata(METADATA_DC_CONTRIBUTOR))
            {
                getModifiableDCCompositeMetadata(object).removeMetadata(METADATA_DC_CONTRIBUTOR);
            }
        }
    }
    
    /**
     * Set the Dublin Core date.
     * @param object the ametys object.
     * @param date the Dublin Core date to set. Can be null.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public static void setDCDate(ModifiableMetadataAwareAmetysObject object, Date date) throws AmetysRepositoryException
    {
        if (date != null)
        {
            getModifiableDCCompositeMetadata(object).setMetadata(METADATA_DC_DATE, date);
        }
        else
        {
            if (getModifiableDCCompositeMetadata(object).hasMetadata(METADATA_DC_DATE))
            {
                getModifiableDCCompositeMetadata(object).removeMetadata(METADATA_DC_DATE);
            }
        }
    }
    
    /**
     * Set the Dublin Core type.
     * @param object the ametys object.
     * @param type the Dublin Core type to set. Can be null.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public static void setDCType(ModifiableMetadataAwareAmetysObject object, String type) throws AmetysRepositoryException
    {
        if (type != null)
        {
            getModifiableDCCompositeMetadata(object).setMetadata(METADATA_DC_TYPE, type);
        }
        else
        {
            if (getModifiableDCCompositeMetadata(object).hasMetadata(METADATA_DC_TYPE))
            {
                getModifiableDCCompositeMetadata(object).removeMetadata(METADATA_DC_TYPE);
            }
        }
    }
    
    /**
     * Set the Dublin Core format.
     * @param object the ametys object.
     * @param format the Dublin Core format to set. Can be null.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public static void setDCFormat(ModifiableMetadataAwareAmetysObject object, String format) throws AmetysRepositoryException
    {
        if (format != null)
        {
            getModifiableDCCompositeMetadata(object).setMetadata(METADATA_DC_FORMAT, format);
        }
        else
        {
            if (getModifiableDCCompositeMetadata(object).hasMetadata(METADATA_DC_FORMAT))
            {
                getModifiableDCCompositeMetadata(object).removeMetadata(METADATA_DC_FORMAT);
            }
        }
    }
    
    /**
     * Set the Dublin Core identifier.
     * @param object the ametys object.
     * @param identifier the Dublin Core identifier to set. Can be null.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public static void setDCIdentifier(ModifiableMetadataAwareAmetysObject object, String identifier) throws AmetysRepositoryException
    {
        if (identifier != null)
        {
            getModifiableDCCompositeMetadata(object).setMetadata(METADATA_DC_IDENTIFIER, identifier);
        }
        else
        {
            if (getModifiableDCCompositeMetadata(object).hasMetadata(METADATA_DC_IDENTIFIER))
            {
                getModifiableDCCompositeMetadata(object).removeMetadata(METADATA_DC_IDENTIFIER);
            }
        }
    }
    
    /**
     * Set the Dublin Core source.
     * @param object the ametys object.
     * @param source the Dublin Core source to set. Can be null.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public static void setDCSource(ModifiableMetadataAwareAmetysObject object, String source) throws AmetysRepositoryException
    {
        if (source != null)
        {
            getModifiableDCCompositeMetadata(object).setMetadata(METADATA_DC_SOURCE, source);
        }
        else
        {
            if (getModifiableDCCompositeMetadata(object).hasMetadata(METADATA_DC_SOURCE))
            {
                getModifiableDCCompositeMetadata(object).removeMetadata(METADATA_DC_SOURCE);
            }
        }
    }
    
    /**
     * Set the Dublin Core language.
     * @param object the ametys object.
     * @param language the Dublin Core language to set. Can be null.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public static void setDCLanguage(ModifiableMetadataAwareAmetysObject object, String language) throws AmetysRepositoryException
    {
        if (language != null)
        {
            getModifiableDCCompositeMetadata(object).setMetadata(METADATA_DC_LANGUAGE, language);
        }
        else
        {
            if (getModifiableDCCompositeMetadata(object).hasMetadata(METADATA_DC_LANGUAGE))
            {
                getModifiableDCCompositeMetadata(object).removeMetadata(METADATA_DC_LANGUAGE);
            }
        }
    }
    
    /**
     * Set the Dublin Core relation.
     * @param object the ametys object.
     * @param relation the Dublin Core relation to set. Can be null.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public static void setDCRelation(ModifiableMetadataAwareAmetysObject object, String relation) throws AmetysRepositoryException
    {
        if (relation != null)
        {
            getModifiableDCCompositeMetadata(object).setMetadata(METADATA_DC_RELATION, relation);
        }
        else
        {
            if (getModifiableDCCompositeMetadata(object).hasMetadata(METADATA_DC_RELATION))
            {
                getModifiableDCCompositeMetadata(object).removeMetadata(METADATA_DC_RELATION);
            }
        }
    }
    
    /**
     * Set the Dublin Core coverage.
     * @param object the ametys object.
     * @param coverage the Dublin Core coverage to set. Can be null.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public static void setDCCoverage(ModifiableMetadataAwareAmetysObject object, String coverage) throws AmetysRepositoryException
    {
        if (coverage != null)
        {
            getModifiableDCCompositeMetadata(object).setMetadata(METADATA_DC_COVERAGE, coverage);
        }
        else
        {
            if (getModifiableDCCompositeMetadata(object).hasMetadata(METADATA_DC_COVERAGE))
            {
                getModifiableDCCompositeMetadata(object).removeMetadata(METADATA_DC_COVERAGE);
            }
        }
    }
    
    /**
     * Set the Dublin Core rights.
     * @param object the ametys object.
     * @param rights the Dublin Core rights to set. Can be null.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public static void setDCRights(ModifiableMetadataAwareAmetysObject object, String rights) throws AmetysRepositoryException
    {
        if (rights != null)
        {
            getModifiableDCCompositeMetadata(object).setMetadata(METADATA_DC_RIGHTS, rights);
        }
        else
        {
            if (getModifiableDCCompositeMetadata(object).hasMetadata(METADATA_DC_RIGHTS))
            {
                getModifiableDCCompositeMetadata(object).removeMetadata(METADATA_DC_RIGHTS);
            }
        }
    }
    
    /**
     * Get the Dublin Core composite metadata of the given ametys object.
     * @param object the ametys object.
     * @return Dublin Core composite metadata.
     * @throws UnknownMetadataException if the CompositeMetadata does not exist.
     */
    static CompositeMetadata getDCCompositeMetadata(MetadataAwareAmetysObject object) throws UnknownMetadataException
    {
        return object.getMetadataHolder().getCompositeMetadata(METADATA_DC_CONTAINER);
    }
    
    /**
     * Get the Dublin Core modifiable composite metadata of the given ametys object,
     * creating it if it doesn't exist.
     * @param object the ametys object.
     * @return Dublin Core composite metadata.
     * @throws UnknownMetadataException if the CompositeMetadata does not exist (should never happen).
     */
    static ModifiableCompositeMetadata getModifiableDCCompositeMetadata(ModifiableMetadataAwareAmetysObject object) throws UnknownMetadataException
    {
        return object.getMetadataHolder().getCompositeMetadata(METADATA_DC_CONTAINER, true);
    }
    
}
