/*
 *  Copyright 2012 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.jcr;

import javax.jcr.Node;
import javax.jcr.PathNotFoundException;
import javax.jcr.RepositoryException;
import javax.jcr.Value;
import javax.jcr.lock.Lock;
import javax.jcr.lock.LockManager;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.logger.AbstractLogEnabled;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.lock.UnlockHelper;

/**
 * Component that provides methods for lock management on {@link JCRAmetysObject}s.
 */
public class LockComponent extends AbstractLogEnabled implements Serviceable, Component
{
    
    /** The avalon component role. */
    public static final String ROLE = LockComponent.class.getName();
    
    /** The unlock helper. */
    protected UnlockHelper _unlockHelper;
    
    /** The current user provider. */
    protected CurrentUserProvider _currentUserProvider;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _unlockHelper = (UnlockHelper) manager.lookup(UnlockHelper.ROLE);
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
    }
    
    /**
     * Register a locked content for automatic unlocking
     * @param object the locked {@link AmetysObject}
     */
    public void addLockedContent(AmetysObject object)
    {
        _unlockHelper.scheduleUnlocking(object);
    }
    
    /**
     * Unregister a locked content for automatic unlocking
     * @param object the locked {@link AmetysObject}
     */
    public void removeLockedContent(AmetysObject object)
    {
        _unlockHelper.cancelUnlocking(object);
    }
    
    /**
     * Provides the current user.
     * @return the current user.
     */
    public UserIdentity getCurrentUser()
    {
        return _currentUserProvider.getUser();
    }
    
    /**
     * Lock a {@link JCRAmetysObject}.
     * @param object the object to lock.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public void lock(JCRAmetysObject object) throws AmetysRepositoryException
    {
        try
        {
            Node node = getNode(object);
            
            if (!node.isCheckedOut())
            {
                throw new AmetysRepositoryException("Unable to lock a checked-in Node");
            }
            
            LockManager lockManager = node.getSession().getWorkspace().getLockManager();
            
            Lock lock = lockManager.lock(node.getPath(), false, false, Long.MAX_VALUE, null);
            node.setProperty(RepositoryConstants.METADATA_LOCKTOKEN, lock.getLockToken());
            
            UserIdentity currentUser = getCurrentUser();
            if (currentUser != null)
            {
                node.setProperty(RepositoryConstants.METADATA_LOCKOWNER, UserIdentity.userIdentityToString(currentUser));
            }
            node.getSession().save();

            // On détache immédiatement le lock de la session, pour permettre
            // les utilisations concurrentes
            lockManager.removeLockToken(lock.getLockToken());

            addLockedContent(object);
        }
        catch (RepositoryException ex)
        {
            throw new AmetysRepositoryException("Unable to lock object " + object, ex);
        }
    }
    
    /**
     * Unlock a {@link JCRAmetysObject}.
     * @param object the object to unlock.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public void unlock(JCRAmetysObject object) throws AmetysRepositoryException
    {
        try
        {
            Node node = getNode(object);
            
            LockManager lockManager = node.getSession().getWorkspace().getLockManager();
            
            Lock lock = lockManager.getLock(node.getPath());
            Node lockHolder = lock.getNode();
            String lockToken = lockHolder.getProperty(RepositoryConstants.METADATA_LOCKTOKEN).getString();
            
            lockManager.addLockToken(lockToken);

            lockManager.unlock(node.getPath());
            
            // Remove residual properties
            node.setProperty(RepositoryConstants.METADATA_LOCKTOKEN, (Value) null);
            node.setProperty(RepositoryConstants.METADATA_LOCKOWNER, (Value) null);
            node.getSession().save();
            
            removeLockedContent(object);
        }
        catch (RepositoryException ex)
        {
            throw new AmetysRepositoryException("Unable to unlock object " + object, ex);
        }
    }
    
    /**
     * Sets a lock token on the current session for the given {@link JCRAmetysObject}
     * @param object the object for which the lock token is set.
     * @throws AmetysRepositoryException if an error occurs
     */
    public void setLockTokenOnCurrentSession(JCRAmetysObject object) throws AmetysRepositoryException
    {
        try
        {
            Node node = getNode(object);
            if (node.isLocked())
            {
                LockManager lockManager = node.getSession().getWorkspace().getLockManager();
                
                Lock lock = lockManager.getLock(node.getPath());
                Node lockHolder = lock.getNode();
                String lockToken = lockHolder.getProperty(RepositoryConstants.METADATA_LOCKTOKEN).getString();
                
                lockManager.addLockToken(lockToken);
            }
        }
        catch (RepositoryException ex)
        {
            throw new AmetysRepositoryException("Unable to add the lock token on object " + object, ex);
        }
    }
    
    /**
     * Test if a {@link JCRAmetysObject} is locked.
     * @param object the object to test.
     * @return true if the object is locked, false otherwise.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public boolean isLocked(JCRAmetysObject object) throws AmetysRepositoryException
    {
        try
        {
            return getNode(object).isLocked();
        }
        catch (RepositoryException ex)
        {
            throw new AmetysRepositoryException("Unable to get lock status on object " + object, ex);
        }
    }
    
    /**
     * Get the lock owner of a {@link JCRAmetysObject}.
     * @param object the object of which to get the lock owner.
     * @return the lock owner or null if there is no lock owner.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public UserIdentity getLockOwner(JCRAmetysObject object) throws AmetysRepositoryException
    {
        try
        {
            String userIdentity = getNode(object).getProperty(RepositoryConstants.METADATA_LOCKOWNER).getString();
            return UserIdentity.stringToUserIdentity(userIdentity);
        }
        catch (PathNotFoundException e)
        {
            // No lock owner
            return null;
        }
        catch (RepositoryException ex)
        {
            throw new AmetysRepositoryException("Unable to get lock owner on object " + object, ex);
        }
    }
    
    /**
     * Get an object's base node.
     * @param object the object which node to get.
     * @return the object's base node.
     */
    protected Node getNode(JCRAmetysObject object)
    {
        return object instanceof DefaultAmetysObject defaultAmetysObject
                ? defaultAmetysObject.getBaseNode()
                : object.getNode();
    }
    
}
