/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.ametys.plugins.repository.jcr;

import java.text.Normalizer;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.jackrabbit.util.Text;

import org.ametys.core.util.StringUtils;
import org.ametys.plugins.repository.TraversableAmetysObject;

/**
 * Helper for implementing {@link TraversableAmetysObject} stored in JCR.
 */
public final class NameHelper
{
    private static final Pattern __NODE_NAME_PATTERN = Pattern.compile("^([0-9-_]*)[a-z].*$");
    
    /**
     * Mode of computation for the name if it already exists in JCR.
     */
    public enum NameComputationMode
    {
        /**
         * Use the legacy incremental mode: add a suffix which is automatically increment while the node name already exists.
         * Prefix: [node name in lower case without special characters]
         * Suffix: -[incremental number]
         */
        INCREMENTAL,

        /**
         * Use {@link StringUtils}.generatedKey() method to create a random suffix.
         * Prefix: [node name in lower case without special characters]
         * Suffix: -[code with 8 lower-case alphanumerics characters]
         */
        GENERATED_KEY,

        /**
         * Use for user-friendly pattern as: Non-formatted title (incremented number).
         * Prefix: [given name]
         * Suffix: [space character]([incremental number])
         */
        USER_FRIENDLY
    }
    
    // group 1 is based name, group 2 is not captured it is the suffix (optional)
    
    // my-name-is-robby, my-name-is-robby-azerty12, my-name-is-robby-qwerty45 (group 1: my-name-is-robby)
    private static final Pattern __NAME_GENERATED_KEY_PATTERN = Pattern.compile("^(.+?)(?:-[0-9a-z]{8})?$");
    // my-name-is-robby, my-name-is-robby-1, my-name-is-robby-2 (group 1: my-name-is-robby)
    private static final Pattern __NAME_INCREMENTAL_PATTERN = Pattern.compile("^(.+?)(?:-[0-9]+)?$");
    // My name is Robby, My name is Robby (1), My name is Robby (2) (group 1: My name is Robby)
    private static final Pattern __NAME_USER_FRIENDLY_PATTERN = Pattern.compile("^(.+?)(?: \\([0-9]+\\))?$");
    
    private NameHelper()
    {
        // empty
    }
    
    /**
     * Filter a name for using it into an URI.
     * @param name the name to filter.
     * @return the name filtered.
     */
    public static String filterName(String name)
    {
        // Use lower case
        // then remove accents
        // then replace contiguous spaces with one dash
        // and finally remove non-alphanumeric characters except -
        String filteredName = Normalizer.normalize(name.toLowerCase(), Normalizer.Form.NFD).replaceAll("[\\p{InCombiningDiacriticalMarks}]", "").trim();
        filteredName = filteredName.replaceAll("œ", "oe").replaceAll("æ", "ae").replaceAll(" +", "-").replaceAll("[^\\w-]", "-").replaceAll("-+", "-");

        Matcher m = __NODE_NAME_PATTERN.matcher(filteredName);
        if (!m.matches())
        {
            throw new IllegalArgumentException(filteredName + " doesn't match the expected regular expression : " + __NODE_NAME_PATTERN.pattern());
        }

        filteredName = filteredName.substring(m.end(1));

        // Remove characters '-' and '_' at the start and the end of the string
        return org.apache.commons.lang3.StringUtils.strip(filteredName, "-_");
    }
    
    /**
     * Get a unique child ametys object name under a traversable ametys object. Default parameters: Incremental mode, may not be suffixed.
     * The name is filtered and suffixed if necessary. If a suffix is already present, it is removed to be replaced by another one.
     * @param parent The parent
     * @param baseName The base name
     * @return A unique ametys object name (for the given parent)
     */
    public static String getUniqueAmetysObjectName(TraversableAmetysObject parent, String baseName)
    {
        return getUniqueAmetysObjectName(parent, baseName, NameComputationMode.INCREMENTAL, false);
    }
    
    /**
     * Get a unique child ametys object name under a traversable ametys object.
     * The name is filtered and suffixed if necessary. If a suffix is already present, it is removed to be replaced by another one.
     * @param parent The parent
     * @param baseName The base name
     * @param computationMode The computation mode of the name: incremental (old method) or generated key
     * @param mayBeSuffixed <code>true</code> to consider that the given base name may already contains a suffix, it will be removed if needed
     * @return A unique ametys object name (for the given parent)
     */
    public static String getUniqueAmetysObjectName(TraversableAmetysObject parent, String baseName, NameComputationMode computationMode, boolean mayBeSuffixed)
    {
        // Only compute the name if the base name already exists
        switch (computationMode)
        {
            case GENERATED_KEY:
                return _getUniqueAmetysObjectNameGeneratedKey(parent, baseName, mayBeSuffixed);
            case USER_FRIENDLY:
                return _getUniqueAmetysObjectNameUserFriendly(parent, Text.escapeIllegalJcrChars(baseName), mayBeSuffixed);
            case INCREMENTAL:
            default:
                return _getUniqueAmetysObjectNameIncremental(parent, baseName, mayBeSuffixed);
        }
    }
    
    private static String _getUniqueAmetysObjectNameIncremental(TraversableAmetysObject parent, String baseName, boolean mayBeSuffixed)
    {
        String filteredName = filterName(baseName);
        
        if (!parent.hasChild(filteredName))
        {
            return filteredName;
        }
        
        String realBaseName = _getRealBaseName(baseName, __NAME_INCREMENTAL_PATTERN, mayBeSuffixed);
        String uniqueName;
        long index = 2;
        do
        {
            uniqueName = realBaseName + (index++);
        }
        while (parent.hasChild(uniqueName));
        return uniqueName;
    }
    
    private static String _getUniqueAmetysObjectNameGeneratedKey(TraversableAmetysObject parent, String baseName, boolean mayBeSuffixed)
    {
        String filteredName = filterName(baseName);
        
        if (!parent.hasChild(filteredName))
        {
            return filteredName;
        }
        
        String realBaseName = _getRealBaseName(baseName, __NAME_GENERATED_KEY_PATTERN, mayBeSuffixed);
        String uniqueName;
        do
        {
            uniqueName = realBaseName + StringUtils.generateKey().toLowerCase();
        }
        while (parent.hasChild(uniqueName));
        return uniqueName;
    }
    
    private static String _getUniqueAmetysObjectNameUserFriendly(TraversableAmetysObject parent, String baseName, boolean mayBeSuffixed)
    {
        if (!parent.hasChild(baseName))
        {
            return baseName;
        }
        
        String realBaseName = baseName;
        if (mayBeSuffixed)
        {
            Matcher m = __NAME_USER_FRIENDLY_PATTERN.matcher(baseName);
            m.matches();
            realBaseName = m.group(1);
        }
        realBaseName += " (${number})";
        
        String uniqueName;
        Long index = 2L;
        do
        {
            uniqueName = realBaseName.replaceFirst("\\$\\{number\\}", index.toString());
            index++;
        }
        while (parent.hasChild(uniqueName));
        
        return uniqueName;
    }
    
    private static String _getRealBaseName(String baseName, Pattern pattern, boolean mayBeSuffixed)
    {
        if (!mayBeSuffixed)
        {
            return filterName(baseName) + "-";
        }

        // "-" is not in the first group, because the second group is optional so it may be not present
        // Add it there to avoid string computation
        Matcher m = pattern.matcher(baseName);
        m.matches();
        return filterName(m.group(1)) + "-";
    }
}
