/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.jcr;

import java.util.ArrayList;
import java.util.List;

import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.Property;
import javax.jcr.PropertyIterator;
import javax.jcr.RepositoryException;

import org.apache.cocoon.util.HashUtil;
import org.apache.commons.lang3.StringUtils;
import org.apache.jackrabbit.JcrConstants;
import org.apache.jackrabbit.core.NodeImpl;

import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.collection.AmetysObjectCollectionFactory;

/**
 * Provides helper methods on nodes.
 */
public final class NodeHelper
{
    private NodeHelper()
    {
        // Hides the default constructor.
    }

    /**
     * Rename the given {@link Node} with the given new name
     * @param node the node to rename
     * @param newName the new name of the node
     * @throws AmetysRepositoryException if an error occurs.
     */
    public static void rename(Node node, String newName) throws AmetysRepositoryException
    {
        try
        {
            Node parentNode = node.getParent();
            boolean order = parentNode.getPrimaryNodeType().hasOrderableChildNodes();
            Node nextSibling = null;
            
            if (order)
            {
                // iterate over the siblings to find the following
                NodeIterator siblings = parentNode.getNodes();
                boolean iterate = true;
                
                while (siblings.hasNext() && iterate)
                {
                    Node sibling = siblings.nextNode();
                    iterate = !sibling.getName().equals(node.getName());
                }
                
                // iterator is currently on the node
                while (siblings.hasNext() && nextSibling == null)
                {
                    Node sibling = siblings.nextNode();
                    String path = sibling.getPath();
                    if (node.getSession().itemExists(path))
                    {
                        nextSibling = sibling;
                    }
                }
            }
            
            node.getSession().move(node.getPath(), node.getParent().getPath() + "/" + newName);
            
            if (order)
            {
                // nextSibling is either null meaning that the Node must be ordered last or is equals to the following sibling
                if (nextSibling != null)
                {
                    parentNode.orderBefore(newName, nextSibling.getName());
                }
                else
                {
                    parentNode.orderBefore(newName, null);
                }
            }
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException(e);
        }
    }

    /**
     * Clone the node to the given parent node keeping the identifiers.
     * To be able to do that, parent node should be in another workspace than the node to copy.
     * The destination node should not already contains a node with the same name as the node to copy except if same name siblings is allowed.
     * This method does not save the parent node.
     * @param nodeToCopy The node to copy.
     * @param parentNode The parent node of the destination node.
     * @return the create node
     * @throws AmetysRepositoryException if an error occurs
     */
    public static Node cloneNode(Node nodeToCopy, Node parentNode) throws AmetysRepositoryException
    {
        try
        {
            return cloneNode(nodeToCopy, parentNode, nodeToCopy.getName());
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException(e);
        }
    }
    
    /**
     * Clone the node to the given parent node keeping the identifiers.
     * To be able to do that, parent node should be in another workspace than the node to copy.
     * The destination node should not already contains a node with the given node name except if same name siblings is allowed.
     * This method does not save the parent node.
     * @param nodeToCopy The node to copy.
     * @param parentNode The parent node of the destination node.
     * @param nodeName The destination node name.
     * @return the create node
     * @throws AmetysRepositoryException if an error occurs
     */
    public static Node cloneNode(Node nodeToCopy, Node parentNode, String nodeName) throws AmetysRepositoryException
    {
        try
        {
            // Create the base node with the identifier if the node is referenceable
            Node destNode = nodeToCopy.isNodeType(JcrConstants.MIX_REFERENCEABLE)
                    ? ((NodeImpl) parentNode).addNodeWithUuid(nodeName, nodeToCopy.getPrimaryNodeType().getName(), nodeToCopy.getIdentifier())
                    : parentNode.addNode(nodeName, nodeToCopy.getPrimaryNodeType().getName());
            
            // Copy properties
            PropertyIterator properties = nodeToCopy.getProperties();
            while (properties.hasNext())
            {
                Property property = properties.nextProperty();
                
                if (!property.getDefinition().isProtected())
                {
                    if (property.isMultiple())
                    {
                        destNode.setProperty(property.getName(), property.getValues(), property.getType());
                    }
                    else
                    {
                        destNode.setProperty(property.getName(), property.getValue(), property.getType());
                    }
                }
            }
            
            // Copy sub-nodes
            NodeIterator subNodes = nodeToCopy.getNodes();
            while (subNodes.hasNext())
            {
                Node subNode = subNodes.nextNode();
                if (subNode.getDefinition().isAutoCreated())
                {
                    destNode.getNode(subNode.getName()).remove();
                }
                cloneNode(subNode, destNode);
            }
            
            return destNode;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException(e);
        }
    }
    
    /**
     * Computes a hashed path in the JCR tree from the name of the child object.<br>
     * Subclasses may override this method to provide a more suitable hash function.<br>
     * This implementation relies on the buzhash algorithm.
     * This method MUST return an array of the same length for each name.
     * @param name the name of the child object
     * @return a hashed path of the name.
     */
    public static List<String> hashAsList(String name)
    {
        long hash = Math.abs(HashUtil.hash(name));
        String hashStr = Long.toString(hash, 16);
        hashStr = StringUtils.leftPad(hashStr, 4, '0');
        return List.of(hashStr.substring(0, 2), hashStr.substring(2, 4));
    }
    
    /**
     * Get the path with hashed nodes.
     * @param name the name of the final node
     * @return the path with hashed nodes
     */
    public static String getFullHashPath(String name)
    {
        List<String> pathList = new ArrayList<>(hashAsList(name));
        pathList.add(name);
        return StringUtils.join(pathList, "/");
    }

    /**
     * Get or create hash nodes for the given name, intermediate nodes are of type {@value AmetysObjectCollectionFactory#COLLECTION_ELEMENT_NODETYPE}.
     * The final node is not created.
     * The session is not saved.
     * @param parentNode the parent node of the hashed nodes
     * @param name the name of the final node.
     * @return the last level of hashed nodes
     * @throws AmetysRepositoryException if an error occurs
     */
    public static Node getOrCreateFinalHashNode(Node parentNode, String name) throws AmetysRepositoryException
    {
        return getOrCreateFinalHashNode(parentNode, name, AmetysObjectCollectionFactory.COLLECTION_ELEMENT_NODETYPE);
    }
    
    /**
     * Get or create hash nodes for the given name and intermediate primary type.
     * The final node is not created.
     * The session is not saved.
     * @param parentNode the parent node of the hashed nodes
     * @param name the name of the final node.
     * @param hashType the primary type of hashed nodes
     * @return the last level of hashed nodes
     * @throws AmetysRepositoryException if an error occurs
     */
    public static Node getOrCreateFinalHashNode(Node parentNode, String name, String hashType) throws AmetysRepositoryException
    {
        try
        {
            Node finalNode = parentNode;
            
            for (String hashPart : hashAsList(name))
            {
                finalNode = finalNode.hasNode(hashPart)
                        ? finalNode.getNode(hashPart)
                        : finalNode.addNode(hashPart, hashType);
            }
            
            return finalNode;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException(e);
        }
    }
}
