/*
 *  Copyright 2012 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.jcr;

import javax.jcr.Node;
import javax.jcr.RepositoryException;

import org.apache.jackrabbit.JcrConstants;

/**
 * Provides helper methods on node types.
 */
public final class NodeTypeHelper
{
    
    private NodeTypeHelper()
    {
        // Hides the default constructor.
    }
    
    /**
     * Get the real nodetype name of a given node (even if it's a frozen node).
     * @param node the node.
     * @return the real node's nodetype name.
     * @throws RepositoryException if an error occurs.
     */
    public static String getNodeTypeName(Node node) throws RepositoryException
    {
        String nodeTypeName = node.getPrimaryNodeType().getName();
        
        String realNodeTypeName = nodeTypeName;
        
        if (JcrConstants.NT_FROZENNODE.equals(nodeTypeName))
        {
            realNodeTypeName = node.getProperty(JcrConstants.JCR_FROZENPRIMARYTYPE).getString();
        }
        
        return realNodeTypeName;
    }
    
    /**
     * Test the real nodetype name of a given node (even if it's a frozen node).
     * @param node the node to test.
     * @param nodeTypeName the node type to test.
     * @return true if the node is of the given nodetype, false otherwise.
     * @throws RepositoryException if an error occurs.
     */
    public static boolean isNodeType(Node node, String nodeTypeName) throws RepositoryException
    {
        if (node.isNodeType(nodeTypeName))
        {
            return true;
        }
        
        String theNodeTypeName = node.getPrimaryNodeType().getName();
        
        if (JcrConstants.NT_FROZENNODE.equals(theNodeTypeName))
        {
            String realNodeTypeName = node.getProperty(JcrConstants.JCR_FROZENPRIMARYTYPE).getString();
            
            return nodeTypeName.equals(realNodeTypeName);
        }
        
        return false;
    }
    
}
