/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.ametys.plugins.repository.jcr;

import javax.jcr.Node;
import javax.jcr.RepositoryException;
import javax.jcr.nodetype.ConstraintViolationException;

import org.apache.jackrabbit.util.Text;

import org.ametys.plugins.repository.AbstractAmetysObject;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.RepositoryIntegrityViolationException;
import org.ametys.plugins.repository.metadata.CompositeMetadata;
import org.ametys.plugins.repository.metadata.ModifiableCompositeMetadata;
import org.ametys.plugins.repository.metadata.jcr.JCRCompositeMetadata;

/**
 * Default implementation of an {@link AmetysObject}, backed by a JCR node.<br>
 * This implementation heavily relies on its {@link SimpleAmetysObjectFactory} counterpart.
 * @param <F> the actual type of factory.
 */
public class SimpleAmetysObject<F extends SimpleAmetysObjectFactory> extends AbstractAmetysObject implements JCRAmetysObject 
{
    /** The corresponding {@link SimpleAmetysObjectFactory} */
    private F _factory;
    
    private Node _node;
    
    private String _name;
    private String _parentPath;

    /**
     * Creates an {@link SimpleAmetysObject}.
     * @param node the node backing this {@link AmetysObject}
     * @param parentPath the parentPath in the Ametys hierarchy
     * @param factory the DefaultTraversableAmetysObjectFactory which created the AmetysObject
     */
    public SimpleAmetysObject(Node node, String parentPath, F factory)
    {
        _node = node;
        _parentPath = parentPath;
        _factory = factory;
        
        try
        {
            _name = Text.unescapeIllegalJcrChars(_node.getName());
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get node name", e);
        }
    }
    
    /**
     * Retrieves the factory.
     * @return the factory.
     */
    protected F _getFactory()
    {
        return _factory;
    }
    
    public String getName() throws AmetysRepositoryException
    {
        return _name;
    }
    
    /**
     * Returns the root {@link CompositeMetadata} of this {@link AmetysObject}.
     * @return the root {@link CompositeMetadata} of this {@link AmetysObject}
     * @deprecated use getRepositoryData instead
     */
    @Deprecated
    public ModifiableCompositeMetadata getMetadataHolder()
    {
        return new JCRCompositeMetadata(getNode(), _getFactory()._getResolver());
    }
    
    public String getParentPath() throws AmetysRepositoryException
    {
        if (_parentPath == null)
        {
            _parentPath = getParent().getPath();
        }
        
        return _parentPath;
    }
    
    /**
     * Invalidates cached parent path.
     */
    protected void _invalidateParentPath()
    {
        _parentPath = null;
    }
    
    /**
     * Recompute the name from the Node.<br>
     * To be used when the node name changesd internally.
     */
    protected void _invalidateName()
    {
        try
        {
            _name = Text.unescapeIllegalJcrChars(getNode().getName());
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException(e);
        }
    }
    
    public String getPath() throws AmetysRepositoryException
    {
        return getParentPath() + "/" + getName();
    }
    
    public Node getNode()
    {
        return _node;
    }
    
    public String getId()
    {
        try
        {
            return _factory.getScheme() + "://" + _node.getIdentifier();
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get node UUID", e);
        }
    }
    
    @Override
    public void rename(String newName) throws AmetysRepositoryException
    {
        try
        {
            NodeHelper.rename(getNode(), newName);
            _name = Text.unescapeIllegalJcrChars(getNode().getName());
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException(e);
        }
    }
    
    public void remove() throws AmetysRepositoryException, RepositoryIntegrityViolationException
    {
        try
        {
            getNode().remove();
        }
        catch (ConstraintViolationException e)
        {
            throw new RepositoryIntegrityViolationException(e);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException(e);
        }
    }

    @SuppressWarnings("unchecked")
    public <A extends AmetysObject> A getParent() throws AmetysRepositoryException
    {
        return (A) _factory.getParent(this);
    }
    
    @Override
    public boolean needsSave() throws AmetysRepositoryException
    {
        try
        {
            return _node.getSession().hasPendingChanges();
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException(e);
        }
    }

    public void saveChanges() throws AmetysRepositoryException
    {
        try
        {
            getNode().getSession().save();
        }
        catch (javax.jcr.RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to save changes", e);
        }
    }
    
    @Override
    public void revertChanges() throws AmetysRepositoryException
    {
        try
        {
            getNode().refresh(false);
        }
        catch (javax.jcr.RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to revert changes.", e);
        }
    }
    
}
