/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.jcr;

import java.util.Collection;
import java.util.Collections;

import javax.jcr.ItemNotFoundException;
import javax.jcr.Node;
import javax.jcr.Repository;
import javax.jcr.RepositoryException;
import javax.jcr.Session;

import org.apache.avalon.framework.configuration.Configurable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.logger.AbstractLogEnabled;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectFactoryExtensionPoint;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.provider.AbstractRepository;

/**
 * Default implementation of an {@link JCRAmetysObjectFactory},
 * handling {@link SimpleAmetysObject}.<br>
 * This implementation takes its scheme and nodetype through a configuration:<br>
 * <code>
 * &lt;extension point="org.ametys.plugins.repository.AmetysObjectFactoryExtensionPoint"<br>
 * &nbsp;&nbsp;&nbsp;&nbsp;id="XXXX"
 * class="org.ametys.plugins.repository.DefaultAmetysObjectFactory"&gt;<br>
 * &nbsp;&nbsp;&lt;scheme&gt;your_scheme&lt;/scheme&gt;<br>
 * &nbsp;&nbsp;&lt;nodetype&gt;your:nodetype&lt;/nodetype&gt;<br>
 * &nbsp;&nbsp;[&lt;nodetype&gt;your:nodetype2&lt;/nodetype&gt;]<br>
 * &nbsp;&nbsp;[...]<br>
 * &lt;/extension&gt;<br>
 * This implementation manages only one nodetype.
 * </code>
 */
public class SimpleAmetysObjectFactory extends AbstractLogEnabled implements JCRAmetysObjectFactory<SimpleAmetysObject>, Configurable, Serviceable
{
    /** The application {@link AmetysObjectResolver} */
    protected AmetysObjectResolver _resolver;
    
    /** The {@link AmetysObjectFactoryExtensionPoint} */
    protected AmetysObjectFactoryExtensionPoint _ametysFactoryExtensionPoint;
    
    /** The configured scheme */
    protected String _scheme;

    /** The configured nodetype */
    protected String _nodetype;
    
    /** JCR Repository */
    protected Repository _repository;
    
    /** The Avalon {@link ServiceManager} */
    protected ServiceManager _manager;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _manager = manager;
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _repository = (Repository) manager.lookup(AbstractRepository.ROLE);
        _ametysFactoryExtensionPoint = (AmetysObjectFactoryExtensionPoint) manager.lookup(AmetysObjectFactoryExtensionPoint.ROLE);
    }

    public void configure(Configuration configuration) throws ConfigurationException
    {
        _scheme = configuration.getChild("scheme").getValue();
        
        Configuration[] nodetypesConf = configuration.getChildren("nodetype");
        
        if (nodetypesConf.length != 1)
        {
            throw new ConfigurationException("A SimpleAmetysObjectFactory must have one and only one associated nodetype. "
                                           + "The '" + configuration.getAttribute("id") + "' component has " + nodetypesConf.length);
        }
        
        _nodetype = nodetypesConf[0].getValue();
    }
    
    AmetysObjectResolver _getResolver()
    {
        return _resolver;
    }

    public String getScheme()
    {
        return _scheme;
    }
    
    public Collection<String> getNodetypes()
    {
        return Collections.singletonList(_nodetype);
    }

    @SuppressWarnings("unchecked")
    public SimpleAmetysObject getAmetysObject(Node node, String parentPath) throws AmetysRepositoryException, RepositoryException
    {
        return new SimpleAmetysObject(node, parentPath, this);
    }
    
    public SimpleAmetysObject getAmetysObjectById(String id) throws AmetysRepositoryException
    {
        try
        {
            return getAmetysObjectById(id, null);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get AmetysObject for id: " + id, e);
        }
    }
    
    @Override
    public SimpleAmetysObject getAmetysObjectById(String id, Session session) throws AmetysRepositoryException, RepositoryException
    {
        Node node = getNode(id, session);
        
        if (!node.getPath().startsWith('/' + AmetysObjectResolver.ROOT_REPO))
        {
            throw new AmetysRepositoryException("Cannot resolve a Node outside Ametys tree");
        }
        
        return getAmetysObject(node, null);
    }
    
    public boolean hasAmetysObjectForId(String id) throws AmetysRepositoryException
    {
        try
        {
            getNode(id, null);
            return true;
        }
        catch (UnknownAmetysObjectException e)
        {
            return false;
        }
    }
    
    /**
     * Returns the JCR Node associated with the given object id.<br>
     * This implementation assumes that the id is like <code>&lt;scheme&gt;://&lt;uuid&gt;</code>
     * @param id the unique id of the object
     * @param session the JCR Session to use to retrieve the Node.
     * @return the JCR Node associated with the given id
     */
    protected Node getNode(String id, Session session)
    {
        // l'id est de la forme <scheme>://uuid
        String uuid = id.substring(getScheme().length() + 3);
        
        Session jcrSession = null;
        try
        {
            jcrSession = session != null ? session : _repository.login();
            Node node = jcrSession.getNodeByIdentifier(uuid);
            return node;
        }
        catch (ItemNotFoundException e)
        {
            if (session == null && jcrSession != null)
            {
                // logout only if the session was created here
                jcrSession.logout();
            }

            throw new UnknownAmetysObjectException("There's no node for id " + id, e);
        }
        catch (RepositoryException e)
        {
            if (session == null && jcrSession != null)
            {
                // logout only if the session was created here
                jcrSession.logout();
            }

            throw new AmetysRepositoryException("Unable to get AmetysObject for id: " + id, e);
        }
    }

    /**
     * Returns the parent of the given {@link SimpleAmetysObject} .
     * @param object a {@link SimpleAmetysObject}.
     * @return the parent of the given {@link SimpleAmetysObject}. 
     * @throws AmetysRepositoryException if an error occurs.
     */
    public AmetysObject getParent(SimpleAmetysObject object) throws AmetysRepositoryException
    {
        if (getLogger().isDebugEnabled())
        {
            getLogger().debug("Entering DefaultTraversableAmetysObjectFactory.getParent with object of name: " + object.getName());
        }

        Node node = getWorkspaceNode (object);
        
        try
        {
            Node parentNode = node.getParent();
            String parentNodetype = NodeTypeHelper.getNodeTypeName(parentNode);

            if (getLogger().isDebugEnabled())
            {
                getLogger().debug("Parent nodetype is " + parentNodetype);
            }

            if (parentNodetype.equals(_nodetype))
            {
                if (getLogger().isDebugEnabled())
                {
                    getLogger().debug("The parent node has the same nodetype than this ObjectFactory: " + _nodetype);
                }

                // si le nodetype est celui de la factory, on peut éviter de passer par le resolver
                return getAmetysObject(parentNode, null);
            }
        
            return _resolver.resolve(parentNode, false);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to retrieve parent object of object " + object.getName(), e);
        }
    }
    
    /**
     * Returns the JCR node backing this {@link SimpleAmetysObject} in the JCR workspace. May be overridden to deal with e.g. versionning 
     * @param object a {@link SimpleAmetysObject}.
     * @return the JCR node backing this {@link SimpleAmetysObject}.
     */
    protected Node getWorkspaceNode(SimpleAmetysObject object)
    {
        return object.getNode();
    }
}
