/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.ametys.plugins.repository.jcr;

import java.util.ArrayList;
import java.util.List;

import javax.jcr.ItemExistsException;
import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.RepositoryException;
import javax.jcr.Value;
import javax.jcr.lock.Lock;
import javax.jcr.lock.LockManager;
import javax.jcr.nodetype.NodeType;

import org.apache.avalon.framework.logger.Logger;
import org.apache.jackrabbit.util.Text;

import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectFactory;
import org.ametys.plugins.repository.AmetysObjectFactoryExtensionPoint;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.ChainedAmetysObjectIterable;
import org.ametys.plugins.repository.CollectionIterable;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.RepositoryIntegrityViolationException;
import org.ametys.plugins.repository.TraversableAmetysObject;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.virtual.VirtualAmetysObjectFactory;

/**
 * Helper for implementing {@link TraversableAmetysObject} stored in JCR.
 */
public final class TraversableAmetysObjectHelper
{
    private TraversableAmetysObjectHelper()
    {
        // empty
    }
    
    /**
     * Returns the {@link AmetysObject} at the given subPath,
     * relative to the given {@link DefaultTraversableAmetysObject}.
     * @param <A> the actual type of {@link AmetysObject}.
     * @param object the context {@link DefaultTraversableAmetysObject}.
     * @param factory the corresponding {@link JCRAmetysObjectFactory}.
     * @param path the sub path. Cannot be <code>null</code>, empty or absolute.
     * @param resolver the {@link AmetysObjectResolver}.
     * @param logger a {@link Logger} for traces.
     * @return the {@link AmetysObject} at the given subPath,
     *         relative to the given {@link DefaultTraversableAmetysObject}.
     * @throws AmetysRepositoryException if an error occurs.
     * @throws UnknownAmetysObjectException if no such object exists.
     */
    @SuppressWarnings("unchecked")
    public static <A extends AmetysObject> A getChild(JCRTraversableAmetysObject object, JCRAmetysObjectFactory factory, String path, AmetysObjectResolver resolver, Logger logger) throws AmetysRepositoryException, UnknownAmetysObjectException
    {
        if (logger.isDebugEnabled())
        {
            logger.debug("Entering DefaultTraversableAmetysObjectFactory.getChild with path=" + path + ", object=" + object);
        }

        if (path == null || "".equals(path) || path.charAt(0) == '/')
        {
            throw new AmetysRepositoryException("Child path cannot be null, empty or absolute");
        }
        
        Node node = object.getNode();
        
        try
        {
            // instead of going through resolver for each path segment, we first test the nodetype of the sub Node. 
            // If it's same than this factory, there's no need to resolve anything.   
            String[] pathElements = path.split("/");
            
            Node contextNode = node;
            String contextPath = object.getPath();
            String contextParentPath = null;
            
            int i = 0;
            while (i < pathElements.length)
            {
                if (logger.isDebugEnabled())
                {
                    logger.debug("contextPath=" + contextPath + ", pathElement=" + pathElements[i]);
                }

                if (".".equals(pathElements[i]) || "..".equals(pathElements[i]))
                {
                    throw new AmetysRepositoryException("Path cannot contain segment with . or ..");
                }
                
                // handle special characters for local part
                String jcrName = _escape(pathElements[i]);

                if (contextNode.hasNode(jcrName))
                {
                    // the path element corresponds to a JCR Node
                    Node subNode = contextNode.getNode(jcrName);
                    String type = NodeTypeHelper.getNodeTypeName(subNode);
                    
                    if (factory.getNodetypes().contains(type))
                    {
                        if (logger.isDebugEnabled())
                        {
                            logger.debug("The nodetype is the same as the current factory, no need to go through resolver: " + type);
                        }

                        contextParentPath = contextPath;
                        contextPath += "/" + subNode.getName();
                        contextNode = subNode;
                        i++;
                    }
                    else
                    {
                        return (A) resolver.resolve(contextPath, subNode, _computeSubPath(pathElements, i + 1), false);
                    }
                }
                else if (contextNode.hasProperty(AmetysObjectResolver.VIRTUAL_PROPERTY))
                {
                    // the sub node does not exist, but there may be virtual children
                    JCRAmetysObject contextObject = resolver.resolve(contextParentPath, contextNode, null, true);
                    
                    if (contextObject == null)
                    {
                        throw new UnknownAmetysObjectException("There's no object at path " + jcrName + " from path " + contextPath);
                    }
                    
                    return (A) resolver.resolveVirtualChild(contextObject, _computeSubPath(pathElements, i));
                }
                else
                {
                    // there's no children
                    throw new UnknownAmetysObjectException("There's no object at path " + jcrName + " from path " + contextPath);
                }
            }
            
            // on est arrivés au bout du subPath, ce qui signifie que tous les descendants sont connus par cette factory 
            return (A) factory.getAmetysObject(contextNode, contextParentPath);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to resolve AmetysObject at path " + path + " relative to AmetysObject at path " + object.getPath(), e);
        }
    }
    
    private static String _computeSubPath(String[] pathElements, int beginIndex)
    {
        String subPath = null;
        
        for (int j = beginIndex; j < pathElements.length; j++)
        {
            subPath = subPath == null ? pathElements[j] : subPath + "/" + pathElements[j];
        }
        
        return subPath;
    }

    /**
     * Returns all children of the given {@link DefaultTraversableAmetysObject}.
     * @param <A> the actual type of {@link AmetysObject}s
     * @param object a {@link DefaultTraversableAmetysObject}.
     * @param factory the corresponding {@link JCRAmetysObjectFactory}.
     * @param resolver the {@link AmetysObjectResolver}.
     * @param logger a {@link Logger} for traces.
     * @return a List containing all children object in the Ametys hierarchy.
     * @throws AmetysRepositoryException if an error occurs.
     */
    @SuppressWarnings("unchecked")
    public static <A extends AmetysObject> AmetysObjectIterable<A> getChildren(JCRTraversableAmetysObject object, JCRAmetysObjectFactory factory, AmetysObjectResolver resolver, Logger logger) throws AmetysRepositoryException
    {
        if (logger.isDebugEnabled())
        {
            logger.debug("Entering DefaultTraversableAmetysObjectFactory.getChildren with object=" + object);
        }

        try
        {
            Node node = object.getNode();
            NodeIterator it = node.getNodes();
            List<A> children = new ArrayList<>((int) it.getSize());
            
            while (it.hasNext())
            {
                Node child = it.nextNode();
                String type = NodeTypeHelper.getNodeTypeName(child);
                
                if (factory.getNodetypes().contains(type))
                {
                    // if the node type correspond to the factory, do not go trough resolver
                    children.add((A) factory.getAmetysObject(child, object.getPath()));
                }
                else
                {
                    A obj = resolver.<A>resolve(object.getPath(), child, null, true);
                    
                    if (obj != null)
                    {
                        children.add(obj);
                    }
                }
            }
            
            AmetysObjectIterable<A> childrenIt = new CollectionIterable<>(children);
            
            // on regarde les virtuels
            if (node.hasProperty(AmetysObjectResolver.VIRTUAL_PROPERTY))
            {
                AmetysObjectIterable<A> virtualIt = resolver.resolveVirtualChildren(object);
                
                List<AmetysObjectIterable<A>> chainedList = new ArrayList<>();
                chainedList.add(childrenIt);
                chainedList.add(virtualIt);
                
                return new ChainedAmetysObjectIterable<>(chainedList);
            }
            else
            {
                return childrenIt;
            }
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to retrieve children", e);
        }
    }
    
    /**
     * Tests if a given object has a child with a given name.
     * @param object the context object.
     * @param name the name to test.
     * @param ametysFactoryExtensionPoint the {@link AmetysObjectFactoryExtensionPoint}.
     * @param logger a {@link Logger} for traces.
     * @return <code>true</code> is the given object has a child with the given name,
     *         <code>false</code> otherwise.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public static boolean hasChild(JCRTraversableAmetysObject object, String name, AmetysObjectFactoryExtensionPoint ametysFactoryExtensionPoint, Logger logger) throws AmetysRepositoryException
    {
        if (logger.isDebugEnabled())
        {
            logger.debug("Entering DefaultTraversableAmetysObjectFactory.hasChild with object=" + object + ", name=" + name);
        }

        if (name == null || "".equals(name) || name.charAt(0) == '/')
        {
            throw new AmetysRepositoryException("Child name cannot be null, empty or absolute");
        }
        
        if (".".equals(name) || "..".equals(name))
        {
            throw new AmetysRepositoryException("Child name cannot be . or ..");
        }
        
        Node node = object.getNode();
        
        try
        {
            String jcrName = _escape(name);
            if (node.hasNode(jcrName))
            {
                if (logger.isDebugEnabled())
                {
                    logger.debug("Child node exists: " + jcrName);
                }

                // if a physical node exists, its an Ametys child if and only if its nodetype is known 
                Node childNode = node.getNode(jcrName);
                String nodetype = NodeTypeHelper.getNodeTypeName(childNode);
                return ametysFactoryExtensionPoint.getFactoryForNodetype(nodetype) != null;
            }
            else if (node.hasProperty(AmetysObjectResolver.VIRTUAL_PROPERTY))
            {
                if (logger.isDebugEnabled())
                {
                    logger.debug("Looking for virtuals...");
                }

                // looking at virtuals...
                Value[] values = node.getProperty(AmetysObjectResolver.VIRTUAL_PROPERTY).getValues();
                for (Value value : values)
                {
                    String virtual = value.getString();

                    VirtualAmetysObjectFactory virtualFactory = _getVirtualFactory(virtual, ametysFactoryExtensionPoint, logger);
                    
                    if (virtualFactory.hasChild(object, name))
                    {
                        return true;
                    }
                }
            }
            
            return false;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to test if the underlying Node for object " + object.getId() + " has a child named " + name, e);
        }
    }
    
    private static VirtualAmetysObjectFactory _getVirtualFactory(String id, AmetysObjectFactoryExtensionPoint ametysFactoryExtensionPoint, Logger logger)
    {
        if (logger.isDebugEnabled())
        {
            logger.debug("Found virtual id: " + id);
        }
        
        AmetysObjectFactory factory = ametysFactoryExtensionPoint.getExtension(id);
        
        if (factory == null)
        {
            throw new AmetysRepositoryException("There's no virtual factory for id " + id);
        }
        
        if (!(factory instanceof VirtualAmetysObjectFactory))
        {
            throw new AmetysRepositoryException("A factory handling virtual objects must implement VirtualAmetysObjectFactory: " + id);
        }
        
        VirtualAmetysObjectFactory virtualFactory = (VirtualAmetysObjectFactory) factory;
        
        return virtualFactory;
    }

    /**
     * Creates a child to the given object.
     * @param <A> the actual type of {@link AmetysObject}.
     * @param object the parent {@link AmetysObject}.
     * @param factory the corresponding {@link JCRAmetysObjectFactory}.
     * @param name the new object's name.
     * @param type the new object's type.
     * @param ametysFactoryExtensionPoint the {@link AmetysObjectFactoryExtensionPoint}.
     * @param resolver the {@link AmetysObjectResolver}.
     * @param logger a {@link Logger} for traces.
     * @return the newly created {@link AmetysObject}.
     * @throws AmetysRepositoryException if an error occurs.
     */
    @SuppressWarnings("unchecked")
    public static <A extends AmetysObject> A createChild(JCRTraversableAmetysObject object, JCRAmetysObjectFactory factory, String name, String type, AmetysObjectFactoryExtensionPoint ametysFactoryExtensionPoint, AmetysObjectResolver resolver, Logger logger) throws AmetysRepositoryException
    {
        if (logger.isDebugEnabled())
        {
            logger.debug("Entering DefaultTraversableAmetysObjectFactory.createChild with object=" + object + ", name=" + name + ", type=" + type);
        }

        // the code of this method is mainly duplicated from the AmetysObjectResolver.createAndResolve method, 
        // with the optimization that there's no need to go through resolver when the nodetype id the same than this factory

        if (ametysFactoryExtensionPoint.getFactoryForNodetype(type) == null)
        {
            throw new AmetysRepositoryException("Cannot create a node '" + name +  "' under '" + object.getPath() + " (" + object.getId() + ")': There's no factory for nodetype: " + type);
        }

        Node contextNode = object.getNode();
        
        try
        {
            _checkLock(contextNode);
            
            String legalName = _escape(name);
            Node node = contextNode.addNode(legalName, type);
            NodeType[] mixinNodeTypes = node.getMixinNodeTypes();
            boolean foundMixin = false;
            
            int i = 0;
            while (!foundMixin && i < mixinNodeTypes.length)
            {
                if (AmetysObjectResolver.OBJECT_TYPE.equals(mixinNodeTypes[i].getName()))
                {
                    foundMixin = true;
                }
                
                i++;
            }
            
            if (!foundMixin)
            {
                node.addMixin(AmetysObjectResolver.OBJECT_TYPE);
            }
            
            if (factory.getNodetypes().contains(type))
            {
                // pas besoin de repasser par le resolver si le type est le même que cette factory
                return (A) factory.getAmetysObject(node, object.getPath());
            }
            
            return (A) resolver.resolve(object.getPath(), node, null, false);
        }
        catch (ItemExistsException e)
        {
            throw new RepositoryIntegrityViolationException("The object " + name + " already exist at path " + object.getParentPath(), e);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to add child node for the underlying node for object " + object.getId(), e);
        }
    }
    
    private static String _escape(String qName)
    {
        int index = qName.indexOf(':');
        
        if (index == -1)
        {
            return Text.escapeIllegalJcrChars(qName);
        }
        else
        {
            return qName.substring(0, index) + ':' + Text.escapeIllegalJcrChars(qName.substring(index + 1, qName.length()));
        }
    }
    
    private static void _checkLock(Node node) throws RepositoryException
    {
        if (node.isLocked())
        {
            LockManager lockManager = node.getSession().getWorkspace().getLockManager();
            Lock lock = lockManager.getLock(node.getPath());
            Node lockHolder = lock.getNode();
            
            lockManager.addLockToken(lockHolder.getProperty(RepositoryConstants.METADATA_LOCKTOKEN).getString());
        }
    }
}
