/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.lock;

import java.time.LocalDateTime;
import java.time.ZonedDateTime;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;

import org.apache.avalon.framework.activity.Disposable;
import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.avalon.framework.thread.ThreadSafe;
import org.apache.jackrabbit.JcrConstants;
import org.quartz.JobKey;
import org.quartz.SchedulerException;

import org.ametys.core.schedule.Runnable;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.population.UserPopulationDAO;
import org.ametys.plugins.core.schedule.Scheduler;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.plugin.PluginsManager;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * The helper for scheduling the unlocking of the contents.<br>
 * At initialization all contents are scanned and locked ones will be automatically unlocked.
 */
public class UnlockHelper extends AbstractLogEnabled implements Initializable, ThreadSafe, Component, Serviceable, Disposable
{
    /** Avalon Role */
    public static final String ROLE = UnlockHelper.class.getName();

    // Deploy parameters
    private static final String __ACTIVATE_PARAMETER = "content.unlocktimer.activate";

    private static final String __PERIOD_PARAMETER = "content.unlocktimer.period";

    /** The scheduler */
    protected Scheduler _scheduler;
    
    // To know if the component is activated
    private boolean _activated;

    // The period before unlocking in hours
    private long _period;

    /** The map of locked contents id (String) and their locked time (value) */
    private Map<String, LocalDateTime> _lockedObjects;
    
    private AmetysObjectResolver _resolver;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _scheduler = (Scheduler) manager.lookup(Scheduler.ROLE);
    }

    public void initialize() throws Exception
    {
        Config config = Config.getInstance();
        if (config == null || PluginsManager.getInstance().isSafeMode())
        {
            // Désactiver le composant pour affecter les paramètres de
            // déploiement
            // Le composant doit pouvoir s'initialiser en safe mode, mais ne pas
            // lancer le timer.
            _activated = false;
        }
        else
        {
            // Récupérer les paramètres de déploiement
            _activated = config.getValue(__ACTIVATE_PARAMETER);

            if (_activated)
            {
                _period = config.getValue(__PERIOD_PARAMETER);

                // Vérifier la validité de la période
                if (_period <= 0)
                {
                    throw new RuntimeException("Invalid period (in hours) : '" + _period + "'");
                }
            }
            else
            {
                _period = 0;
            }
        }
        
        _initializeLockedObjects();
    }

    /**
     * Search and stores references to all locked {@link AmetysObject}s in the repository.<br>
     * Should not be called by clients.
     */
    private void _initializeLockedObjects()
    {
        _lockedObjects = new HashMap<>();
        
        // search for all locked AmetysObjects
        try (AmetysObjectIterable<LockableAmetysObject> it = _resolver.query("//element(*, ametys:object)[@" + JcrConstants.JCR_LOCKOWNER + "]");)
        {
            for (LockableAmetysObject object : it)
            {
                if (object.isLocked() && _activated)
                {
                    scheduleUnlocking(object);
                    getLogger().info("Scheduled unlocking in " + _period + " hour(s) for object '{}'", object.getName());
                }
                else if (object.isLocked() && !_activated)
                {
                    // If the unlocking was activated with remaining locked contents, and now it is disabled
                    // there are stil remaining quartz job for unlocking those contents
                    synchronized (_lockedObjects)
                    {
                        _lockedObjects.put(object.getId(), LocalDateTime.now());
                    }
                    _cancelUnlocking(object);
                }
            }
        }
    }

    /**
     * Check if the unlocking is enabled.
     * @return true if the unlocking is activated, false otherwise.
     */
    public boolean isUnlockingActivated()
    {
        return _activated;
    }

    /**
     * Get the period before unlocking a content.
     * @return The period in hours.
     */
    public long getTimeBeforeUnlock()
    {
        return _period;
    }

    /**
     * Schedule a task to unlock an object.
     * @param object The {@link AmetysObject} to be unlocked.
     */
    public void scheduleUnlocking(AmetysObject object)
    {
        synchronized (_lockedObjects)
        {
            _lockedObjects.put(object.getId(), LocalDateTime.now());
        }
            
        if (_activated)
        {
            try
            {
                ZonedDateTime unlockDate = ZonedDateTime.now().plusHours(_period);
                
                UserIdentity lockUser = Optional.ofNullable(object)
                        .filter(LockableAmetysObject.class::isInstance)
                        .map(LockableAmetysObject.class::cast)
                        .map(LockableAmetysObject::getLockOwner)
                        .orElse(UserPopulationDAO.SYSTEM_USER_IDENTITY);
                Runnable unlockRunnable = new UnlockRunnable(object.getId(), object.getName(), unlockDate, lockUser);
                JobKey jobKey = new JobKey(unlockRunnable.getId(), Scheduler.JOB_GROUP);
                if (_scheduler.getScheduler().checkExists(jobKey))
                {
                    _scheduler.getScheduler().deleteJob(jobKey);
                }
                _scheduler.scheduleJob(unlockRunnable);
                getLogger().info("Scheduled unlocking in " + _period + " hour(s) for object '{}'", object.getName());
            }
            catch (SchedulerException e)
            {
                getLogger().error("An error occured when trying to schedule the unlocking of the object " + object.getId(), e);
            }
        }
    }

    /**
     * Cancel the unlocking of an {@link AmetysObject}.
     * @param object The object to cancel the unlock scheduling.
     * @return true if the unlocking was cancelled, false otherwise.
     */
    public boolean cancelUnlocking(AmetysObject object)
    {
        synchronized (_lockedObjects)
        {
            _lockedObjects.remove(object.getId());
        }
            
        if (_activated)
        {
            return _cancelUnlocking(object);
        }
        else
        {
            return false;
        }
    }
    
    private boolean _cancelUnlocking(AmetysObject object)
    {
        try
        {
            String jobName = UnlockRunnable.class.getName() + "." + object.getId();
            JobKey jobKey = new JobKey(jobName, Scheduler.JOB_GROUP);
            if (_scheduler.getScheduler().checkExists(jobKey))
            {
                _scheduler.getScheduler().deleteJob(jobKey);
                return true;
            }
        }
        catch (SchedulerException e)
        {
            getLogger().error("An error occured when trying to schedule the unlocking of the object " + object.getId(), e);
        }
        return false;
    }

    /**
     * Get all current locked {@link LockableAmetysObject}.
     * @param <A> the generic type extending {@link LockableAmetysObject}.
     * @return The contents as a Map of Content to LocalDateTime (the date time when they were locked).
     */
    @SuppressWarnings("unchecked")
    public <A extends LockableAmetysObject> Map<A, LocalDateTime> getLockedObjects()
    {
        Map<A, LocalDateTime> result = new HashMap<>();
        
        synchronized (_lockedObjects)
        {
            for (Entry<String, LocalDateTime> entry : _lockedObjects.entrySet())
            {
                try
                {
                    LockableAmetysObject object = _resolver.resolveById(entry.getKey());
                    
                    if (object.isLocked())
                    {
                        // Add the current object and its locking time.
                        // We test if it's still locked because it could have been manually unlocked in the meantime.
                        result.put((A) object, entry.getValue());
                    }
                }
                catch (UnknownAmetysObjectException e)
                {
                    // The object doesn't exist anymore: remove it from the map.
                    if (getLogger().isWarnEnabled())
                    {
                        getLogger().warn("The object of ID " + entry.getKey() + " is referenced as locked but it doesn't exist anymore.", e);
                    }
                    _lockedObjects.remove(entry.getKey());
                }
                
            }
        }
        
        return result;
    }
    
    @Override
    public void dispose()
    {
        _activated = false;
        
        setLogger(null);
        _resolver = null;
        _lockedObjects = null;
    }
    
}
