/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.lock;

import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.ametys.core.schedule.Runnable;
import org.ametys.core.user.UserIdentity;
import org.ametys.runtime.i18n.I18nizableText;

/**
 * A {@link Runnable} which schedules a {@link UnlockSchedulable} for unlocking an object.
 */
public class UnlockRunnable implements Runnable
{
    /** The id of the object to unlock */
    protected String _objectId;
    /** The name of the object to unlock */
    protected String _objectName;
    /** The date when to unlock the content */
    protected ZonedDateTime _date;
    /** The {@link UserIdentity} to launch the runnable task */
    protected UserIdentity _userIdentity;

    /**
     * Constructor
     * @param objectId The id of the object to unlock
     * @param objectName The name of the object to unlock
     * @param date The date when to unlock the content
     * @param userIdentity The {@link UserIdentity} to launch the runnable task
     */
    public UnlockRunnable(String objectId, String objectName, ZonedDateTime date, UserIdentity userIdentity)
    {
        _objectId = objectId;
        _objectName = objectName;
        _date = date.withZoneSameInstant(ZoneId.systemDefault());
        _userIdentity = userIdentity;
    }
    
    @Override
    public String getId()
    {
        return this.getClass().getName() + "." + _objectId;
    }

    @Override
    public I18nizableText getLabel()
    {
        return new I18nizableText("plugin.repository", "PLUGINS_REPOSITORY_UNLOCK_RUNNABLE_LABEL", Collections.singletonList(_objectName));
    }

    @Override
    public I18nizableText getDescription()
    {
        List<String> parameters = new ArrayList<>();
        parameters.add(_objectId);
        parameters.add(_objectName);
        return new I18nizableText("plugin.repository", "PLUGINS_REPOSITORY_UNLOCK_RUNNABLE_DESCRIPTION", parameters);
    }
    
    @Override
    public FireProcess getFireProcess()
    {
        return FireProcess.CRON;
    }

    @Override
    public String getCronExpression()
    {
        return _date.getSecond() + " " + _date.getMinute() + " " + _date.getHour() + " " + _date.getDayOfMonth() + " " + _date.getMonthValue() + " ? " + _date.getYear();
    }

    @Override
    public String getSchedulableId()
    {
        return "org.ametys.plugins.repository.lock.UnlockSchedulable";
    }

    @Override
    public boolean isRemovable()
    {
        return false;
    }

    @Override
    public boolean isModifiable()
    {
        return false;
    }

    @Override
    public boolean isDeactivatable()
    {
        return false;
    }

    @Override
    public MisfirePolicy getMisfirePolicy()
    {
        return MisfirePolicy.FIRE_ONCE;
    }

    @Override
    public boolean isVolatile()
    {
        return false;
    }

    @Override
    public Map<String, Object> getParameterValues()
    {
        Map<String, Object> values = new HashMap<>();
        values.put(UnlockSchedulable.OBJECT_ID_KEY, _objectId);
        return values;
    }

    @Override
    public UserIdentity getUserIdentity()
    {
        return _userIdentity;
    }
}
