/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.lock;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.quartz.JobDataMap;
import org.quartz.JobExecutionContext;

import org.ametys.core.schedule.Schedulable;
import org.ametys.core.schedule.progression.ContainerProgressionTracker;
import org.ametys.plugins.core.impl.schedule.AbstractStaticSchedulable;
import org.ametys.plugins.core.schedule.Scheduler;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.UnknownAmetysObjectException;

/**
 * A {@link Schedulable} job which unlock an object.
 */
public class UnlockSchedulable extends AbstractStaticSchedulable
{
    /** The key for the id of the object to unlock */
    public static final String OBJECT_ID_KEY = "objectId";
    
    private static final String __JOBDATAMAP_OBJECT_ID_KEY = Scheduler.PARAM_VALUES_PREFIX + OBJECT_ID_KEY;
    
    /** The Ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
    }
    
    @Override
    public void execute(JobExecutionContext context, ContainerProgressionTracker progressionTracker) throws Exception
    {
        JobDataMap jobDataMap = context.getJobDetail().getJobDataMap();
        String objectId = (String) jobDataMap.get(__JOBDATAMAP_OBJECT_ID_KEY);
        
        getLogger().info("Trying to unlock the object '{}'", objectId);
        try
        {
            LockableAmetysObject object = _resolver.resolveById(objectId);
            
            if (object.isLocked())
            {
                object.unlock();
                getLogger().info("Successfully unlocked the object '{}'", objectId);
            }
            else
            {
                getLogger().warn("The object of ID '{}' was scheduled to be unlocked but it already is.", objectId);
            }
        }
        catch (UnknownAmetysObjectException e)
        {
            // The object doesn't exist anymore
            if (getLogger().isWarnEnabled())
            {
                getLogger().warn("The object of ID '" + objectId + "' was scheduled to be unlocked but it doesn't exist anymore.", e);
            }
        }
    }
}
