/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.maintenance;

import java.util.List;

import javax.jcr.Node;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.SimpleCredentials;

import org.apache.jackrabbit.core.RepositoryContext;
import org.apache.jackrabbit.core.id.NodeId;
import org.apache.jackrabbit.core.id.PropertyId;
import org.apache.jackrabbit.core.persistence.IterablePersistenceManager;
import org.apache.jackrabbit.core.persistence.pool.BundleDbPersistenceManager;
import org.apache.jackrabbit.core.state.ItemStateException;
import org.apache.jackrabbit.core.state.NodeReferences;
import org.slf4j.LoggerFactory;

import org.ametys.workspaces.repository.maintenance.AbstractMaintenanceTask;

/**
 * Repository maintenance task that crawls every node and check that the parent references still exists 
 */
public class CleanReferenceTask extends AbstractMaintenanceTask
{
    private static final int __BUNDLE_SIZE = 100_000;
    long _handled;
    int _inconsistent;
    int _partiallyInconsistent;
    int _cleaned;
    private Session _session;
    private NodeId _last;

    @Override
    public boolean requiresOffline()
    {
        return false;
    }
    
    @Override
    protected void initialize() throws RepositoryException
    {
        _session = getOrCreateRepository().login(new SimpleCredentials("__MAINTENANCE_TASK__", "".toCharArray()));
    }
    
    @Override
    protected void apply() throws RepositoryException
    {
        RepositoryContext repositoryContext = getOrCreateRepositoryContext();
        List<IterablePersistenceManager> pmList = getAllPersistenceManager(repositoryContext);
        
        for (IterablePersistenceManager pm : pmList)
        {
            if (pm instanceof BundleDbPersistenceManager bundleDbPersistenceManager)
            {
                _logger.info("Cleaning nodes for persistence manager {}", pm);
                _cleanReferences(bundleDbPersistenceManager);
            }
            else
            {
                _logger.info("The persistence manager {} doesn't support the operation. It will be skipped.", pm.getClass().getName());
            }
        }
    }    
    
    private void _cleanReferences(BundleDbPersistenceManager bundleDbPersistenceManager) throws RepositoryException
    {
        try
        {
            // initialized the report
            _handled = 0;
            _inconsistent = 0;
            _partiallyInconsistent = 0;
            _cleaned = 0;
            
            _last = null;
            // if this fail then we can't do anything so we just throw the exception
            List<NodeId> allNodeIds = bundleDbPersistenceManager.getAllNodeIds(_last, __BUNDLE_SIZE);
            long total = 0L;
            while (!allNodeIds.isEmpty())
            {
                total += allNodeIds.size();
                
                for (NodeId id : allNodeIds)
                {
                    _last = id;
                    try
                    {
                        if (bundleDbPersistenceManager.existsReferencesTo(id))
                        {
                            _cleanReferences(id, bundleDbPersistenceManager);
                        }
                    }
                    catch (ItemStateException e)
                    {
                        _logger.warn("Failed to retrieve references to node " + id.toString() + ". The node will be skipped.", e);
                    }
                }
                
                _logger.info(String.format("%,d nodes processed...", total));
                
                // if this fail then we can't do anything so we just throw the exception
                allNodeIds = bundleDbPersistenceManager.getAllNodeIds(_last, __BUNDLE_SIZE);
            }
            
            _logger.info(String.format("The operation is over. Out of %,d nodes found, %,d had references.%n"
                    + "%,d inconsistent nodes and %,d partially inconsistent nodes were found.%n"
                    + "%,d have been removed.", total, _handled, _inconsistent, _partiallyInconsistent, _cleaned));
        }
        catch (ItemStateException e)
        {
            _logger.error("Failed to retrieve the node ids. Can't clean the reference", e);
            throw new RepositoryException("Failed to retrieve the node ids. Can't clean the reference", e);
        }
    }
    
    private void _cleanReferences(NodeId nodeId, BundleDbPersistenceManager persistenceManager)
    {
        try
        {
            NodeReferences nodeRefs = persistenceManager.loadReferencesTo(nodeId);
            List<PropertyId> refs = nodeRefs.getReferences();
            final int totalRefs = refs.size();
            int localInconsistent = 0;
            
            for (PropertyId ref: refs)
            {
                String uuid = ref.getParentId().toString();
                try
                {
                    _session.getNodeByIdentifier(uuid);
                }
                catch (RepositoryException e)
                {
                    _logger.debug("Inconsistent reference: " + nodeId.toString() + " <- " + uuid);
                    localInconsistent++;
                }
            }
            
            if (localInconsistent > 0)
            {
                if (localInconsistent == totalRefs)
                {
                    _inconsistent++;
                    try
                    {
                        Node node = _session.getNodeByIdentifier(nodeId.toString());
                        _logger.debug("Inconsistent references to " + node.getPath() + " (" + nodeId.toString() + ") will be deleted");
                        persistenceManager.destroy(nodeRefs);
                        _cleaned++;
                    }
                    catch (RepositoryException e)
                    {
                        _logger.warn("Failed to retrieve node " + nodeId.toString() + ". "
                                + "This node had inconsistencies. The node will be skipped and the process will continue.", e);
                    }
                    catch (ItemStateException e)
                    {
                        _logger.warn("Failed to destroy node references to inconsistent node " + nodeId.toString() + ". "
                                + "The node will be skipped and the process will continue.", e);
                    }
                }
                else
                {
                    _partiallyInconsistent++;
                    _logger.info("Node " + nodeId.toString() + " is partially inconsistent. "
                            + "Nothing has been implemented to handle those case. It will be ignored by the cleaning operation.");
                }
            }
            _handled++;
        }
        catch (ItemStateException e)
        {
            _logger.warn("Failed to retrieve references to node " + nodeId.toString() + ". "
                    + "The node will be skipped and the process will continue.", e);
        }
    }

    @Override
    protected void close()
    {
        _session.logout();
        
        super.close();
    }
        
    @Override
    protected void setLogger()
    {
        setLogger(LoggerFactory.getLogger(CleanReferenceTask.class));
    }
}
