/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.maintenance;

import java.io.File;
import java.sql.SQLException;

import javax.jcr.RepositoryException;
import javax.sql.DataSource;

import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.jackrabbit.core.RepositoryContext;
import org.apache.jackrabbit.core.persistence.PersistenceManager;
import org.apache.jackrabbit.core.persistence.pool.BundleDbPersistenceManager;
import org.apache.jackrabbit.core.persistence.pool.DerbyPersistenceManager;
import org.apache.jackrabbit.core.util.db.ConnectionFactory;
import org.slf4j.LoggerFactory;

import org.ametys.core.schedule.progression.ProgressionTrackerFactory;
import org.ametys.plugins.repository.provider.AmetysPersistenceManager;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.workspaces.repository.maintenance.AbstractMaintenanceTask;
import org.ametys.workspaces.repository.maintenance.DataStoreGarbageCollectorTask;

/**
 * Maintenance Task to reclaim unused space from the Derby tables
 */
public class ReclaimUnusedSpaceTask extends AbstractMaintenanceTask
{
    @Override
    protected void initialize() throws RepositoryException
    {
        _progress = ProgressionTrackerFactory.createContainerProgressionTracker(new I18nizableText("plugin.repository", "PLUGINS_REPOSITORY_BUTTON_MAINTENANCE_RECLAIMUNUSEDSPACE"), _logger);
    }
    
    @Override
    protected void apply() throws RepositoryException
    {
        _logger.info("Reclaiming unused space, this may take several minutes depending on the size of the data store.\nPlease be patient.");
        
        RepositoryContext repositoryContext = getOrCreateRepositoryContext();
        String[] workspaceNames = repositoryContext.getWorkspaceManager().getWorkspaceNames();
        
        _progress.setSize(workspaceNames.length + 1);
        
        for (String workspaceName : workspaceNames)
        {
            try
            {
                _logger.info("Handling workspace " + workspaceName + "...");
                PersistenceManager persistenceManager = repositoryContext.getWorkspaceInfo(workspaceName).getPersistenceManager();
                _reclaimUnusedSpace(persistenceManager);
            }
            catch (Exception e)
            {
                _logger.warn("An error occurred while handling workspace " + workspaceName + ". The workspace will be skipped.", e);
            }
            _progress.increment();
        }
        
        try
        {
            _logger.info("Handling version history...");
            PersistenceManager versionPersistenceManager = repositoryContext.getInternalVersionManager().getPersistenceManager();
            _reclaimUnusedSpace(versionPersistenceManager);
        }
        catch (Exception e)
        {
            _logger.warn("An error occurred while handling version data source. The datasource will be skipped.", e);
        }
        _progress.increment();
    }

    private void _reclaimUnusedSpace(PersistenceManager persistenceManager) throws RepositoryException
    {
        if (persistenceManager instanceof AmetysPersistenceManager ametysPersistenceManager)
        {
            // the AmetysPersistenceManager is a wrapper, we want the actual PM
            _reclaimUnusedSpace(ametysPersistenceManager.getWrappedPM());
        }
        else if (persistenceManager instanceof DerbyPersistenceManager derbyPersistenceManager)
        {
            _reclaimUnusedSpace(derbyPersistenceManager);
        }
    }
    
    private void _reclaimUnusedSpace(DerbyPersistenceManager derbyPersistenceManager) throws RepositoryException
    {
        String url = derbyPersistenceManager.getUrl();
        String path = StringUtils.substringAfter(url, "jdbc:derby:");
        path = StringUtils.substringBefore(path, ";");
        File dbDirectory = new File(path);
        
        DataSource dataSource = _getDataSource(derbyPersistenceManager);
        String schemaObjectPrefix = derbyPersistenceManager.getSchemaObjectPrefix();
        long startSize = FileUtils.sizeOfDirectory(dbDirectory);
        
        DataStoreGarbageCollectorTask.derbyCompressTable(dataSource, schemaObjectPrefix + "REFS", _logger);
        DataStoreGarbageCollectorTask.derbyCompressTable(dataSource, schemaObjectPrefix + "BUNDLE", _logger);
        
        long endSize = FileUtils.sizeOfDirectory(dbDirectory);
        
        if (startSize > endSize)
        {
            _logger.info(String.format("%s freed. (%s -> %s)", FileUtils.byteCountToDisplaySize(startSize - endSize), FileUtils.byteCountToDisplaySize(startSize), FileUtils.byteCountToDisplaySize(endSize)));
        }
        else
        {
            _logger.info("No space freed");
        }
    }

    /** 
     * Try to retrieve the data source from a persistence manager.
     * The data source will be retrieve either by its name or using its URL
     * @param pm the persistence manager
     * @return the data source
     */
    private DataSource _getDataSource(BundleDbPersistenceManager pm) throws RepositoryException
    {
        try
        {
            ConnectionFactory cf = getRepositoryConfig().getConnectionFactory();
            if (pm.getDataSourceName() == null || "".equals(pm.getDataSourceName()))
            {
                return cf.getDataSource(pm.getDriver(), pm.getUrl(), pm.getUser(), pm.getPassword());
            }
            else
            {
                return cf.getDataSource(pm.getDataSourceName());
            }
        }
        catch (SQLException e)
        {
            _logger.error(e.getLocalizedMessage(), e);
            throw new RuntimeException(e);
        }
        catch (RepositoryException e)
        {
            _logger.warn("Failed to retrieve the datasource from persistence manager.", e);
            throw e;
        }
        
    }

    @Override
    protected void setLogger()
    {
        setLogger(LoggerFactory.getLogger(ReclaimUnusedSpaceTask.class));
    }
}
