/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.metadata;

import java.util.Date;
import java.util.Locale;

import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.TraversableAmetysObject;
import org.ametys.plugins.repository.data.repositorydata.RepositoryData;

/**
 * Metadata container for an {@link AmetysObject}.<br>
 * Note that this can be recursive so that a CompositeMetadata can handle other CompositeMetadata.
 * @deprecated Use {@link RepositoryData} instead
 */
@Deprecated
public interface CompositeMetadata
{
    /**
     * Enumeration for metadata types.
     */
    public enum MetadataType 
    {
        /** Constant for composite metadata */
        COMPOSITE,
        /** Constant for type binary */
        BINARY,
        /** Constant for type richtext */
        RICHTEXT,
        /** Constant for type string */
        STRING,
        /** Constant for type a multilingual string */
        MULTILINGUAL_STRING,
        /** Constant for type boolean */
        BOOLEAN,
        /** Constant for type date */
        DATE,
        /** Constant for type double */
        DOUBLE,
        /** Constant for type long */
        LONG,
        /** Constant for object collection metadata */
        OBJECT_COLLECTION,
        /** Constant for type user */
        USER
    }

    /**
     * Tests if a metadata with a given name exists.
     * @param metadataName the metadataName to test.
     * @return <code>true</code> if the given metadata exists, <code>false</code> otherwise.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public boolean hasMetadata(String metadataName);

    /**
     * Returns the type of the given metadata.
     * @param metadataName metadata name.
     * @return the type of the given metadata.
     * @throws UnknownMetadataException if the named metadata does not exist.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public MetadataType getType(String metadataName) throws UnknownMetadataException, AmetysRepositoryException;
    
    /**
     * Returns the named metadata's value as {@link BinaryMetadata}.<br> 
     * If the metadata does not exist, an {@link UnknownMetadataException} is thrown. 
     * @param metadataName the metadata name.
     * @return the metadata value as BinaryMetadata.
     * @throws UnknownMetadataException if the named metadata does not exist.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public BinaryMetadata getBinaryMetadata(String metadataName) throws UnknownMetadataException, AmetysRepositoryException;

    /**
     * Returns the named metadata's value as {@link RichText}.<br> 
     * If the metadata does not exist, an {@link UnknownMetadataException} is thrown. 
     * @param metadataName the metadata name.
     * @return the metadata value as RichText.
     * @throws UnknownMetadataException if the named metadata does not exist.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public RichText getRichText(String metadataName) throws UnknownMetadataException, AmetysRepositoryException;
    
    /**
     * Returns the named metadata's value as String.<br> 
     * If the metadata is multi-valued, one of the value is returned.<br>
     * If the metadata does not exist, an {@link UnknownMetadataException} is thrown.
     * @param metadataName the metadata name.
     * @return the metadata value as String.
     * @throws UnknownMetadataException if the named metadata does not exist.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public String getString(String metadataName) throws UnknownMetadataException, AmetysRepositoryException;

    /**
     * Returns the named metadata's value as String.<br> 
     * If the metadata is multi-valued, one of the value is returned.<br>
     * If the metadata does not exist, the default value is returned.
     * @param metadataName the metadata name.
     * @param defaultValue the default value.
     * @return the metadata value as String or the default value if metadata is not set.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public String getString(String metadataName, String defaultValue) throws AmetysRepositoryException;

    /**
     * Returns the named metadata's value of a multilingual metadata as String.<br> 
     * @param metadataName the metadata name.
     * @param locale the locale of value to retrieve
     * @return the metadata value as String
     * @throws UnknownMetadataException if the named metadata does not exist for the given locale.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public String getLocalizedString (String metadataName, Locale locale) throws UnknownMetadataException, AmetysRepositoryException;
    
    /**
     * Returns the named metadata's value of a multilingual metadata as String.<br> 
     * If the metadata does not exist, the default value is returned.
     * @param metadataName the metadata name.
     * @param locale the locale of value to retrieve
     * @param defaultValue the default value.
     * @return the metadata value as String or the default value if metadata is not set.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public String getLocalizedString (String metadataName, Locale locale, String defaultValue) throws AmetysRepositoryException;
    
    /**
     * Returns the named metadata's values of a multilingual metadata.<br> 
     * @param metadataName the metadata name.
     * @return the metadata values for all existing locales
     * @throws AmetysRepositoryException if an error occurs.
     */
    public MultilingualString getMultilingualString(String metadataName) throws AmetysRepositoryException;
    
    /**
     * Returns the named metadata's value as String array.
     * @param metadataName metadata name.
     * @return metadata value as String array.
     * @throws UnknownMetadataException if the named metadata does not exist.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public String[] getStringArray(String metadataName) throws UnknownMetadataException, AmetysRepositoryException;

    /**
     * Returns the named metadata's value as String array.<br>
     * If the metadata does not exist, the default values are returned.
     * @param metadataName metadata name.
     * @param defaultValues the default values.
     * @return metadata value as String array or the default values if metadata is not set.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public String[] getStringArray(String metadataName, String[] defaultValues) throws AmetysRepositoryException;

    /**
     * Returns the named metadata's value as Date.<br> 
     * If the metadata is multi-valued, one of the value is returned.<br>
     * If the metadata does not exist, an {@link UnknownMetadataException} is thrown.
     * @param metadataName the metadata name.
     * @return the metadata value as Date.
     * @throws UnknownMetadataException if the named metadata does not exist.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public Date getDate(String metadataName) throws UnknownMetadataException, AmetysRepositoryException;

    /**
     * Returns the named metadata's value as Date.<br> 
     * If the metadata is multi-valued, one of the value is returned.<br>
     * If the metadata does not exist, the default value is returned.
     * @param metadataName the metadata name.
     * @param defaultValue the default value.
     * @return the metadata value as Date or the default value if metadata is not set.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public Date getDate(String metadataName, Date defaultValue) throws AmetysRepositoryException;

    /**
     * Returns the named metadata's value as Date array.
     * @param metadataName metadata name.
     * @return metadata value as Date array.
     * @throws UnknownMetadataException if the named metadata does not exist.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public Date[] getDateArray(String metadataName) throws UnknownMetadataException, AmetysRepositoryException;
    
    /**
     * Returns the named metadata's value as Date array.<br>
     * If the metadata does not exist, the default values are returned.
     * @param metadataName metadata name.
     * @param defaultValues the default values.
     * @return metadata value as Date array or the default values if metadata is not set.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public Date[] getDateArray(String metadataName, Date[] defaultValues) throws AmetysRepositoryException;

    /**
     * Returns the named metadata's value as long.<br> 
     * If the metadata is multi-valued, one of the value is returned.<br>
     * If the metadata does not exist, an {@link UnknownMetadataException} is thrown.
     * @param metadataName the metadata name.
     * @return the metadata value as long.
     * @throws UnknownMetadataException if the named metadata does not exist.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public long getLong(String metadataName) throws UnknownMetadataException, AmetysRepositoryException;

    /**
     * Returns the named metadata's value as long.<br> 
     * If the metadata is multi-valued, one of the value is returned.<br>
     * If the metadata does not exist, the default value is returned.
     * @param metadataName the metadata name.
     * @param defaultValue the default value.
     * @return the metadata value as long or the default value if metadata is not set.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public long getLong(String metadataName, long defaultValue) throws AmetysRepositoryException;
    
    /**
     * Returns the named metadata's value as long array.
     * @param metadataName metadata name.
     * @return metadata value as long array.
     * @throws UnknownMetadataException if the named metadata does not exist.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public long[] getLongArray(String metadataName) throws UnknownMetadataException, AmetysRepositoryException;
    
    /**
     * Returns the named metadata's value as long array.<br>
     * If the metadata does not exist, the default values are returned.
     * @param metadataName metadata name.
     * @param defaultValues the default values.
     * @return metadata value as long array or the default values if metadata is not set.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public long[] getLongArray(String metadataName, long[] defaultValues) throws AmetysRepositoryException;

    /**
     * Returns the named metadata's value as double.<br> 
     * If the metadata is multi-valued, one of the value is returned.<br>
     * If the metadata does not exist, an {@link UnknownMetadataException} is thrown.
     * @param metadataName the metadata name.
     * @return the metadata value as double.
     * @throws UnknownMetadataException if the named metadata does not exist.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public double getDouble(String metadataName) throws UnknownMetadataException, AmetysRepositoryException;
    
    /**
     * Returns the named metadata's value as double.<br> 
     * If the metadata is multi-valued, one of the value is returned.<br>
     * If the metadata does not exist, the default value is returned.
     * @param metadataName the metadata name.
     * @param defaultValue the default value.
     * @return the metadata value as double or the default value if metadata is not set.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public double getDouble(String metadataName, double defaultValue) throws AmetysRepositoryException;
    
    /**
     * Returns the named metadata's value as double array.
     * @param metadataName metadata name.
     * @return metadata value as double array.
     * @throws UnknownMetadataException if the named metadata does not exist.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public double[] getDoubleArray(String metadataName) throws UnknownMetadataException, AmetysRepositoryException;
    
    /**
     * Returns the named metadata's value as double array.<br>
     * If the metadata does not exist, the default values are returned.
     * @param metadataName metadata name.
     * @param defaultValues the default values.
     * @return metadata value as double array or the default values if metadata is not set.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public double[] getDoubleArray(String metadataName, double[] defaultValues) throws AmetysRepositoryException;

    /**
     * Returns the named metadata's value as boolean.<br> 
     * If the metadata is multi-valued, one of the value is returned.<br>
     * If the metadata does not exist, an {@link UnknownMetadataException} is thrown.
     * @param metadataName the metadata name.
     * @return the metadata value as boolean.
     * @throws UnknownMetadataException if the named metadata does not exist.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public boolean getBoolean(String metadataName) throws UnknownMetadataException, AmetysRepositoryException;
    
    /**
     * Returns the named metadata's value as boolean.<br> 
     * If the metadata is multi-valued, one of the value is returned.<br>
     * If the metadata does not exist, the default value is returned.
     * @param metadataName the metadata name.
     * @param defaultValue the default value.
     * @return the metadata value as boolean or the default value if metadata is not set.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public boolean getBoolean(String metadataName, boolean defaultValue) throws AmetysRepositoryException;
    
    /**
     * Returns the named metadata's value as boolean array.
     * @param metadataName metadata name.
     * @return metadata value as boolean array.
     * @throws UnknownMetadataException if the named metadata does not exist.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public boolean[] getBooleanArray(String metadataName) throws UnknownMetadataException, AmetysRepositoryException;
    
    /**
     * Returns the named metadata's value as boolean array.<br>
     * If the metadata does not exist, the default values are returned.
     * @param metadataName metadata name.
     * @param defaultValues the default values.
     * @return metadata value as boolean array or the default values if metadata is not set.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public boolean[] getBooleanArray(String metadataName, boolean[] defaultValues) throws AmetysRepositoryException;
    
    /**
     * Returns the named metadata's value as {@link UserIdentity}.<br> 
     * If the metadata does not exist, an {@link UnknownMetadataException} is thrown.
     * @param metadataName the metadata name.
     * @return the metadata value as user identity.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public UserIdentity getUser(String metadataName) throws AmetysRepositoryException;
    
    /**
     * Returns the named metadata's value as {@link UserIdentity}.<br> 
     * If the metadata does not exist, the default value is returned.
     * @param metadataName the metadata name.
     * @param defaultValue  the default value.
     * @return the metadata value as user identity or the default value if metadata is not set.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public UserIdentity getUser(String metadataName, UserIdentity defaultValue) throws AmetysRepositoryException;
    
    /**
     * Returns the named metadata's value as {@link UserIdentity} array.
     * @param metadataName the metadata name.
     * @return metadata value as user identity array.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public UserIdentity[] getUserArray(String metadataName) throws AmetysRepositoryException;
    
    /**
     * Returns the named metadata's value as {@link UserIdentity} array.<br>
     * If the metadata does not exist, the default values are returned.
     * @param metadataName the metadata name.
     * @param defaultValues  the default values.
     * @return metadata value as user identity array or the default values if metadata is not set.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public UserIdentity[] getUserArray(String metadataName, UserIdentity[] defaultValues) throws AmetysRepositoryException;

    /**
     * Returns the named metadata's value as {@link CompositeMetadata}.<br> 
     * If the metadata is multi-valued, one of the value is returned.<br>
     * If the metadata does not exist an {@link UnknownMetadataException} is thrown.
     * @param metadataName the metadata name.
     * @return the metadata value as {@link CompositeMetadata}.
     * @throws UnknownMetadataException if the named metadata does not exist.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public CompositeMetadata getCompositeMetadata(String metadataName) throws UnknownMetadataException, AmetysRepositoryException;
    
    /**
     * Returns an object collection metadata as a {@link TraversableAmetysObject}<br>.
     * @param metadataName the metadata name.
     * @return the metadata as a {@link TraversableAmetysObject}.
     * @throws UnknownMetadataException if the named metadata does not exist.
     * @throws AmetysRepositoryException if an error occurs.
     */
    TraversableAmetysObject getObjectCollection(String metadataName) throws AmetysRepositoryException;
    
    /**
     * Returns an array containing metadata names.
     * @return an array containing metadata names.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public String[] getMetadataNames() throws AmetysRepositoryException;

    /**
     * Test if the given metadata is multiple.
     * @param metadataName the metadata name.
     * @return <code>true</code> if the given metadata is multiple,
     *         <code>false</code> otherwise.
     * @throws UnknownMetadataException if the named metadata does not exist.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public boolean isMultiple(String metadataName) throws UnknownMetadataException, AmetysRepositoryException;
    
    /**
     * Copy the current {@link CompositeMetadata} to the given composite metadata.
     * @param metadata The parent composite metadata. Can not be null.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public void copyTo (ModifiableCompositeMetadata metadata) throws AmetysRepositoryException;
}
