/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.metadata;

import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;

import org.ametys.plugins.repository.AmetysRepositoryException;

/**
 * Class representing a multilingual String
 */
public class MultilingualString
{
    private Map<Locale, String> _values;

    /**
     * Construct an empty multilingual string
     */
    public MultilingualString ()
    {
        _values = new HashMap<>();
    }
    
    /**
     * Construct an multilingual string
     * @param values The values of each locale
     */
    public MultilingualString (Map<Locale, String> values)
    {
        _values = values;
    }
    
    /**
     * Add or replace value for a specific locale.
     * @param locale The locale. Can not be null.
     * @param value The value for this locale.
     */
    public void add(Locale locale, String value)
    {
        _values.put(locale, value);
    }
    
    /**
     * Returns the value in the given locale as String.<br> 
     * @param locale the locale of value to retrieve
     * @return the value as String or <code>null</code> if not found
     */
    public String getValue(Locale locale)
    {
        return _values.get(locale);
    }
    
    /**
     * Returns the existing {@link Locale}s
     * @return the locales
     */
    public Set<Locale> getLocales()
    {
        return _values.keySet();
    }
    
    /**
     * Get all non-empty values
     * @return the values
     */
    public List<String> getValues()
    {
        return _values.values().stream().filter(v -> StringUtils.isNotEmpty(v)).collect(Collectors.toList());
    }
    
    /**
     * Determines if this multilingual string has value for the given locale
     * @param locale The locale to test
     * @return <code>true</code> if a value exists for this locale
     * @throws AmetysRepositoryException if an error occurs.
     */
    public boolean hasLocale(Locale locale)
    {
        return _values.containsKey(locale);
    }
    
    
    @Override
    public boolean equals(Object obj)
    {
        if (this == obj)
        {
            return true;
        }
        if (obj == null)
        {
            return false;
        }
        if (!(obj instanceof MultilingualString))
        {
            return false;
        }
        
        MultilingualString other = (MultilingualString) obj;
        
        Locale[] locales = getLocales().stream().toArray(Locale[]::new);
        Locale[] otherLocales = other.getLocales().stream().toArray(Locale[]::new);
        
        if (!Arrays.deepEquals(locales, otherLocales))
        {
            return false;
        }
        
        for (Locale locale : locales)
        {
            if (!getValue(locale).equals(other.getValue(locale)))
            {
                return false;
            }
        }
        
        return true;
    }
    
    @Override
    public int hashCode()
    {
        return _values.hashCode();
    }
}
