/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.metadata;

import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;

import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.LocaleUtils;
import org.apache.commons.lang3.StringUtils;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.core.util.dom.DOMUtils;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.data.holder.ModelAwareDataHolder;
import org.ametys.plugins.repository.data.holder.ModelLessDataHolder;

/**
 * Helper methods for {@link MultilingualString} metadata
 */
public final class MultilingualStringHelper
{
    /** The default locale */
    public static final Locale DEFAULT_LOCALE = Locale.ENGLISH;
    
    /** The separator between the locale and the value for the string representation of a multilingual string */
    private static final String __LOCALE_AND_VALUE_SEPARATOR = ":";
    /** The separator between the entries for the string representation of a multilingual string */
    private static final String __ENTRIES_SEPARATOR = "#";
 
    private MultilingualStringHelper()
    {
        // Hide default constructor.
    }
    
    /**
     * Returns the closest non-empty value for a {@link MultilingualString} metadata in a given locale.
     * If no close locale is found, return the value for {@link Locale#ENGLISH} if exists.<br>
     * Otherwise, the value of first stored locale will be returned.<br>
     * @param parentMetadata The metadata holder
     * @param metadataName The metadata name
     * @param locale The requested locale
     * @return the closest non-empty localized value or <code>null</code> if not found.
     * @throws AmetysRepositoryException if an errors occurs.
     * @deprecated Use {@link #getValue(ModelAwareDataHolder, String, Locale)} or {@link #getValue(ModelLessDataHolder, String, Locale)} instead
     */
    @Deprecated
    public static String getValue(CompositeMetadata parentMetadata, String metadataName, Locale locale) throws AmetysRepositoryException
    {
        if (parentMetadata.hasMetadata(metadataName))
        {
            MultilingualString multilingualString = parentMetadata.getMultilingualString(metadataName);
            
            return getValue(multilingualString, locale);
        }
        
        return null;
    }
    
    /**
     * Returns the closest non-empty value for a {@link MultilingualString} element in a given locale.
     * If no close locale is found, return the value for {@link Locale#ENGLISH} if exists.<br>
     * Otherwise, the value of first stored locale will be returned.<br>
     * @param dataHolder data holder
     * @param dataName The data name
     * @param locale The requested locale
     * @return the closest non-empty localized value or <code>null</code> if not found.
     * @throws AmetysRepositoryException if an errors occurs.
     */
    public static String getValue(ModelLessDataHolder dataHolder, String dataName, Locale locale) throws AmetysRepositoryException
    {
        if (dataHolder.hasValue(dataName))
        {
            MultilingualString multilingualString = dataHolder.getValue(dataName);
            return getValue(multilingualString, locale);
        }
        return null;
    }
    
    /**
     * Returns the closest non-empty value for a {@link MultilingualString} element in a given locale.
     * If no close locale is found, return the value for {@link Locale#ENGLISH} if exists.<br>
     * Otherwise, the value of first stored locale will be returned.<br>
     * @param dataHolder The data holder
     * @param dataName The data name
     * @param locale The requested locale
     * @return the closest non-empty localized value or <code>null</code> if not found.
     * @throws AmetysRepositoryException if an errors occurs.
     */
    public static String getValue(ModelAwareDataHolder dataHolder, String dataName, Locale locale) throws AmetysRepositoryException
    {
        if (dataHolder.hasValue(dataName))
        {
            MultilingualString multilingualString = dataHolder.getValue(dataName);
            return getValue(multilingualString, locale);
        }
        return null;
    }
    
    /**
     * Returns the closest non-empty value for a {@link MultilingualString} in a given locale.
     * If no close locale is found, return the value for {@link Locale#ENGLISH} if exists.<br>
     * Otherwise, the value of first stored locale will be returned.<br>
     * @param multilingualString The multilingual string
     * @param locale The requested locale. Can be null.
     * @return the closest non-empty localized value or <code>null</code> if not found.
     * @throws AmetysRepositoryException if an errors occurs.
     */
    public static String getValue(MultilingualString multilingualString, Locale locale) throws AmetysRepositoryException
    {
        Locale closestLocale = getClosestNonEmptyLocale(multilingualString, locale);
        if (closestLocale != null)
        {
            return multilingualString.getValue(closestLocale);
        }
        else
        {
            return null;
        }
    }
    
    /**
     * Returns the closest non-empty locale for a {@link MultilingualString} in a given locale.
     * If no close locale is found, return the {@link Locale#ENGLISH} if a values exists.<br>
     * Otherwise, the first stored locale will be returned.<br>
     * @param multilingualString The multilingual string
     * @param locale The requested locale. Can be null.
     * @return the closest non-empty locale or <code>null</code> if not found.
     * @throws AmetysRepositoryException if an errors occurs.
     */
    public static Locale getClosestNonEmptyLocale(MultilingualString multilingualString, Locale locale) throws AmetysRepositoryException
    {
        if (locale != null && multilingualString.hasLocale(locale))
        {
            return locale;
        }
        else
        {
            // Try to find the closest locale
            
            List<Locale> closedLocales = localeLookupList(locale);
            for (Locale closedLocale : closedLocales)
            {
                if (multilingualString.hasLocale(closedLocale))
                {
                    return closedLocale;
                }
            }
            
            // No locale found, get the first stored locale
            Set<Locale> allLocales = multilingualString.getLocales();
            if (!allLocales.isEmpty())
            {
                return allLocales.iterator().next();
            }
            
            return null;
        }
    }
    
    /**
     * Return the list of closest locales to search
     * @param locale the locale to start from. If null, returns the default locale
     * @return the unmodifiable list of Locale objects, 0 being locale, not null
     */
    public static List<Locale> localeLookupList(Locale locale)
    {
        if (locale != null)
        {
            return org.apache.commons.lang3.LocaleUtils.localeLookupList(locale, DEFAULT_LOCALE);
        }
        else
        {
            return Collections.singletonList(DEFAULT_LOCALE);
        }
    }
    
    /**
     * Saxes the given multilingual string
     * @param contentHandler the content handler where to SAX into.
     * @param tagName the name of the tag to sax the multilingual string
     * @param multilingualString the multilingual string to sax
     * @param locale the requested locale. Can be null.
     * @throws SAXException if an errors occurs during the value saxing
     * @deprecated Use {@link #sax(ContentHandler, String, MultilingualString, AttributesImpl, Locale)} instead
     */
    @Deprecated
    public static void sax(ContentHandler contentHandler, String tagName, MultilingualString multilingualString, Locale locale) throws SAXException
    {
        sax(contentHandler, tagName, multilingualString, new AttributesImpl(), locale);
    }
    
    /**
     * Saxes the given multilingual string
     * @param contentHandler the content handler where to SAX into.
     * @param tagName the name of the tag to sax the multilingual string
     * @param multilingualString the multilingual string to sax
     * @param attributes the attributes to sax the multilingual string
     * @param locale the requested locale. Can be null.
     * @throws SAXException if an errors occurs during the value saxing
     */
    public static void sax(ContentHandler contentHandler, String tagName, MultilingualString multilingualString, AttributesImpl attributes, Locale locale) throws SAXException
    {
        AttributesImpl localAttributes = new AttributesImpl(attributes);
        if (locale == null)
        {
            // Given locale is null, sax all existing locales
            XMLUtils.startElement(contentHandler, tagName, localAttributes);
            for (Locale valueLocale : multilingualString.getLocales())
            {
                XMLUtils.createElement(contentHandler, valueLocale.toString(), multilingualString.getValue(valueLocale));
            }
            XMLUtils.endElement(contentHandler, tagName);
        }
        else
        {
            Locale closestLocale = getClosestNonEmptyLocale(multilingualString, locale);
            if (closestLocale != null)
            {
                localAttributes.addCDATAAttribute("lang", closestLocale.toString());
                XMLUtils.createElement(contentHandler, tagName, localAttributes, multilingualString.getValue(closestLocale));
            }
        }
    }
    
    /**
     * Get the {@link MultilingualString} object from the given {@link Node}
     * @param element the DOM element containing the multilingual string data
     * @return the {@link MultilingualString} object
     */
    public static MultilingualString fromXML(Element element)
    {
        if (element != null)
        {
            String lang = element.getAttribute("lang");
            if (StringUtils.isNotEmpty(lang))
            {
                MultilingualString multilingualString = new MultilingualString();
                String value = element.getTextContent();
                multilingualString.add(LocaleUtils.toLocale(lang), value);
                return multilingualString;
            }
            else
            {
                MultilingualString multilingualString = new MultilingualString();
                for (Element entry : DOMUtils.getChildElements(element))
                {
                    String languageTag = entry.getNodeName();
                    String value = entry.getTextContent();
                    multilingualString.add(LocaleUtils.toLocale(languageTag), value);
                }
                return multilingualString;
            }
        }
        
        return null;
    }
    
    /**
     * Get the JSON representation of a {@link MultilingualString}
     * @param multilingualString The multilingual string. Cannot be null.
     * @return A map with the locales and values.
     * @throws AmetysRepositoryException if an error occurs
     */
    public static Map<String, Object> toJson(MultilingualString multilingualString) throws AmetysRepositoryException
    {
        Map<String, Object> json = new LinkedHashMap<>();
        
        for (Locale locale : multilingualString.getLocales())
        {
            json.put(locale.toString(), multilingualString.getValue(locale));
        }
        
        return json;
    }
    
    /**
     * Get the {@link MultilingualString} object from its JSON representation
     * @param json the JSON representation of the multilingual string
     * @return the {@link MultilingualString} object
     */
    public static MultilingualString fromJSON(Map<String, ? extends Object> json)
    {
        if (json == null)
        {
            return null;
        }
        
        MultilingualString multilingualString = new MultilingualString();
        
        for (Map.Entry<String, ? extends Object> entry : json.entrySet())
        {
            Locale locale = LocaleUtils.toLocale(entry.getKey());
            String value = entry.getValue().toString();
            multilingualString.add(locale, value);
        }
        
        return multilingualString;
    }
    
    /**
     * Retrieves a string representation of a {@link MultilingualString}
     * @param multilingualString the multilingual string
     * @return thestring representation of the multilingual string
     */
    public static String toString(MultilingualString multilingualString)
    {
        if (multilingualString != null)
        {
            StringBuilder asString = new StringBuilder();
            Iterator<Locale> localesIterator = multilingualString.getLocales().iterator();
            while (localesIterator.hasNext())
            {
                Locale locale = localesIterator.next();
                asString.append(locale.toString()).append(__LOCALE_AND_VALUE_SEPARATOR).append(multilingualString.getValue(locale));
                if (localesIterator.hasNext())
                {
                    asString.append(__ENTRIES_SEPARATOR);
                }
            }
            return asString.toString();
        }
        else
        {
            return null;
        }
    }
    
    /**
     * Retrieves the {@link MultilingualString} from its string representation
     * @param string the string representation of the multilingual string
     * @return the multilingual string from its string representation, or <code>null</code> if the given string value is null or empty
     * @throws IllegalArgumentException if the given string value can't be cast to a multilingual string
     */
    public static MultilingualString fromString(String string) throws IllegalArgumentException
    {
        if (StringUtils.isEmpty(string))
        {
            return null;
        }
        
        if (string.contains(__LOCALE_AND_VALUE_SEPARATOR))
        {
            MultilingualString multilingualString = new MultilingualString();
            
            String[] entries = string.split(__ENTRIES_SEPARATOR);
            for (String entry : entries)
            {
                String localeAsString = StringUtils.substringBeforeLast(entry, __LOCALE_AND_VALUE_SEPARATOR);
                String value = StringUtils.substringAfterLast(entry, __LOCALE_AND_VALUE_SEPARATOR);
                multilingualString.add(LocaleUtils.toLocale(localeAsString), value);
            }
            
            return multilingualString;
        }
        else
        {
            throw new IllegalArgumentException("Unable to cast '" + string + "' to a multilingual string");
        }
    }
    
    /**
     * Check if a given string matches the Multilingual string pattern
     * @param string the string representation of the multilingual string to check
     * @return <code>true</code> if the string have a correct pattern, <code>false</code> otherwise
     */
    public static boolean matchesMultilingualStringPattern(String string)
    {
        if (string.contains(__LOCALE_AND_VALUE_SEPARATOR))
        {
            boolean localeExists = true;
            
            String[] entries = string.split(__ENTRIES_SEPARATOR);
            for (String entry : entries)
            {
                String localeAsString = StringUtils.substringBeforeLast(entry, __LOCALE_AND_VALUE_SEPARATOR);
                
                try
                {
                    // Try to convert the string part to a Locale
                    LocaleUtils.toLocale(localeAsString);
                }
                catch (IllegalArgumentException e)
                {
                    // If at least one string part that should represent a locale is not a locale,
                    // then the string is not considered as a multilingual string
                    return false;
                }
            }
            return localeExists;
        }
        return false;
    }
}
