/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.metadata.jcr;

import java.time.Instant;
import java.time.ZoneOffset;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Set;
import java.util.regex.Pattern;

import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.PathNotFoundException;
import javax.jcr.Property;
import javax.jcr.PropertyIterator;
import javax.jcr.PropertyType;
import javax.jcr.RepositoryException;
import javax.jcr.Value;
import javax.jcr.lock.Lock;
import javax.jcr.lock.LockManager;

import org.apache.commons.lang3.LocaleUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.jackrabbit.JcrConstants;

import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.ModifiableTraversableAmetysObject;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.data.repositorydata.impl.JCRRepositoryData;
import org.ametys.plugins.repository.jcr.NodeTypeHelper;
import org.ametys.plugins.repository.metadata.ModifiableBinaryMetadata;
import org.ametys.plugins.repository.metadata.ModifiableCompositeMetadata;
import org.ametys.plugins.repository.metadata.ModifiableFile;
import org.ametys.plugins.repository.metadata.ModifiableFolder;
import org.ametys.plugins.repository.metadata.ModifiableResource;
import org.ametys.plugins.repository.metadata.ModifiableRichText;
import org.ametys.plugins.repository.metadata.MultilingualString;
import org.ametys.plugins.repository.metadata.UnknownMetadataException;

/**
 * JCR implementation of a CompositeMetadata.<br>
 * This implementation is based on a child node of the Node supporting the actual content.
 * @deprecated Use {@link JCRRepositoryData} instead
 */
@Deprecated
public class JCRCompositeMetadata implements ModifiableCompositeMetadata
{
    /** Prefix for jcr properties and nodes names */
    public static final String METADATA_PREFIX = RepositoryConstants.NAMESPACE_PREFIX + ':';
    
    /** The JCR nodetype of object collection metadata. */
    public static final String OBJECT_COLLECTION_METADATA_NODETYPE = METADATA_PREFIX + "objectCollectionMetadata";
    
    /** The JCR nodetype of multilingual metadata. */
    public static final String MULTILINGUAL_STRING_METADATA_NODETYPE = METADATA_PREFIX + "multilingualString";
    
    private static Pattern __metadataNamePattern;

    private Node _node;
    private boolean _lockAlreadyChecked;
    private AmetysObjectResolver _resolver;

    /**
     * Constructor
     * @param metadataNode the Node supporting this composite metadata
     * @param resolver The resolver, used to resolve object collections.
     */
    public JCRCompositeMetadata(Node metadataNode, AmetysObjectResolver resolver)
    {
        _node = metadataNode;
        _resolver = resolver;
    }
    
    /**
     * Retrieves the underlying node.
     * @return the underlying node.
     */
    public Node getNode()
    {
        return _node;
    }
    
    public boolean hasMetadata(String metadataName) throws AmetysRepositoryException
    {
        try
        {
            if (_node.hasProperty(METADATA_PREFIX + metadataName))
            {
                Property property = _node.getProperty(METADATA_PREFIX + metadataName);
                if (property.getDefinition().isMultiple())
                {
                    return true;
                }
                else
                {
                    MetadataType type = getType(metadataName);
                    switch (type)
                    {
                        case STRING:
                            return !"".equals(property.getString());
                        default:
                            return true;
                    }
                }
            }
            else
            {
                // TODO filter sub-nodes by type (like in getMetadataNames).
                boolean hasNode = _node.hasNode(METADATA_PREFIX + metadataName);
                if (!hasNode)
                {
                    return false;
                }
                
                Node childNode = _node.getNode(METADATA_PREFIX + metadataName);
                boolean isEmpty = childNode.hasProperty(RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":isEmpty") && childNode.getProperty(RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":isEmpty").getBoolean();
                if (isEmpty)
                {
                    return false;
                }
                
                if (NodeTypeHelper.isNodeType(childNode, RepositoryConstants.MULTIPLE_ITEM_NODETYPE))
                {
                    NodeIterator it = childNode.getNodes(METADATA_PREFIX + "*");
                    if (it.getSize() == 0)
                    {
                        return false;
                    }
                }
                
                if (NodeTypeHelper.isNodeType(childNode, RepositoryConstants.USER_NODETYPE))
                {
                    try
                    {
                        String login = childNode.getProperty("ametys:login").getString();
                        String population = childNode.getProperty("ametys:population").getString();
                        
                        if (StringUtils.isEmpty(login) || StringUtils.isEmpty(population))
                        {
                            return false;
                        }
                    }
                    catch (RepositoryException e)
                    {
                        return false;
                    }
                }
                
                if (NodeTypeHelper.isNodeType(childNode, RepositoryConstants.GEOCODE_NODETYPE))
                {
                    try
                    {
                        return childNode.hasProperty("ametys:latitude") && childNode.hasProperty("ametys:longitude");
                    }
                    catch (RepositoryException e)
                    {
                        return false;
                    }
                }
                
                return true;
            }
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException(e);
        }
    }
    
    @Override
    public void rename(String newName) throws AmetysRepositoryException
    {
        try
        {
            getNode().getSession().move(getNode().getPath(), getNode().getParent().getPath() + "/" + newName);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException(e);
        }
    }

    public void removeMetadata(String metadataName) throws AmetysRepositoryException
    {
        try
        {
            _checkLock();
            String realMetadataName = METADATA_PREFIX + metadataName;
            
            if (_node.hasProperty(realMetadataName))
            {
                _node.getProperty(realMetadataName).remove();
            }
            else if (_node.hasNode(realMetadataName))
            {
                // Remove all existing nodes
                NodeIterator it = _node.getNodes(realMetadataName);
                while (it.hasNext())
                {
                    Node next = it.nextNode();
                    next.remove();
                }
            }
            else
            {
                throw new UnknownMetadataException("No metadata for name: " + metadataName);
            }
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to remove metadata " + metadataName, e);
        }
    }
    
    @Override
    public void copyTo(ModifiableCompositeMetadata metadata) throws AmetysRepositoryException
    {
        String[] metadataNames = getMetadataNames();
        for (String metadataName : metadataNames)
        {
            if (hasMetadata(metadataName))
            {
                MetadataType type = getType(metadataName);
                
                boolean multiple = isMultiple(metadataName);
                switch (type)
                {
                    case COMPOSITE:
                        ModifiableCompositeMetadata compositeMetadata = metadata.getCompositeMetadata(metadataName, true);
                        getCompositeMetadata(metadataName).copyTo(compositeMetadata);
                        break;
                    case USER:
                        if (multiple)
                        {
                            metadata.setMetadata(metadataName, getUserArray(metadataName));
                        }
                        else
                        {
                            metadata.setMetadata(metadataName, getUser(metadataName));
                        }
                        break;
                    case BOOLEAN:
                        if (multiple)
                        {
                            metadata.setMetadata(metadataName, getBooleanArray(metadataName));
                        }
                        else
                        {
                            metadata.setMetadata(metadataName, getBoolean(metadataName));
                        }
                        break;
                    case DATE:
                        if (multiple)
                        {
                            metadata.setMetadata(metadataName, getDateArray(metadataName));
                        }
                        else
                        {
                            metadata.setMetadata(metadataName, getDate(metadataName));
                        }
                        break;
                    case DOUBLE:
                        if (multiple)
                        {
                            metadata.setMetadata(metadataName, getDoubleArray(metadataName));
                        }
                        else
                        {
                            metadata.setMetadata(metadataName, getDouble(metadataName));
                        }
                        break;
                    case LONG:
                        if (multiple)
                        {
                            metadata.setMetadata(metadataName, getLongArray(metadataName));
                        }
                        else
                        {
                            metadata.setMetadata(metadataName, getLong(metadataName));
                        }
                        break;
                    case BINARY:
                        ModifiableBinaryMetadata binaryMetadata = metadata.getBinaryMetadata(metadataName, true);
                        ModifiableBinaryMetadata srcBinary = getBinaryMetadata(metadataName);
                        
                        binaryMetadata.setFilename(srcBinary.getFilename());
                        binaryMetadata.setMimeType(srcBinary.getMimeType());
                        binaryMetadata.setLastModified(srcBinary.getLastModified());
                        binaryMetadata.setEncoding(srcBinary.getEncoding());
                        binaryMetadata.setInputStream(srcBinary.getInputStream());
                        break;
                    case RICHTEXT:
                        ModifiableRichText richText = metadata.getRichText(metadataName, true);
                        ModifiableRichText srcRichText = getRichText(metadataName);
                        
                        richText.setInputStream(srcRichText.getInputStream());
                        richText.setEncoding(srcRichText.getEncoding());
                        richText.setLastModified(srcRichText.getLastModified());
                        richText.setMimeType(srcRichText.getMimeType());
                        
                        // Copy additional data
                        _copyDataFolder (srcRichText.getAdditionalDataFolder(), richText.getAdditionalDataFolder());
                        break;
                    case MULTILINGUAL_STRING:
                        MultilingualString srcMultilingualString = getMultilingualString(metadataName);
                        metadata.setMetadata(metadataName, srcMultilingualString);
                        break;
                    default:
                        if (multiple)
                        {
                            metadata.setMetadata(metadataName, getStringArray(metadataName));
                        }
                        else
                        {
                            metadata.setMetadata(metadataName, getString(metadataName));
                        }
                        break;
                }
            }
        }
    }
    
    private void _copyDataFolder (ModifiableFolder srcDataFolder, ModifiableFolder targetDataFolder)
    {
        // Copy folders
        Collection<ModifiableFolder> folders = srcDataFolder.getFolders();
        for (ModifiableFolder srcSubfolder : folders)
        {
            ModifiableFolder targetSubFolder = targetDataFolder.addFolder(srcSubfolder.getName());
            _copyDataFolder (srcSubfolder, targetSubFolder);
        }
        
        // Copy files
        Collection<ModifiableFile> files = srcDataFolder.getFiles();
        for (ModifiableFile srcFile : files)
        {
            _copyFile (srcFile, targetDataFolder);
        }
    }
    
    private void _copyFile (ModifiableFile srcFile, ModifiableFolder targetDataFolder)
    {
        ModifiableResource srcResource = srcFile.getResource();
        
        ModifiableFile targetFile = targetDataFolder.addFile(srcFile.getName());
        ModifiableResource targetResource = targetFile.getResource();
        
        targetResource.setLastModified(srcResource.getLastModified());
        targetResource.setMimeType(srcResource.getMimeType());
        targetResource.setEncoding(srcResource.getEncoding());
        targetResource.setInputStream(srcResource.getInputStream());
    }

    public MetadataType getType(String metadataName) throws AmetysRepositoryException
    {
        try
        {
            String realMetadataName = METADATA_PREFIX + metadataName;
            
            if (_node.hasNode(realMetadataName))
            {
                Node node = _node.getNode(realMetadataName);
                String type = node.getPrimaryNodeType().getName();
                if (JcrConstants.NT_FROZENNODE.equals(_node.getPrimaryNodeType().getName()))
                {
                    // on version history
                    type = node.getProperty(JcrConstants.JCR_FROZENPRIMARYTYPE).getString();
                }
                
                if (type.equals(OBJECT_COLLECTION_METADATA_NODETYPE))
                {
                    return MetadataType.OBJECT_COLLECTION;
                }
                else if (type.equals(RepositoryConstants.BINARY_NODETYPE))
                {
                    return MetadataType.BINARY;
                }
                else if (type.equals(METADATA_PREFIX + "richText"))
                {
                    return MetadataType.RICHTEXT;
                }
                else if (type.equals(METADATA_PREFIX + "user"))
                {
                    return MetadataType.USER;
                }
                else if (type.equals(MULTILINGUAL_STRING_METADATA_NODETYPE))
                {
                    return MetadataType.MULTILINGUAL_STRING;
                }
                else
                {
                    return MetadataType.COMPOSITE;
                }
            }

            if (_node.hasProperty(realMetadataName))
            {
                Property property = _node.getProperty(realMetadataName);
                int type = property.getType();
    
                switch (type)
                {
                    case PropertyType.STRING:
                        return MetadataType.STRING;
                    case PropertyType.BOOLEAN:
                        return MetadataType.BOOLEAN;
                    case PropertyType.DATE:
                        return MetadataType.DATE;
                    case PropertyType.DOUBLE:
                        return MetadataType.DOUBLE;
                    case PropertyType.LONG:
                        return MetadataType.LONG;
                    default:
                        throw new AmetysRepositoryException("Unhandled JCR property type : " + PropertyType.nameFromValue(type));
                }
            }
            
            throw new UnknownMetadataException("No metadata for name: " + metadataName);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get property type", e);
        }
    }

    public String getString(String metadataName)
    {
        return _getString(metadataName, null, true);
    }

    @Override
    public String getString(String metadataName, String defaultValue)
    {
        return _getString(metadataName, defaultValue, false);
    }
    
    private String _getString(String metadataName, String defaultValue, boolean failIfNotExist)
    {
        try
        {
            Property property = _node.getProperty(METADATA_PREFIX + metadataName);

            if (property.getDefinition().isMultiple())
            {
                Value[] values = property.getValues();

                if (values.length > 0)
                {
                    return values[0].getString();
                }
                else
                {
                    throw new AmetysRepositoryException("Cannot get a String from an empty array");
                }
            }

            return property.getString();
        }
        catch (PathNotFoundException e)
        {
            if (failIfNotExist)
            {
                throw new UnknownMetadataException("Unknown metadata " + METADATA_PREFIX + metadataName + " of " + _node, e);
            }
            else
            {
                return defaultValue;
            }
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get String property: " + METADATA_PREFIX + metadataName + " of " + _node, e);
        }
    }

    public String[] getStringArray(String metadataName) throws AmetysRepositoryException
    {
        return _getStringArray(metadataName, null, true);
    }

    public String[] getStringArray(String metadataName, String[] defaultValues)
    {
        return _getStringArray(metadataName, defaultValues, false);
    }
    
    private String[] _getStringArray(String metadataName, String[] defaultValues, boolean failIfNotExist)
    {
        try
        {
            Property property = _node.getProperty(METADATA_PREFIX + metadataName);

            if (property.getDefinition().isMultiple())
            {
                Value[] values = property.getValues();

                String[] results = new String[values.length];
                for (int i = 0; i < values.length; i++)
                {
                    Value value = values[i];
                    results[i] = value.getString();
                }

                return results;
            }

            Value value = property.getValue();
            return new String[] {value.getString()};
        }
        catch (PathNotFoundException e)
        {
            if (failIfNotExist)
            {
                throw new UnknownMetadataException("Unknown metadata " + METADATA_PREFIX + metadataName + " of " + _node, e);
            }
            else
            {
                return defaultValues;
            }
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get String array property: " + METADATA_PREFIX + metadataName + " of " + _node, e);
        }
    }
    
    @Override
    public String getLocalizedString(String metadataName, Locale locale) throws UnknownMetadataException, AmetysRepositoryException
    {
        return _getString(metadataName, locale, null, true);
    }
    
    @Override
    public String getLocalizedString(String metadataName, Locale locale, String defaultValue) throws AmetysRepositoryException
    {
        return _getString(metadataName, locale, defaultValue, false);
    }
    
    @Override
    public MultilingualString getMultilingualString(String metadataName) throws AmetysRepositoryException
    {
        try
        {
            if (_node.hasNode(METADATA_PREFIX + metadataName))
            {
                Node node = _node.getNode(METADATA_PREFIX + metadataName);
                
                MultilingualString values = new MultilingualString();
                
                PropertyIterator properties = node.getProperties(METADATA_PREFIX + "*");
                while (properties.hasNext())
                {
                    Property property = properties.nextProperty();
                    String name = property.getName().substring(METADATA_PREFIX.length());
                    
                    try
                    {
                        Locale locale = LocaleUtils.toLocale(name);
                        values.add(locale, property.getString());
                    }
                    catch (IllegalArgumentException e)
                    {
                        // Property is not a valid locale, ignore it.
                    }
                }
                
                return values;
            }
            else
            {
                throw new UnknownMetadataException("No multilingual String named " + metadataName);
            }
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get multilingual String property", e);
        }
    }
    
    private String _getString(String metadataName, Locale locale, String defaultValue, boolean failIfNotExist)
    {
        try
        {
            Node node = _node.getNode(METADATA_PREFIX + metadataName);
            
            Property property = node.getProperty(METADATA_PREFIX + locale.toString());
            return property.getString();
        }
        catch (PathNotFoundException e)
        {
            if (failIfNotExist)
            {
                throw new UnknownMetadataException("Unknown metadata " + METADATA_PREFIX + metadataName + " of " + _node, e);
            }
            else
            {
                return defaultValue;
            }
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get String property: " + METADATA_PREFIX + metadataName + " of " + _node, e);
        }
    }

    public boolean getBoolean(String metadataName) throws UnknownMetadataException, AmetysRepositoryException
    {
        return _getBoolean(metadataName, null, true);
    }

    public boolean getBoolean(String metadataName, boolean defaultValue) throws AmetysRepositoryException
    {
        return _getBoolean(metadataName, defaultValue, false);
    }
    
    private boolean _getBoolean(String metadataName, Boolean defaultValue, boolean failIfNotExist)
    {
        try
        {
            Property property = _node.getProperty(METADATA_PREFIX + metadataName);

            if (property.getDefinition().isMultiple())
            {
                Value[] values = property.getValues();

                if (values.length > 0)
                {
                    return values[0].getBoolean();
                }
                else
                {
                    throw new AmetysRepositoryException("Cannot get a boolean from an empty array");
                }
            }

            return property.getBoolean();
        }
        catch (PathNotFoundException e)
        {
            if (failIfNotExist)
            {
                throw new UnknownMetadataException("Unknown metadata " + METADATA_PREFIX + metadataName + " of " + _node, e);
            }
            else
            {
                return defaultValue;
            }
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get boolean property: " + METADATA_PREFIX + metadataName + " of " + _node, e);
        }
    }

    public boolean[] getBooleanArray(String metadataName) throws AmetysRepositoryException
    {
        return _getBooleanArray(metadataName, null, true);
    }

    public boolean[] getBooleanArray(String metadataName, boolean[] defaultValues)
    {
        return _getBooleanArray(metadataName, defaultValues, false);
    }

    private boolean[] _getBooleanArray(String metadataName, boolean[] defaultValues, boolean failIfNotExist)
    {
        try
        {
            Property property = _node.getProperty(METADATA_PREFIX + metadataName);

            if (property.getDefinition().isMultiple())
            {
                Value[] values = property.getValues();

                boolean[] results = new boolean[values.length];
                for (int i = 0; i < values.length; i++)
                {
                    Value value = values[i];
                    results[i] = value.getBoolean();
                }

                return results;
            }

            Value value = property.getValue();
            return new boolean[] {value.getBoolean()};
        }
        catch (PathNotFoundException e)
        {
            if (failIfNotExist)
            {
                throw new UnknownMetadataException("Unknown metadata " + METADATA_PREFIX + metadataName + " of " + _node, e);
            }
            else
            {
                return defaultValues;
            }
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get boolean array property: " + METADATA_PREFIX + metadataName + " of " + _node, e);
        }
    }

    public Date getDate(String metadataName) throws AmetysRepositoryException
    {
        return _getDate(metadataName, null, true);
    }

    public Date getDate(String metadataName, Date defaultValue) throws AmetysRepositoryException
    {
        return _getDate(metadataName, defaultValue, false);
    }

    private Date _getDate(String metadataName, Date defaultValue, boolean failIfNotExist)
    {
        try
        {
            Property property = _node.getProperty(METADATA_PREFIX + metadataName);

            if (property.getDefinition().isMultiple())
            {
                Value[] values = property.getValues();

                if (values.length > 0)
                {
                    return values[0].getDate().getTime();
                }
                else
                {
                    throw new AmetysRepositoryException("Cannot get a date from an empty array");
                }
            }
            
            return property.getDate().getTime();
        }
        catch (PathNotFoundException e)
        {
            if (failIfNotExist)
            {
                throw new UnknownMetadataException("Unknown metadata " + METADATA_PREFIX + metadataName + " of " + _node, e);
            }
            else
            {
                return defaultValue;
            }
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get date property: " + METADATA_PREFIX + metadataName + " of " + _node, e);
        }
    }

    public Date[] getDateArray(String metadataName) throws AmetysRepositoryException
    {
        return _getDateArray(metadataName, null, true);
    }

    public Date[] getDateArray(String metadataName, Date[] defaultValues)
    {
        return _getDateArray(metadataName, defaultValues, false);
    }

    private Date[] _getDateArray(String metadataName, Date[] defaultValues, boolean failIfNotExist)
    {
        try
        {
            Property property = _node.getProperty(METADATA_PREFIX + metadataName);

            if (property.getDefinition().isMultiple())
            {
                Value[] values = property.getValues();

                Date[] results = new Date[values.length];
                for (int i = 0; i < values.length; i++)
                {
                    Value value = values[i];
                    results[i] = value.getDate().getTime();
                }

                return results;
            }

            Value value = property.getValue();
            return new Date[] {value.getDate().getTime()};
        }
        catch (PathNotFoundException e)
        {
            if (failIfNotExist)
            {
                throw new UnknownMetadataException("Unknown metadata " + METADATA_PREFIX + metadataName + " of " + _node, e);
            }
            else
            {
                return defaultValues;
            }
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get date array property", e);
        }
    }
    
    public long getLong(String metadataName) throws AmetysRepositoryException
    {
        return _getLong(metadataName, null, true);
    }

    public long getLong(String metadataName, long defaultValue) throws AmetysRepositoryException
    {
        return _getLong(metadataName, defaultValue, false);
    }

    private long _getLong(String metadataName, Long defaultValue, boolean failIfNotExist)
    {
        try
        {
            Property property = _node.getProperty(METADATA_PREFIX + metadataName);

            if (property.getDefinition().isMultiple())
            {
                Value[] values = property.getValues();

                if (values.length > 0)
                {
                    return values[0].getLong();
                }
                else
                {
                    throw new AmetysRepositoryException("Cannot get a long from an empty array");
                }
            }

            return property.getLong();
        }
        catch (PathNotFoundException e)
        {
            if (failIfNotExist)
            {
                throw new UnknownMetadataException("Unknown metadata " + METADATA_PREFIX + metadataName + " of " + _node, e);
            }
            else
            {
                return defaultValue;
            }
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get long property: " + METADATA_PREFIX + metadataName + " of " + _node, e);
        }
    }

    public long[] getLongArray(String metadataName) throws AmetysRepositoryException
    {
        return _getLongArray(metadataName, null, true);
    }

    public long[] getLongArray(String metadataName, long[] defaultValues)
    {
        return _getLongArray(metadataName, defaultValues, false);
    }

    private long[] _getLongArray(String metadataName, long[] defaultValues, boolean failIfNotExist)
    {
        try
        {
            Property property = _node.getProperty(METADATA_PREFIX + metadataName);

            if (property.getDefinition().isMultiple())
            {
                Value[] values = property.getValues();

                long[] results = new long[values.length];
                for (int i = 0; i < values.length; i++)
                {
                    Value value = values[i];
                    results[i] = value.getLong();
                }

                return results;
            }

            Value value = property.getValue();
            return new long[] {value.getLong()};
        }
        catch (PathNotFoundException e)
        {
            if (failIfNotExist)
            {
                throw new UnknownMetadataException("Unknown metadata " + METADATA_PREFIX + metadataName + " of " + _node, e);
            }
            else
            {
                return defaultValues;
            }
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get long array property: " + METADATA_PREFIX + metadataName + " of " + _node, e);
        }
    }

    public double getDouble(String metadataName) throws AmetysRepositoryException
    {
        return _getDouble(metadataName, null, true);
    }

    public double getDouble(String metadataName, double defaultValue) throws AmetysRepositoryException
    {
        return _getDouble(metadataName, defaultValue, false);
    }

    private double _getDouble(String metadataName, Double defaultValue, boolean failIfNotExist)
    {
        try
        {
            Property property = _node.getProperty(METADATA_PREFIX + metadataName);

            if (property.getDefinition().isMultiple())
            {
                Value[] values = property.getValues();

                if (values.length > 0)
                {
                    return values[0].getDouble();
                }
                else
                {
                    throw new AmetysRepositoryException("Cannot get a double from an empty array");
                }
            }

            return property.getDouble();
        }
        catch (PathNotFoundException e)
        {
            if (failIfNotExist)
            {
                throw new UnknownMetadataException("Unknown metadata " + METADATA_PREFIX + metadataName + " of " + _node, e);
            }
            else
            {
                return defaultValue;
            }
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get double property: " + METADATA_PREFIX + metadataName + " of " + _node, e);
        }
    }

    public double[] getDoubleArray(String metadataName) throws AmetysRepositoryException
    {
        return _getDoubleArray(metadataName, null, true);
    }

    public double[] getDoubleArray(String metadataName, double[] defaultValues)
    {
        return _getDoubleArray(metadataName, defaultValues, false);
    }

    private double[] _getDoubleArray(String metadataName, double[] defaultValues, boolean failIfNotExist)
    {
        try
        {
            Property property = _node.getProperty(METADATA_PREFIX + metadataName);

            if (property.getDefinition().isMultiple())
            {
                Value[] values = property.getValues();

                double[] results = new double[values.length];
                for (int i = 0; i < values.length; i++)
                {
                    Value value = values[i];
                    results[i] = value.getDouble();
                }

                return results;
            }

            Value value = property.getValue();
            return new double[] {value.getDouble()};
        }
        catch (PathNotFoundException e)
        {
            if (failIfNotExist)
            {
                throw new UnknownMetadataException("Unknown metadata " + METADATA_PREFIX + metadataName + " of " + _node, e);
            }
            else
            {
                return defaultValues;
            }
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get double array property: " + METADATA_PREFIX + metadataName + " of " + _node, e);
        }
    }
    
    public UserIdentity getUser(String metadataName) throws AmetysRepositoryException
    {
        return _getUser(metadataName, null, true);
    }
    
    public UserIdentity getUser(String metadataName, UserIdentity defaultValue) throws AmetysRepositoryException
    {
        return _getUser(metadataName, defaultValue, false);
    }
    
    private UserIdentity _getUser(String metadataName, UserIdentity defaultValue, boolean failIfNotExist) throws AmetysRepositoryException
    {
        try
        {
            Node userNode = _node.getNode(METADATA_PREFIX + metadataName);
            
            String login = userNode.getProperty("ametys:login").getString();
            String population = userNode.getProperty("ametys:population").getString();
            
            if (StringUtils.isEmpty(login) || StringUtils.isEmpty(population))
            {
                if (failIfNotExist)
                {
                    throw new UnknownMetadataException("Unknown metadata " + METADATA_PREFIX + metadataName + " of " + _node);
                }
                else
                {
                    return defaultValue;
                }
            }
            
            return new UserIdentity(login, population);
        }
        catch (PathNotFoundException e)
        {
            if (failIfNotExist)
            {
                throw new UnknownMetadataException("Unknown metadata " + METADATA_PREFIX + metadataName + " of " + _node, e);
            }
            else
            {
                return defaultValue;
            }
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get User property: " + METADATA_PREFIX + metadataName + " of " + _node, e);
        }
    }
    
    @Override
    public UserIdentity[] getUserArray(String metadataName) throws AmetysRepositoryException
    {
        return _getUserArray(metadataName, null, true);
    }
    
    @Override
    public UserIdentity[] getUserArray(String metadataName, UserIdentity[] defaultValues) throws AmetysRepositoryException
    {
        return _getUserArray(metadataName, defaultValues, false);
    }
    
    private UserIdentity[] _getUserArray(String metadataName, UserIdentity[] defaultValues, boolean failIfNotExist) throws AmetysRepositoryException
    {
        try
        {
            if (_node.hasNode(METADATA_PREFIX + metadataName))
            {
                List<UserIdentity> results = new ArrayList<>();
                
                if (NodeTypeHelper.isNodeType(_node.getNode(METADATA_PREFIX + metadataName), RepositoryConstants.USER_NODETYPE))
                {
                    NodeIterator it = _node.getNodes(METADATA_PREFIX + metadataName);
                    while (it.hasNext())
                    {
                        Node next = (Node) it.next();
                        
                        String login = next.getProperty("ametys:login").getString();
                        String population = next.getProperty("ametys:population").getString();
                        
                        if (StringUtils.isEmpty(login) || StringUtils.isEmpty(population))
                        {
                            if (failIfNotExist)
                            {
                                throw new UnknownMetadataException("Unknown metadata " + METADATA_PREFIX + metadataName + " of " + _node);
                            }
                            else
                            {
                                return defaultValues;
                            }
                        }

                        results.add(new UserIdentity(login, population));
                    }
                }
                else if (NodeTypeHelper.isNodeType(_node.getNode(METADATA_PREFIX + metadataName), RepositoryConstants.MULTIPLE_ITEM_NODETYPE))
                {
                    // The new API setted the values, with the multiple item node
                    Node multipleItemNode = _node.getNode(METADATA_PREFIX + metadataName);
                    NodeIterator it = multipleItemNode.getNodes(METADATA_PREFIX + "*");

                    while (it.hasNext())
                    {
                        Node next = it.nextNode();
                        if (NodeTypeHelper.isNodeType(next, RepositoryConstants.USER_NODETYPE))
                        {
                            String login = next.getProperty("ametys:login").getString();
                            String population = next.getProperty("ametys:population").getString();
                            
                            if (StringUtils.isEmpty(login) || StringUtils.isEmpty(population))
                            {
                                if (failIfNotExist)
                                {
                                    throw new UnknownMetadataException("Unknown metadata " + METADATA_PREFIX + metadataName + " of " + _node);
                                }
                                else
                                {
                                    return defaultValues;
                                }
                            }

                            results.add(new UserIdentity(login, population));
                        }
                    }
                }
                
                return results.toArray(new UserIdentity[results.size()]);
            }
            else if (failIfNotExist)
            {
                throw new UnknownMetadataException("Unknown metadata " + METADATA_PREFIX + metadataName + " of " + _node);
            }
            else
            {
                return defaultValues;
            }
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get User array property: " + METADATA_PREFIX + metadataName + " of " + _node, e);
        }
    }

    @Override
    public void setMetadata(String metadataName, String value) throws AmetysRepositoryException
    {
        if (value == null)
        {
            throw new NullPointerException("metadata value cannot be null");
        }

        _checkMetadataName(metadataName);

        try
        {
            _checkLock();
            _node.setProperty(METADATA_PREFIX + metadataName, value);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to set property", e);
        }
    }

    @Override
    public void setMetadata(String metadataName, String[] value) throws AmetysRepositoryException
    {
        if (value == null)
        {
            throw new NullPointerException("metadata value cannot be null");
        }

        _checkMetadataName(metadataName);

        try
        {
            _checkLock();
            _node.setProperty(METADATA_PREFIX + metadataName, value);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to set property", e);
        }
    }
    
    @Override
    public void setMetadata(String metadataName, String value, Locale locale) throws AmetysRepositoryException
    {
        if (value == null)
        {
            throw new NullPointerException("metadata value cannot be null");
        }

        _checkMetadataName(metadataName);

        try
        {
            _checkLock();
            
            Node node = null;
            if (_node.hasNode(METADATA_PREFIX + metadataName))
            {
                node = _node.getNode(METADATA_PREFIX + metadataName);
            }
            else
            {
                node = _node.addNode(METADATA_PREFIX + metadataName, MULTILINGUAL_STRING_METADATA_NODETYPE);
            }
            
            node.setProperty(METADATA_PREFIX + locale.toString(), value);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to set property", e);
        }
    }

    @Override
    public void setMetadata(String metadataName, Date value) throws AmetysRepositoryException
    {
        if (value == null)
        {
            throw new NullPointerException("metadata value cannot be null");
        }

        GregorianCalendar calendar = GregorianCalendar.from(Instant.ofEpochMilli(value.getTime()).atZone(ZoneOffset.UTC));

        _checkMetadataName(metadataName);

        try
        {
            _checkLock();
            _node.setProperty(METADATA_PREFIX + metadataName, calendar);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to set property", e);
        }
    }

    @Override
    public void setMetadata(String metadataName, Date[] values) throws AmetysRepositoryException
    {
        if (values == null)
        {
            throw new NullPointerException("metadata value cannot be null");
        }

        _checkMetadataName(metadataName);

        try
        {
            _checkLock();
            Value[] jcrValues = new Value[values.length];

            for (int i = 0; i < values.length; i++)
            {
                Date value = values[i];

                GregorianCalendar calendar = GregorianCalendar.from(Instant.ofEpochMilli(value.getTime()).atZone(ZoneOffset.UTC));

                jcrValues[i] = _node.getSession().getValueFactory().createValue(calendar);
            }

            _node.setProperty(METADATA_PREFIX + metadataName, jcrValues);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to set property", e);
        }
    }

    @Override
    public void setMetadata(String metadataName, long value) throws AmetysRepositoryException
    {
        _checkMetadataName(metadataName);

        try
        {
            _checkLock();
            _node.setProperty(METADATA_PREFIX + metadataName, value);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to set property", e);
        }
    }

    @Override
    public void setMetadata(String metadataName, double value) throws AmetysRepositoryException
    {
        _checkMetadataName(metadataName);

        try
        {
            _checkLock();
            _node.setProperty(METADATA_PREFIX + metadataName, value);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to set property", e);
        }
    }

    @Override
    public void setMetadata(String metadataName, long[] values) throws AmetysRepositoryException
    {
        if (values == null)
        {
            throw new NullPointerException("metadata value cannot be null");
        }

        _checkMetadataName(metadataName);

        try
        {
            _checkLock();
            Value[] jcrValues = new Value[values.length];

            for (int i = 0; i < values.length; i++)
            {
                long value = values[i];
                jcrValues[i] = _node.getSession().getValueFactory().createValue(value);
            }

            _node.setProperty(METADATA_PREFIX + metadataName, jcrValues);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to set property", e);
        }
    }

    @Override
    public void setMetadata(String metadataName, double[] values) throws AmetysRepositoryException
    {
        if (values == null)
        {
            throw new NullPointerException("metadata value cannot be null");
        }

        _checkMetadataName(metadataName);

        try
        {
            _checkLock();
            Value[] jcrValues = new Value[values.length];

            for (int i = 0; i < values.length; i++)
            {
                double value = values[i];
                jcrValues[i] = _node.getSession().getValueFactory().createValue(value);
            }

            _node.setProperty(METADATA_PREFIX + metadataName, jcrValues);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to set property", e);
        }
    }

    @Override
    public void setMetadata(String metadataName, boolean value) throws AmetysRepositoryException
    {
        _checkMetadataName(metadataName);

        try
        {
            _checkLock();
            _node.setProperty(METADATA_PREFIX + metadataName, value);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to set property", e);
        }
    }
    
    @Override
    public void setMetadata(String metadataName, boolean[] values) throws AmetysRepositoryException
    {
        if (values == null)
        {
            throw new NullPointerException("metadata value cannot be null");
        }

        _checkMetadataName(metadataName);

        try
        {
            _checkLock();
            Value[] jcrValues = new Value[values.length];

            for (int i = 0; i < values.length; i++)
            {
                boolean value = values[i];
                jcrValues[i] = _node.getSession().getValueFactory().createValue(value);
            }

            _node.setProperty(METADATA_PREFIX + metadataName, jcrValues);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to set property", e);
        }
    }
    
    @Override
    public void setMetadata(String metadataName, UserIdentity value) throws AmetysRepositoryException
    {
        _checkMetadataName(metadataName);
        
        try
        {
            _checkLock();
            
            Node userNode = null;
            if (_node.hasNode(METADATA_PREFIX + metadataName))
            {
                userNode = _node.getNode(METADATA_PREFIX + metadataName);
            }
            else
            {
                userNode = _node.addNode(METADATA_PREFIX + metadataName, RepositoryConstants.USER_NODETYPE);
            }
            userNode.setProperty(METADATA_PREFIX + "login", value.getLogin());
            userNode.setProperty(METADATA_PREFIX + "population", value.getPopulationId());
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to set property", e);
        }
    }
    
    @Override
    public void setMetadata(String metadataName, UserIdentity[] values) throws AmetysRepositoryException
    {
        if (values == null)
        {
            throw new NullPointerException("metadata value cannot be null");
        }

        _checkMetadataName(metadataName);

        try
        {
            _checkLock();
            
            NodeIterator it = _node.getNodes(METADATA_PREFIX + metadataName);
            while (it.hasNext())
            {
                Node next = (Node) it.next();
                next.remove();
            }
            
            for (UserIdentity value : values)
            {
                Node userNode = _node.addNode(METADATA_PREFIX + metadataName, RepositoryConstants.USER_NODETYPE);
                userNode.setProperty(METADATA_PREFIX + "login", value.getLogin());
                userNode.setProperty(METADATA_PREFIX + "population", value.getPopulationId());
            }
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to set property", e);
        }
    }
    
    @Override
    public void setMetadata(String metadataName, MultilingualString value) throws AmetysRepositoryException
    {
        if (value == null)
        {
            throw new NullPointerException("metadata value cannot be null");
        }
        
        _checkMetadataName(metadataName);

        try
        {
            _checkLock();
            
            NodeIterator it = _node.getNodes(METADATA_PREFIX + metadataName);
            while (it.hasNext())
            {
                Node next = (Node) it.next();
                next.remove();
            }
            
            Node node = null;
            if (_node.hasNode(METADATA_PREFIX + metadataName))
            {
                node = _node.getNode(METADATA_PREFIX + metadataName);
            }
            else
            {
                node = _node.addNode(METADATA_PREFIX + metadataName, MULTILINGUAL_STRING_METADATA_NODETYPE);
            }
            
            for (Locale locale : value.getLocales())
            {
                node.setProperty(METADATA_PREFIX + locale.toString(), value.getValue(locale));
            }
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to set property", e);
        }
    }

    public String[] getMetadataNames() throws AmetysRepositoryException
    {
        try
        {
            PropertyIterator itProp = _node.getProperties(METADATA_PREFIX + "*");
            NodeIterator itNode = _node.getNodes(METADATA_PREFIX + "*");

            Set<String> names = new HashSet<>();

            while (itProp.hasNext())
            {
                Property property = itProp.nextProperty();
                names.add(property.getName().substring(METADATA_PREFIX.length()));
            }

            while (itNode.hasNext())
            {
                Node node = itNode.nextNode();
                
                // Filtering out AmetysObjects: they're not metadata (except object collections and users).
                if (!NodeTypeHelper.isNodeType(node, AmetysObjectResolver.OBJECT_TYPE) || NodeTypeHelper.isNodeType(node, OBJECT_COLLECTION_METADATA_NODETYPE) || NodeTypeHelper.isNodeType(node, RepositoryConstants.USER_NODETYPE) || NodeTypeHelper.isNodeType(node, MULTILINGUAL_STRING_METADATA_NODETYPE))
                {
                    names.add(node.getName().substring(METADATA_PREFIX.length()));
                }
            }

            String[] result = new String[names.size()];
            names.toArray(result);

            return result;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get metadata names", e);
        }
    }

    public boolean isMultiple(String metadataName) throws AmetysRepositoryException
    {
        try
        {
            String realMetadataName = METADATA_PREFIX + metadataName;
            
            if (_node.hasProperty(realMetadataName))
            {
                Property property = _node.getProperty(realMetadataName);
    
                if (property.getDefinition().isMultiple())
                {
                    return true;
                }
    
                return false;
            }
            
            if (_node.hasNode(realMetadataName))
            {
                String nodeType = NodeTypeHelper.getNodeTypeName(_node.getNode(realMetadataName));
                
                // Object collection is multi-valued.
                if (OBJECT_COLLECTION_METADATA_NODETYPE.equals(nodeType))
                {
                    return true;
                }
                
                // User metadata
                if (nodeType.equals(METADATA_PREFIX + "user"))
                {
                    return _node.getNodes(realMetadataName).getSize() > 1;
                }
                
                // All other complex metadata are single valued.
                return false;
            }
            
            throw new UnknownMetadataException("No metadata for name: " + metadataName);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to determine if metadata " + metadataName + " is multiple", e);
        }
    }

    /**
     * Check if the metadata name is valid.
     * @param metadataName The metadata name to check.
     * @throws AmetysRepositoryException If the metadata name is invalid.
     */
    protected void _checkMetadataName(String metadataName) throws AmetysRepositoryException
    {
        // Id null
        if (metadataName == null)
        {
            throw new AmetysRepositoryException("Invalid metadata name (null)");
        }

        // Id valide
        if (!_getMetadataNamePattern().matcher(metadataName).matches())
        {
            throw new AmetysRepositoryException("Invalid metadata name '" + metadataName + "'. This value is not permitted: only [a-zA-Z][a-zA-Z0-9-_]* are allowed.");
        }
    }

    /**
     * Get the metadata name pattern to test validity.
     * @return The metadata name pattern.
     */
    protected static Pattern _getMetadataNamePattern()
    {
        if (__metadataNamePattern == null)
        {
            // [a-zA-Z][a-zA-Z0-9_]*
            __metadataNamePattern = Pattern.compile("[a-z][a-z0-9-_]*", Pattern.CASE_INSENSITIVE);
        }

        return __metadataNamePattern;
    }

    public ModifiableBinaryMetadata getBinaryMetadata(String metadataName) throws AmetysRepositoryException
    {
        return getBinaryMetadata(metadataName, false);
    }
    
    public ModifiableBinaryMetadata getBinaryMetadata(String metadataName, boolean createNew) throws AmetysRepositoryException
    {
        try
        {
            if (_node.hasNode(METADATA_PREFIX + metadataName))
            {
                Node node = _node.getNode(METADATA_PREFIX + metadataName);
                return new JCRBinaryMetadata(node);
            }
            else if (createNew)
            {
                _checkLock();
                Node node = _node.addNode(METADATA_PREFIX + metadataName, RepositoryConstants.BINARY_NODETYPE);
                return new JCRBinaryMetadata(node);
            }
            else
            {
                throw new UnknownMetadataException("No BinaryMetadata named " + metadataName);
            }
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException(e);
        }
    }

    public ModifiableCompositeMetadata getCompositeMetadata(String metadataName) throws AmetysRepositoryException
    {
        return getCompositeMetadata(metadataName, false);
    }
    
    public ModifiableCompositeMetadata getCompositeMetadata(String metadataName, boolean createNew) throws AmetysRepositoryException
    {
        try
        {
            if (_node.hasNode(METADATA_PREFIX + metadataName))
            {
                Node node = _node.getNode(METADATA_PREFIX + metadataName);
                return new JCRCompositeMetadata(node, _resolver);
            }
            else if (createNew)
            {
                _checkLock();
                Node node = _node.addNode(METADATA_PREFIX + metadataName, "ametys:compositeMetadata");
                return new JCRCompositeMetadata(node, _resolver);
            }
            else
            {
                throw new UnknownMetadataException("No CompositeMetadata named " + metadataName);
            }
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException(e);
        }
    }
    
    public ModifiableRichText getRichText(String metadataName) throws UnknownMetadataException, AmetysRepositoryException
    {
        return getRichText(metadataName, false);
    }
    
    public ModifiableRichText getRichText(String metadataName, boolean createNew) throws UnknownMetadataException, AmetysRepositoryException
    {
        try
        {
            if (_node.hasNode(METADATA_PREFIX + metadataName))
            {
                Node node = _node.getNode(METADATA_PREFIX + metadataName);
                if (node.hasProperty(RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":isEmpty") && node.getProperty(RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":isEmpty").getBoolean())
                {
                    // new API may write empty RichText
                    throw new UnknownMetadataException("No RichText named " + metadataName);
                }
                
                return new JCRRichText(node, _resolver);
            }
            else if (createNew)
            {
                _checkLock();
                Node node = _node.addNode(METADATA_PREFIX + metadataName, METADATA_PREFIX + "richText");
                return new JCRRichText(node, _resolver);
            }
            else
            {
                throw new UnknownMetadataException("No RichText named " + metadataName);
            }
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException(e);
        }
    }
    
    public ModifiableTraversableAmetysObject getObjectCollection(String metadataName) throws AmetysRepositoryException
    {
        return getObjectCollection(metadataName, false);
    }
    
    public ModifiableTraversableAmetysObject getObjectCollection(String metadataName, boolean createNew) throws AmetysRepositoryException
    {
        try
        {
            if (_node.hasNode(METADATA_PREFIX + metadataName))
            {
                Node node = _node.getNode(METADATA_PREFIX + metadataName);
                return _resolver.resolve(node, false);
            }
            else if (createNew)
            {
                _checkLock();
                Node node = _node.addNode(METADATA_PREFIX + metadataName, OBJECT_COLLECTION_METADATA_NODETYPE);
                return _resolver.resolve(node, false);
            }
            else
            {
                throw new UnknownMetadataException("No object collection metadata named " + metadataName);
            }
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException(e);
        }
    }
    
    private void _checkLock() throws RepositoryException
    {
        if (!_lockAlreadyChecked && _node.isLocked())
        {
            LockManager lockManager = _node.getSession().getWorkspace().getLockManager();
            
            Lock lock = lockManager.getLock(_node.getPath());
            Node lockHolder = lock.getNode();
            
            lockManager.addLockToken(lockHolder.getProperty(RepositoryConstants.METADATA_LOCKTOKEN).getString());
            _lockAlreadyChecked = true;
        }
    }
}
