/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.metadata.jcr;

import java.util.ArrayList;
import java.util.Collection;

import javax.jcr.ItemExistsException;
import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.PathNotFoundException;
import javax.jcr.RepositoryException;
import javax.jcr.lock.Lock;

import org.apache.jackrabbit.JcrConstants;
import org.apache.jackrabbit.util.Text;

import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.RepositoryIntegrityViolationException;
import org.ametys.plugins.repository.jcr.NodeTypeHelper;
import org.ametys.plugins.repository.metadata.ModifiableFile;
import org.ametys.plugins.repository.metadata.ModifiableFolder;
import org.ametys.plugins.repository.metadata.UnknownMetadataException;

/**
 * Java binding of a nt:folder JCR Node.<p>
 * {@link Text#escapeIllegalJcrChars(String)} is used for
 * escaping invalid JCR characters or character sequences.
 * @deprecated org.ametys.cms.data.RichText new class doesn't use folder anymore
 */
@Deprecated
public class JCRFolder implements ModifiableFolder
{
    private Node _node;
    private boolean _lockAlreadyChecked;
    
    /**
     * Constructor
     * @param node the nt:folder Node
     */
    public JCRFolder(Node node)
    {
        _node = node;
    }
    
    /**
     * Retrieves the underlying node.
     * @return the underlying node.
     */
    public Node getNode()
    {
        return _node;
    }
    
    @Override
    public String getName() throws AmetysRepositoryException
    {
        try
        {
            return Text.unescapeIllegalJcrChars(_node.getName());
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException(e);
        }
    }

    public Collection<ModifiableFolder> getFolders()
    {
        ArrayList<ModifiableFolder> folders = new ArrayList<>();
        try
        {
            NodeIterator it = _node.getNodes();
            while (it.hasNext())
            {
                Node node = it.nextNode();
                
                if (NodeTypeHelper.isNodeType(node, JcrConstants.NT_FOLDER))
                {
                    folders.add(new JCRFolder(node));
                }
            }
            
            return folders;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException(e);
        }
    }

    public ModifiableFolder getFolder(String folderName)
    {
        try
        {
            Node node = _node.getNode(Text.escapeIllegalJcrChars(folderName));
            return new JCRFolder(node);
        }
        catch (PathNotFoundException e)
        {
            throw new UnknownMetadataException(e);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException(e);
        }
    }
    
    public ModifiableFolder addFolder(String folderName)
    {
        try
        {
            _checkLock();
            Node node = _node.addNode(Text.escapeIllegalJcrChars(folderName), JcrConstants.NT_FOLDER);
            return new JCRFolder(node);
        }
        catch (ItemExistsException e)
        {
            throw new RepositoryIntegrityViolationException(e);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException(e);
        }
    }
    
    public Collection<ModifiableFile> getFiles()
    {
        ArrayList<ModifiableFile> files = new ArrayList<>();
        try
        {
            NodeIterator it = _node.getNodes();
            while (it.hasNext())
            {
                Node node = it.nextNode();
                
                if (NodeTypeHelper.isNodeType(node, JcrConstants.NT_FILE))
                {
                    files.add(new JCRFile(node));
                }
            }
            
            return files;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException(e);
        }
    }
    
    @Override
    public boolean hasFile(String fileName) throws AmetysRepositoryException
    {
        try
        {
            return _node.hasNode(Text.escapeIllegalJcrChars(fileName));
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException(e);
        }
    }

    public ModifiableFile getFile(String fileName)
    {
        try
        {
            Node node = _node.getNode(Text.escapeIllegalJcrChars(fileName));
            return new JCRFile(node);
        }
        catch (PathNotFoundException e)
        {
            throw new UnknownMetadataException(e);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException(e);
        }
    }
    
    public ModifiableFile addFile(String fileName)
    {
        try
        {
            _checkLock();
            Node node = _node.addNode(Text.escapeIllegalJcrChars(fileName), JcrConstants.NT_FILE);
            return new JCRFile(node);
        }
        catch (ItemExistsException e)
        {
            throw new RepositoryIntegrityViolationException(e);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException(e);
        }
    }

    @Override
    public void remove(String name) throws AmetysRepositoryException
    {
        try
        {
            _checkLock();
            Node node = _node.getNode(Text.escapeIllegalJcrChars(name));
            node.remove();
        }
        catch (PathNotFoundException e)
        {
            throw new UnknownMetadataException(e);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException(e);
        }
    }

    @Override
    public void removeAll() throws AmetysRepositoryException
    {
        try
        {
            _checkLock();
            NodeIterator it = _node.getNodes();
            while (it.hasNext())
            {
                Node node = it.nextNode();
                node.remove();
            }
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException(e);
        }
    }
    
    private void _checkLock() throws RepositoryException
    {
        if (!_lockAlreadyChecked && _node.isLocked())
        {
            Lock lock = _node.getSession().getWorkspace().getLockManager().getLock(_node.getPath());
            Node lockHolder = lock.getNode();
            
            lockHolder.getSession().getWorkspace().getLockManager().addLockToken(lockHolder.getProperty(RepositoryConstants.METADATA_LOCKTOKEN).getString());
            _lockAlreadyChecked = true;
        }
    }
}
