/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.metadata.jcr;

import java.io.InputStream;
import java.io.OutputStream;
import java.util.Date;
import java.util.GregorianCalendar;

import javax.jcr.Binary;
import javax.jcr.Node;
import javax.jcr.PathNotFoundException;
import javax.jcr.RepositoryException;
import javax.jcr.lock.Lock;

import org.apache.jackrabbit.JcrConstants;

import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.metadata.ModifiableResource;

/**
 * Java binding of a nt:resource JCR Node
 * @deprecated Use org.ametys.cms.data.Resource instead
 */
@Deprecated
public class JCRResource implements ModifiableResource
{
    private Node _node;
    private boolean _lockAlreadyChecked;
    
    /**
     * Constructor
     * @param node the nt:resource Node
     */
    public JCRResource(Node node)
    {
        _node = node;
    }
    
    /**
     * Retrieves the underlying node.
     * @return the underlying node.
     */
    public Node getNode()
    {
        return _node;
    }
    
    @Override
    public void rename(String newName) throws AmetysRepositoryException
    {
        try
        {
            _checkLock();
            getNode().getSession().move(getNode().getPath(), getNode().getParent().getPath() + "/" + newName);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException(e);
        }
    }

    public String getEncoding() throws AmetysRepositoryException
    {
        try
        {
            return _node.getProperty(JcrConstants.JCR_ENCODING).getString();
        }
        catch (PathNotFoundException e)
        {
            // Not found
            return null;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException(e);
        }
    }

    public InputStream getInputStream() throws AmetysRepositoryException
    {
        try
        {
            return _node.getProperty(JcrConstants.JCR_DATA).getBinary().getStream();
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException(e);
        }
    }

    public Date getLastModified() throws AmetysRepositoryException
    {
        try
        {
            return _node.getProperty(JcrConstants.JCR_LASTMODIFIED).getDate().getTime();
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException(e);
        }
    }

    public long getLength() throws AmetysRepositoryException
    {
        try
        {
            return _node.getProperty(JcrConstants.JCR_DATA).getLength();
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException(e);
        }
    }

    public String getMimeType() throws AmetysRepositoryException
    {
        try
        {
            return _node.getProperty(JcrConstants.JCR_MIMETYPE).getString();
        }

        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException(e);
        }
    }

    public void setEncoding(String encoding) throws AmetysRepositoryException
    {
        try
        {
            _checkLock();
            _node.setProperty(JcrConstants.JCR_ENCODING, encoding);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException(e);
        }
    }

    public void setInputStream(InputStream stream) throws AmetysRepositoryException
    {
        try
        {
            _checkLock();
            Binary binary = _node.getSession().getValueFactory().createBinary(stream);
            _node.setProperty(JcrConstants.JCR_DATA, binary);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException(e);
        }
    }

    public void setLastModified(Date lastModifiedDate) throws AmetysRepositoryException
    {
        GregorianCalendar gc = new GregorianCalendar();
        gc.setTime(lastModifiedDate);
        
        try
        {
            _checkLock();
            _node.setProperty(JcrConstants.JCR_LASTMODIFIED, gc);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException(e);
        }
    }

    public void setMimeType(String mimeType) throws AmetysRepositoryException
    {
        try
        {
            _checkLock();
            _node.setProperty(JcrConstants.JCR_MIMETYPE, mimeType);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException(e);
        }
    }
    
    public OutputStream getOutputStream() throws AmetysRepositoryException
    {
        return new JCROutputStream(_node, false);
    }
    
    /**
     * Check if the node is locked
     * @throws RepositoryException if an error occurred
     */
    protected void _checkLock() throws RepositoryException
    {
        if (!_lockAlreadyChecked && _node.isLocked())
        {
            Lock lock = _node.getSession().getWorkspace().getLockManager().getLock(_node.getPath());
            Node lockHolder = lock.getNode();
            
            lockHolder.getSession().getWorkspace().getLockManager().addLockToken(lockHolder.getProperty(RepositoryConstants.METADATA_LOCKTOKEN).getString());
            _lockAlreadyChecked = true;
        }
    }
}
