/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.migration.jcr;

import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.core.migration.MigrationEngine.MigrationComponent;
import org.ametys.core.migration.MigrationException;
import org.ametys.core.migration.MigrationExtensionPoint;
import org.ametys.core.migration.NotMigrableInSafeModeException;
import org.ametys.core.migration.version.Version;
import org.ametys.core.migration.version.VersionConfiguration;
import org.ametys.core.migration.version.storage.VersionStorage;
import org.ametys.core.util.DateUtils;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.RepositoryIntegrityViolationException;
import org.ametys.plugins.repository.migration.jcr.repository.VersionAmetysObject;
import org.ametys.plugins.repository.migration.jcr.repository.VersionComponentAmetysObject;
import org.ametys.plugins.repository.migration.jcr.repository.VersionComponentFactory;
import org.ametys.plugins.repository.migration.jcr.repository.VersionFactory;
import org.ametys.plugins.repository.migration.jcr.repository.VersionsAmetysObject;
import org.ametys.plugins.repository.query.QueryHelper;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.runtime.plugin.component.PluginAware;

/**
 * JCR implementation of {@link VersionStorage}, to manage the list of versions in a repository
 */
public class JcrVersionStorage extends AbstractLogEnabled implements VersionStorage, Serviceable, PluginAware
{
    /** The Ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    /** The migration extension point */
    protected MigrationExtensionPoint _migrationEP;

    /** The versions root helper */
    protected VersionsRootHelper _versionsRootHelper;

    private String _id;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _migrationEP = (MigrationExtensionPoint) manager.lookup(MigrationExtensionPoint.ROLE);
        _versionsRootHelper = (VersionsRootHelper) manager.lookup(VersionsRootHelper.ROLE);
    }
    
    public void setPluginInfo(String pluginName, String featureName, String id)
    {
        _id = id;
    }
    
    public String getId()
    {
        return _id;
    }
    
    public List<Version> getVersions(String componentId, VersionConfiguration configuration, MigrationComponent component) throws MigrationException
    {
        // The versions node doesn't exist (repository from an older version than first migration implementations)
        if (!_versionsRootHelper.hasRootObject())
        {
            throw new MigrationException("Your existing repository does not support automatic migrations (ametys:versions does not exist). Please follow the migration guide.");
        }
        
        Expression expression = new StringExpression(VersionAmetysObject.COMPONENT_ID, Operator.EQ, componentId);
        
        String query = QueryHelper.getXPathQuery(null, VersionFactory.VERSION_NODETYPE, expression);
        return _resolver.<VersionAmetysObject>query(query)
                        .stream()
                        .map(ao -> new Version(component, componentId, this, configuration, ao.getVersionNumber(), ao.getExecutionDate().toInstant(), ao.getComment()))
                        .toList();
    }
    
    public VersionConfiguration createVersionConfiguration(Configuration versionConfiguration) throws ConfigurationException, NotMigrableInSafeModeException
    {
        return null;
    }
    
    public void storeVersion(Version version) throws MigrationException
    {
        VersionComponentAmetysObject component = _getComponentObject(version.getComponentId(), true);
        
        try
        {
            VersionAmetysObject versionAO = component.createChild("v-" + version.getVersionNumber(), VersionFactory.VERSION_NODETYPE);
            versionAO.setComponentId(version.getComponentId());
            versionAO.setVersionNumber(version.getVersionNumber());
            versionAO.setExecutionDate(DateUtils.asCalendar(ZonedDateTime.ofInstant(version.getExecutionInstant(), ZoneId.of("UTC"))));
            versionAO.setComment(version.getComment());
            component.saveChanges();
        }
        catch (RepositoryIntegrityViolationException e)
        {
            throw new MigrationException("JCR data version is instable for: " + version.toString(), e);
        }
    }
    
    public Version createVersion(String componentId, MigrationComponent component, VersionConfiguration storageConfiguration, Map<String, Object> additionalValues) throws MigrationException
    {
        Version version = VersionStorage.super.createVersion(componentId, component, storageConfiguration, additionalValues);
        
        // Case 1 - The repository is a new one (first start of the application), or new repository but no migration finished yet
        if (!_versionsRootHelper.hasKnownPlugins())
        {
            getLogger().debug("End create version for component id: " + componentId + ", case 'new repository'");
            return version;
        }
        
        // Case 2 - The plugin of the component is not referenced into the already loaded plugins (the plugin has been newly added)
        String componentPluginName = component.pluginName();
        if (!_versionsRootHelper.isAKnownPlugin(componentPluginName))
        {
            getLogger().debug("End create version for component id: " + componentId + ", case 'new plugin'");
            return version;
        }
        
        // Case 3 - The plugin has already been referenced (so if you don't have the version for it, it's a new component)
        // It the current component has an initialization, execute it, else assume version 0 by default, all updates will be applied
        if (component.initialization() == null)
        {
            version.setVersionNumber("0");
        }
        
        getLogger().debug("End create version for component id: " + componentId + ", case 'already versionned, but without version stored'");
        return version;
    }

    public void removeVersion(String componentId, String versionNumber, VersionConfiguration configuration) throws MigrationException
    {
        VersionComponentAmetysObject component = _getComponentObject(componentId, false);
        if (component != null)
        {
            String nodeName = "v-" + versionNumber;
            if (component.hasChild(nodeName))
            {
                VersionAmetysObject versionAO = component.getChild(nodeName);
                versionAO.remove();
                component.saveChanges();
            }
        }
    }
    
    @Override
    public void removeAllVersions(String componentIdentifier, VersionConfiguration configuration) throws MigrationException
    {
        getLogger().debug("Start remove all version for component: {}", componentIdentifier);
        VersionComponentAmetysObject component = _getComponentObject(componentIdentifier, false);
        if (component != null)
        {
            VersionsAmetysObject parent = component.getParent();
            component.remove();
            parent.saveChanges();
        }
    }
    
    private VersionComponentAmetysObject _getComponentObject(String componentId, boolean create)
    {
        getLogger().debug("Start _getComponentObject for component: {}", componentId);
        // Build expression
        Expression expression = new StringExpression(VersionAmetysObject.COMPONENT_ID, Operator.EQ, componentId);
        
        String query = QueryHelper.getXPathQuery(null, VersionComponentFactory.VERSION_COMPONENT_NODETYPE, expression);
        AmetysObjectIterable<VersionComponentAmetysObject> versionsAO = _resolver.query(query);
        getLogger().debug("End _getComponentObject for component: {}", componentId);
        return versionsAO.stream()
                         .findFirst()
                         .orElseGet(() -> create ? _createComponentObject(componentId) : null);
    }
    
    private VersionComponentAmetysObject _createComponentObject(String componentId)
    {
        getLogger().debug("Start _createComponentObject for component: {}", componentId);
        VersionsAmetysObject versions = _versionsRootHelper.getRootObject();
        
        // Get the child name
        String componentName = "c-"
            + componentId
                .toLowerCase()
                .replaceAll("[^a-z0-9]", " ")
                .trim()
                .replaceAll("[ ]{2,}", " ")
                .replace(" ", "-");
        
        String childName = componentName;
        int errorCount = 0;
        while (versions.hasChild(childName))
        {
            errorCount++;
            childName = componentName + errorCount;
        }
        
        // Create version component node
        VersionComponentAmetysObject versionComponent = versions.createChild(childName, VersionComponentFactory.VERSION_COMPONENT_NODETYPE);
        versionComponent.setComponentId(componentId);
        versions.saveChanges();
        getLogger().debug("End _createComponentObject for component: {}", componentId);
        
        return versionComponent;
    }
}
