/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.migration.jcr;

import java.util.HashSet;
import java.util.Set;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.ModifiableTraversableAmetysObject;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.migration.jcr.repository.VersionsAmetysObject;
import org.ametys.plugins.repository.migration.jcr.repository.VersionsFactory;
import org.ametys.runtime.plugin.PluginsManager;

/**
 * A helper for the root versions object.
 */
public class VersionsRootHelper implements Component, Serviceable
{
    /** Avalon Role */
    public static final String ROLE = VersionsRootHelper.class.getName();
    
    /** The Ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
    }

    /**
     * <code>true</code> if the root object to store versions is present. It's the node /ametys:versions.
     * @return <code>true</code> if the root object to store versions is present, otherwise <code>false</code>.
     * @throws AmetysRepositoryException if an error occurs
     */
    public boolean hasRootObject() throws AmetysRepositoryException
    {
        try
        {
            _resolver.resolveByPath("/" + VersionsFactory.VERSIONS_NODENAME);
            return true;
        }
        catch (UnknownAmetysObjectException e)
        {
            return false;
        }
        catch (AmetysRepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get the versions root node", e);
        }
    }

    /**
     * Get or create the root object to store versions. It's the node /ametys:versions.
     * @return the root object to store versions
     * @throws AmetysRepositoryException if an error occurs
     */
    public VersionsAmetysObject getRootObject() throws AmetysRepositoryException
    {
        try
        {
            ModifiableTraversableAmetysObject rootNode = _resolver.resolveByPath("/");
            if (rootNode.hasChild(VersionsFactory.VERSIONS_NODENAME))
            {
                return rootNode.getChild(VersionsFactory.VERSIONS_NODENAME);
            }
            
            VersionsAmetysObject versionsNode = rootNode.createChild(VersionsFactory.VERSIONS_NODENAME, VersionsFactory.VERSIONS_NODETYPE);
            rootNode.saveChanges();
            return versionsNode;
        }
        catch (AmetysRepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get the versions root node", e);
        }
    }

    /**
     * <code>true</code> if the repository has registered plugins. Read the attribute /ametys:versions/@ametys:knownPlugins.
     * @return <code>true</code> if there is at least one registered plugin
     */
    public boolean hasKnownPlugins()
    {
        return !getRootObject().getKnownPlugins().isEmpty();
    }
    
    /**
     * <code>true</code> if the repository has already been launched with this plugin before. Read the attribute /ametys:versions/@ametys:knownPlugins.
     * @param plugin the name of the plugin
     * @return <code>true</code> if the plugin is already known
     */
    public boolean isAKnownPlugin(String plugin)
    {
        return getRootObject().getKnownPlugins().contains(plugin);
    }

    /**
     * Update the plugin names in the root versions object. Write the attribute /ametys:versions/@ametys:knownPlugins.
     * Make sure not to "forget" plugins that are not here anymore, for example if we are in safe mode, there can be less plugins and that might cause problem if some extensions have failOnOldData.
     */
    public void updateKnownPlugins()
    {
        Set<String> plugins = PluginsManager.getInstance().getPluginNames();
        VersionsAmetysObject root = getRootObject();
        Set<String> knownPlugins = root.getKnownPlugins();
        
        Set<String> fullList = new HashSet<>();
        fullList.addAll(knownPlugins);
        fullList.addAll(plugins);
        
        root.setKnownPlugins(fullList);
        root.saveChanges();
    }
}
