/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.model;

import org.apache.avalon.framework.service.ServiceException;

import org.ametys.plugins.repository.data.type.ModelItemTypeConstants;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.ModelItemGroup;
import org.ametys.runtime.model.exception.UnknownTypeException;
import org.ametys.runtime.model.type.ModelItemType;
import org.ametys.runtime.plugin.ExtensionPoint;

/**
 * Definition of a composite
 */
public class CompositeDefinition extends ModelItemGroup
{
   /**
    * Default constructor.
    */
    public CompositeDefinition()
    {
        super();
    }

    /**
     * Constructor used to create simple models and items 
     * @param name the name of the definition
     * @param type the type of the composite
     * @param children the composite's children
     */
    public CompositeDefinition(String name, ModelItemType type, ModelItem... children)
    {
        super(name, children);
        setType(type);
    }

    /**
     * Creates a {@link CompositeDefinition}
     * @param name the composite's name
     * @param availableTypesExtensionPoint the role of the extension point containing all available types for this {@link ModelItem}
     * @param children the composite's children
     * @return the created {@link CompositeDefinition}
     * @throws UnknownTypeException if the composite type is not available in the extension point
     * @throws ServiceException if an error occurs while getting the extension point of available types
     */
    public static CompositeDefinition of(String name, String availableTypesExtensionPoint, ModelItem... children) throws UnknownTypeException, ServiceException
    {
        @SuppressWarnings("unchecked")
        ExtensionPoint<ModelItemType> availableTypes = (ExtensionPoint<ModelItemType>) __serviceManager.lookup(availableTypesExtensionPoint);
        if (!availableTypes.hasExtension(ModelItemTypeConstants.COMPOSITE_TYPE_ID))
        {
            throw new UnknownTypeException("The type '" + ModelItemTypeConstants.COMPOSITE_TYPE_ID + "' (used for data '" + name + "') is not available for the given extension point.");
        }
        else
        {
            ModelItemType type = availableTypes.getExtension(ModelItemTypeConstants.COMPOSITE_TYPE_ID);
            return new CompositeDefinition(name, type, children);
        }
    }
}
