/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.model;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.ametys.runtime.model.DefinitionAndValue;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.ModelItemGroup;

/**
 * Pair of a value and its definition for a composite
 * @param <T> Type of the root object
 */
public class CompositeDefinitionAndValue<T> extends DefinitionAndValue<T>
{

    private CompositeDefinition _compositeDefinition;
    private List<DefinitionAndValue> _children = new ArrayList<>();
    
    /**
     * Creates a composite definition and value pair
     * @param root the root object of this definition and value pair
     * @param definition the definition
     * @param value the value
     * @param children the children of the definition and value pair
     */
    public CompositeDefinitionAndValue(T root, ModelItem definition, Object value, DefinitionAndValue... children)
    {
        this(root, definition, value, null, children);
    }
    
    /**
     * Creates a composite definition and value pair
     * @param root the root object of this definition and value pair
     * @param definition the definition
     * @param value the value
     * @param parent the parent of the definition and value pair
     * @param children the children of the definition and value pair
     */
    public CompositeDefinitionAndValue(T root, ModelItem definition, Object value, DefinitionAndValue<T> parent, DefinitionAndValue... children)
    {
        super(root, definition, value, parent);
        
        if (definition instanceof CompositeDefinition)
        {
            _compositeDefinition = (CompositeDefinition) definition;
        }
        else
        {
            throw new IllegalArgumentException("The definition of a composite definition and value pair must be a " + CompositeDefinition.class.getName() + ".");
        }
        
        _children = Arrays.asList(children);
    }
    
    @Override
    public ModelItemGroup getDefinition()
    {
        return _compositeDefinition;
    }
    
    /**
     * Retrieves the children definition and value pairs of this composite 
     * @return the children
     */
    public List<DefinitionAndValue> getChildren()
    {
        return _children;
    }
    
    /**
     * Add a definition and value pair as a child of this composite
     * @param definitionAndValue the definition and value pair to add as child
     */
    public void addChild(DefinitionAndValue definitionAndValue)
    {
        _children.add(definitionAndValue);
    }
}
