/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.model;

import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.core.util.XMLUtils;
import org.ametys.plugins.repository.data.type.ModelItemTypeConstants;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.DefinitionContext;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.ModelItemGroup;
import org.ametys.runtime.model.exception.UnknownTypeException;
import org.ametys.runtime.model.type.ModelItemType;
import org.ametys.runtime.plugin.ExtensionPoint;

/**
 * Definition of a repeater.
 */
public class RepeaterDefinition extends ModelItemGroup
{
    /** The widget parameter name for the mode to use for the repeater's modification */
    public static final String MODIFICATION_MODE_PARAM_NAME = "mode";
    /** The widget parameter value for repeater's modification in a panel */
    public static final String MODIFICATION_MODE_PANEL = "panel";
    /** The widget parameter value for repeater's modification in a table */
    public static final String MODIFICATION_MODE_TABLE = "table";
    
    private int _initializeSize;
    private int _minSize;
    private int _maxSize;
    private I18nizableText _addLabel;
    private I18nizableText _delLabel;
    private String _headerLabel;
    
    /**
     * Default constructor.
     */
    public RepeaterDefinition()
    {
        super();
    }
    
    /**
     * Constructor used to create simple models and items 
     * @param name the name of the definition
     * @param type the type of the repeater
     * @param children the repeater's children
     */
    public RepeaterDefinition(String name, ModelItemType type, ModelItem... children)
    {
        super(name, children);
        setType(type);
    }
    
    /**
     * Retrieves the initial size.
     * @return the initial size.
     */
    public int getInitialSize()
    {
        return _initializeSize;
    }
    
    /**
     * Set the initial size.
     * @param size the initial size.
     */
    public void setInitialSize(int size)
    {
        _initializeSize = size;
    }

    /**
     * Retrieves the minimum size.
     * @return the minimum size.
     */
    public int getMinSize()
    {
        return _minSize;
    }
    
    /**
     * Set the minimum size.
     * @param size the minimum size.
     */
    public void setMinSize(int size)
    {
        _minSize = size;
    }
    
    /**
     * Retrieves the maximum size.
     * @return the maximum size or <code>-1</code> if unbounded.
     */
    public int getMaxSize()
    {
        return _maxSize;
    }

    /**
     * Set the maximum size.
     * @param size the maximum size or <code>-1</code> if unbounded.
     */
    public void setMaxSize(int size)
    {
        _maxSize = size;
    }
    
    /**
     * Retrieves the add label.
     * @return the add label or <code>null</code> if none.
     */
    public I18nizableText getAddLabel()
    {
        return _addLabel;
    }

    /**
     * Set the add label.
     * @param label the add label or <code>null</code> if none.
     */
    public void setAddLabel(I18nizableText label)
    {
        _addLabel = label;
    }
    
    /**
     * Retrieves the delete label.
     * @return the delete label or <code>null</code> if none.
     */
    public I18nizableText getDeleteLabel()
    {
        return _delLabel;
    }

    /**
     * Set the delete label.
     * @param label the delete label or <code>null</code> if none.
     */
    public void setDeleteLabel(I18nizableText label)
    {
        _delLabel = label;
    }
    
    /**
     * Get the header label.
     * @return the header label or <code>null</code> if none.
     */
    public String getHeaderLabel()
    {
        return _headerLabel;
    }
    
    /**
     * Set the header label.
     * @param label the header label or <code>null</code> if none.
     */
    public void setHeaderLabel(String label)
    {
        _headerLabel = label;
    }
    
    @Override
    protected Map<String, Object> _toJSON(DefinitionContext context, boolean includeChildren)
    {
        Map<String, Object> result = super._toJSON(context, includeChildren);
        
        result.put("type", getType().getId());
        result.put("add-label", getAddLabel());
        result.put("del-label", getDeleteLabel());
        result.put("header-label", getHeaderLabel());
        result.put("initial-size", getInitialSize());
        result.put("min-size", getMinSize());
        
        if (getMaxSize() >= 0)
        {
            result.put("max-size", getMaxSize());
        }
        
        return result;
    }
    
    @SuppressWarnings("static-access")
    @Override
    public void toSAX(ContentHandler contentHandler, DefinitionContext context) throws SAXException
    {
        super.toSAX(contentHandler, context);
        
        XMLUtils.createElement(contentHandler, "initial-size", String.valueOf(getInitialSize()));
        XMLUtils.createElement(contentHandler, "min-size", String.valueOf(getMinSize()));
        if (getMaxSize() >= 0)
        {
            XMLUtils.createElement(contentHandler, "max-size", String.valueOf(getMaxSize()));
        }
        
        XMLUtils.createI18nElementIfNotNull(contentHandler, "label", getLabel());
        XMLUtils.createI18nElementIfNotNull(contentHandler, "description", getDescription());
        XMLUtils.createElementIfNotNull(contentHandler, "header-label", getHeaderLabel());
    }
    
    /**
     * Creates a {@link RepeaterDefinition}
     * @param name the repeater's name
     * @param availableTypesExtensionPoint the role of the extension point containing all available types for this {@link ModelItem}
     * @param children the repeater's children
     * @return the created {@link RepeaterDefinition}
     * @throws UnknownTypeException if the repeater type is not available in the extension point
     * @throws ServiceException if an error occurs while getting the extension point of available types
     */
    public static RepeaterDefinition of(String name, String availableTypesExtensionPoint, ModelItem... children) throws UnknownTypeException, ServiceException
    {
        @SuppressWarnings("unchecked")
        ExtensionPoint<ModelItemType> availableTypes = (ExtensionPoint<ModelItemType>) __serviceManager.lookup(availableTypesExtensionPoint);
        if (!availableTypes.hasExtension(ModelItemTypeConstants.REPEATER_TYPE_ID))
        {
            throw new UnknownTypeException("The type '" + ModelItemTypeConstants.REPEATER_TYPE_ID + "' (used for data '" + name + "') is not available for the given extension point.");
        }
        else
        {
            ModelItemType type = availableTypes.getExtension(ModelItemTypeConstants.REPEATER_TYPE_ID);
            return new RepeaterDefinition(name, type, children);
        }
    }
}
