/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.model;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.ametys.runtime.model.DefinitionAndValue;
import org.ametys.runtime.model.ModelItem;

/**
 * Pair of a value and its definition for a repeater
 * @param <T> Type of the root object
 */
public class RepeaterDefinitionAndValue<T> extends DefinitionAndValue<T>
{
    private RepeaterDefinition _repeaterDefinition;
    private List<RepeaterEntryDefinitionAndValue> _entries = new ArrayList<>();
    
    /**
     * Creates a repeater definition and value pair
     * @param root the root object of this definition and value pair
     * @param definition the definition
     * @param value the value
     * @param entries the entries of the repeater definition and value pair
     */
    public RepeaterDefinitionAndValue(T root, ModelItem definition, Object value, RepeaterEntryDefinitionAndValue... entries)
    {
        this(root, definition, value, null, entries);
    }
    
    /**
     * Creates a repeater definition and value pair
     * @param root the root object of this definition and value pair
     * @param definition the definition
     * @param value the value
     * @param parent the parent of the definition and value pair
     * @param entries the entries of the repeater definition and value pair
     */
    public RepeaterDefinitionAndValue(T root, ModelItem definition, Object value, DefinitionAndValue<T> parent, RepeaterEntryDefinitionAndValue... entries)
    {
        super(root, definition, value, parent);
        
        if (definition instanceof RepeaterDefinition)
        {
            _repeaterDefinition = (RepeaterDefinition) definition;
        }
        else
        {
            throw new IllegalArgumentException("The definition of a repeater definition and value pair must be a " + RepeaterDefinition.class.getName() + ".");
        }
        
        _entries = Arrays.asList(entries);
    }
    
    @Override
    public RepeaterDefinition getDefinition()
    {
        return _repeaterDefinition;
    }
    
    /**
     * Retrieves the entries definition and value pairs of this repeater 
     * @return the entries
     */
    public List<RepeaterEntryDefinitionAndValue> getEntries()
    {
        return _entries;
    }
    
    /**
     * Add an entry definition and value pair to the repeater
     * @param entryDefinitionAndValue the definition and value pair to add as entry
     */
    public void addEntry(RepeaterEntryDefinitionAndValue entryDefinitionAndValue)
    {
        _entries.add(entryDefinitionAndValue);
    }
}
