/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.model;

import java.util.HashSet;
import java.util.Optional;
import java.util.Set;

import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.data.ametysobject.DataAwareAmetysObject;
import org.ametys.runtime.model.ModelHelper;
import org.ametys.runtime.model.type.DataContext;

/**
 * Object that gives some context for repository data manipulation
 */
public class RepositoryDataContext extends DataContext
{
    private Optional<? extends DataAwareAmetysObject> _object = Optional.empty();
    private Optional<? extends DataAwareAmetysObject> _rootObject = Optional.empty();
    
    private Set<String> _externalizableData = new HashSet<>();
    private boolean _copyExternalMetadata;
    
    /**
     * Creates a new instance of a {@link RepositoryDataContext}
     */
    protected RepositoryDataContext()
    {
        // Empty constructor
    }

    /**
     * Creates a new instance of a {@link RepositoryDataContext} from another {@link DataContext}
     * @param context the data context to copy
     */
    protected RepositoryDataContext(DataContext context)
    {
        super(context);
        if (context instanceof RepositoryDataContext repositoryContext)
        {
            withObject(repositoryContext.getObject().orElse(null));
            _withRootObject(repositoryContext._getRootObject().orElse(null));
            
            withExternalizableData(repositoryContext.getExternalizableData());
            withExternalMetadataInCopy(repositoryContext.copyExternalMetadata());
        }
    }
    
    /**
     * Creates a new instance of a {@link RepositoryDataContext}
     * @return the created instance
     */
    public static RepositoryDataContext newInstance()
    {
        return new RepositoryDataContext();
    }
    
    /**
     * Creates a new instance of a {@link RepositoryDataContext} from another {@link DataContext}.
     * It can be the same implementation or another one, but it will be casted to the current implementation.
     * @param context the data context to copy
     * @return the created instance
     */
    public static RepositoryDataContext newInstance(DataContext context)
    {
        return new RepositoryDataContext(context);
    }
    
    /**
     * Creates a new instance of a {@link RepositoryDataContext}, with the current context values
     * @return the created instance
     */
    @Override
    public RepositoryDataContext cloneContext()
    {
        return newInstance(this);
    }
    
    /**
     * Retrieves the object from which the data path is computed
     * @param <T> Type of the object
     * @return the object
     */
    @SuppressWarnings("unchecked")
    public <T extends DataAwareAmetysObject> Optional<T> getObject()
    {
        return (Optional<T>) _object;
    }
    
    /**
     * Retrieves the identifier of the object
     * @return the object's identifier
     */
    public Optional<String> getObjectId()
    {
        return _object.map(AmetysObject::getId);
    }
    
    /**
     * Set the object from which the data path is computed
     * @param <T> the type of the retrieved {@link DataContext}
     * @param object the object to set
     * @return the current {@link DataContext}
     */
    @SuppressWarnings("unchecked")
    public <T extends RepositoryDataContext> T withObject(DataAwareAmetysObject object)
    {
        _object = Optional.ofNullable(object);
        
        if (_rootObject.isEmpty())
        {
            _withRootObject(object);
        }

        return (T) this;
    }
    
    /**
     * Retrieves the object from which the full data path is computed
     * @param <T> Type of the root object
     * @return the root object
     */
    @SuppressWarnings("unchecked")
    protected <T extends DataAwareAmetysObject> Optional<T> _getRootObject()
    {
        return (Optional<T>) _rootObject;
    }
    
    /**
     * Retrieves the identifier of the root object
     * @return the root object's identifier
     */
    public Optional<String> getRootObjectId()
    {
        return _rootObject.map(AmetysObject::getId);
    }
    
    /**
     * Set the object from which the full data path is computed
     * @param <T> the type of the retrieved {@link DataContext}
     * @param rootObject the root object to set
     * @return the current {@link DataContext}
     */
    @SuppressWarnings("unchecked")
    protected <T extends RepositoryDataContext> T _withRootObject(DataAwareAmetysObject rootObject)
    {
        _rootObject = Optional.ofNullable(rootObject);
        return (T) this;
    }
    
    /**
     * Retrieves the externalizable data
     * @return the externalizable data
     */
    public Set<String> getExternalizableData()
    {
        return _externalizableData;
    }
    
    /**
     * Check if the current data is externalizable
     * @return <code>true</code> if the data is externalizable, <code>false</code> otherwise
     */
    public boolean isDataExternalizable()
    {
        String dataPath = getDataPath();
        String definitionPath = ModelHelper.getDefinitionPathFromDataPath(dataPath);
        return _externalizableData.contains(definitionPath);
    }
    
    /**
     * Sets the externalizable data
     * @param <T> the type of the retrieved {@link DataContext}
     * @param externalizableData the externalizable data to set
     * @return the current {@link RepositoryDataContext}
     */
    @SuppressWarnings("unchecked")
    public <T extends RepositoryDataContext> T withExternalizableData(Set<String> externalizableData)
    {
        _externalizableData = externalizableData;
        return (T) this;
    }
    
    /**
     * Check if the external metadata (alternative and status) should be copied
     * @return <code>true</code> if the external metadata should be copied, <code>false</code> otherwise
     */
    public boolean copyExternalMetadata()
    {
        return _copyExternalMetadata;
    }
    
    /**
     * Set to <code>true</code> to copy the external metadata (alternative and status) (default to <code>false</code>)
     * @param <T> the type of the retrieved {@link DataContext}
     * @param copyExternalMetadata <code>true</code> to copy the external metadata, <code>false</code> otherwise
     * @return the current {@link RepositoryDataContext}
     */
    @SuppressWarnings("unchecked")
    public <T extends RepositoryDataContext> T withExternalMetadataInCopy(boolean copyExternalMetadata)
    {
        _copyExternalMetadata = copyExternalMetadata;
        return (T) this;
    }
}
