/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.model;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;

import org.apache.commons.lang3.StringUtils;

import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.ElementDefinition;
import org.ametys.runtime.model.ModelHelper;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.ModelItemGroup;
import org.ametys.runtime.model.ModelViewItemGroup;
import org.ametys.runtime.model.ViewElement;
import org.ametys.runtime.model.ViewItem;
import org.ametys.runtime.model.ViewItemAccessor;
import org.ametys.runtime.model.ViewItemGroup;
import org.ametys.runtime.parameter.ValidationResult;
import org.ametys.runtime.parameter.ValidationResults;

/**
 * Helper for manipulating views in the context of the repository plugin (aware of repeaters, composites, ...).
 */
public final class ViewHelper
{
    private ViewHelper()
    {
        // Empty constructor
    }
    
    /**
     * Visit a view, allowing to perform specific actions for view elements.
     * @param viewItemAccessor the {@link ViewItemAccessor} to visit.
     * @param elementConsumer the consumer called on each {@link ViewElement}.
     * @param compositeConsumer the consumer called on each item refering to a {@link CompositeDefinition}.
     * @param repeaterConsumer the consumer called on each item refering to a {@link RepeaterDefinition}.
     * @param groupConsumer the consumer called on each other {@link ViewItemGroup}.
     */
    public static void visitView(ViewItemAccessor viewItemAccessor, BiConsumer<ViewElement, ElementDefinition> elementConsumer, BiConsumer<ModelViewItemGroup, CompositeDefinition> compositeConsumer, BiConsumer<ModelViewItemGroup, RepeaterDefinition> repeaterConsumer, Consumer<ViewItemGroup> groupConsumer)
    {
        for (ViewItem viewItem : viewItemAccessor.getViewItems())
        {
            if (viewItem instanceof ViewElement)
            {
                ElementDefinition definition = ((ViewElement) viewItem).getDefinition();
                elementConsumer.accept((ViewElement) viewItem, definition);
            }
            else if (viewItem instanceof ModelViewItemGroup)
            {
                ModelItemGroup modelItemGroup = ((ModelViewItemGroup) viewItem).getDefinition();
                
                if (modelItemGroup instanceof CompositeDefinition)
                {
                    compositeConsumer.accept((ModelViewItemGroup) viewItem, (CompositeDefinition) modelItemGroup);
                }
                else if (modelItemGroup instanceof RepeaterDefinition)
                {
                    repeaterConsumer.accept((ModelViewItemGroup) viewItem, (RepeaterDefinition) modelItemGroup);
                }
            }
            else if (viewItem instanceof ViewItemGroup)
            {
                groupConsumer.accept((ViewItemGroup) viewItem);
            }
        }
    }
    
    /**
     * Validates the given values according to the view item accessor
     * @param values the values to validate
     * @param viewItemAccessor the view item accessor to visit
     * @return the errors information if the validation fails.
     */
    public static ValidationResults validateValues(ViewItemAccessor viewItemAccessor, Optional<Map<String, Object>> values)
    {
        return _validateValues(viewItemAccessor, values, StringUtils.EMPTY);
    }
    
    private static ValidationResults _validateValues(ViewItemAccessor viewItemAccessor, Optional<Map<String, Object>> values, String prefix)
    {
        ValidationResults results = new ValidationResults();
        
        visitView(viewItemAccessor, 
            (element, definition) -> {
                // simple element
                String name = definition.getName();
                String dataPath = prefix + name;
                
                Object value = values.map(v -> v.get(name)).orElse(null);
                ValidationResult result = ModelHelper.validateValue(definition, value);
                results.addResult(dataPath, result);
            }, 
            (group, definition) -> {
                // composite
                String name = definition.getName();
                String updatedPrefix = prefix + name + ModelItem.ITEM_PATH_SEPARATOR;

                Optional<Map<String, Object>> value = values.map(v -> v.get(name)).filter(Map.class::isInstance).map(Map.class::cast);
                results.addResults(_validateValues(group, value, updatedPrefix));
            }, 
            (group, definition) -> {
                // repeater
                String name = definition.getName();
                String dataPath = prefix + name;
                
                Optional<List<Map<String, Object>>> entries = values.map(v -> v.get(name)).filter(List.class::isInstance).map(List.class::cast);
                results.addResults(_validateRepeaterEntries(group, definition, entries, dataPath));
            }, 
            group -> {
                results.addResults(_validateValues(group, values, prefix));   
            });
        
        return results;
    }
    
    private static ValidationResults _validateRepeaterEntries(ModelViewItemGroup viewItem, RepeaterDefinition definition, Optional<List<Map<String, Object>>> entries, String dataPath)
    {
        ValidationResults results = new ValidationResults();

        int repeaterSize = entries.map(List::size).orElse(0);
        List<I18nizableText> repeaterSizeErrors = _validateRepeaterSize(definition, repeaterSize);
        if (!repeaterSizeErrors.isEmpty())
        {
            ValidationResult repeaterResult = new ValidationResult();
            repeaterResult.addErrors(repeaterSizeErrors);
            results.addResult(dataPath, repeaterResult);
        }
        else if (entries.isPresent())
        {
            for (int i = 0; i < repeaterSize; i++)
            {
                Map<String, Object> entry = entries.get().get(i);
                String prefix = dataPath + "[" + (i + 1)  + "]" + ModelItem.ITEM_PATH_SEPARATOR;
                results.addResults(_validateValues(viewItem, Optional.of(entry), prefix));
            }
        }
        
        return results;
    }

    private static List<I18nizableText> _validateRepeaterSize(RepeaterDefinition definition, int repeaterSize)
    {
        int minSize = definition.getMinSize();
        int maxSize = definition.getMaxSize();
        
        List<I18nizableText> errors = new ArrayList<>();
        
        if (repeaterSize < minSize)
        {
            List<String> parameters = new ArrayList<>();
            parameters.add(definition.getName());
            parameters.add(Integer.toString(minSize));
            errors.add(new I18nizableText("plugin.cms", "CONTENT_EDITION_VALIDATION_ERRORS_REPEATER_MINSIZE", parameters));
        }

        if (maxSize > 0 && repeaterSize > maxSize)
        {
            List<String> parameters = new ArrayList<>();
            parameters.add(definition.getName());
            parameters.add(Integer.toString(maxSize));
            errors.add(new I18nizableText("plugin.cms", "CONTENT_EDITION_VALIDATION_ERRORS_REPEATER_MAXSIZE", parameters));
        }
        
        return errors;
    }
}
