/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.provider;

import javax.jcr.Credentials;
import javax.jcr.LoginException;
import javax.jcr.NoSuchWorkspaceException;
import javax.jcr.Repository;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.SimpleCredentials;
import javax.jcr.Value;

import org.apache.avalon.framework.logger.AbstractLogEnabled;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

/**
 * Base class for JCR repository as a component.
 */
public abstract class AbstractRepository extends AbstractLogEnabled implements Repository, Serviceable 
{
    /**
     * Role which shall be used for JCR repository implementations.
     */
    public static final String ROLE = "javax.jcr.Repository";
    
    /** The context repository key. */
    public static final String CONTEXT_REPOSITORY_KEY = "org.ametys.repository.Repository";
    
    /** The credentials object key. */
    public static final String CONTEXT_CREDENTIALS_KEY = "org.ametys.repository.Credentials";
    
    /** The JNDI variable key. */
    public static final String CONTEXT_IS_JNDI_KEY = "org.ametys.repository.JNDI";
    
    /** Actual JCR Repository */
    protected Repository _delegate;

    /** Avalon ServiceManager */
    protected ServiceManager _manager;
    
    private WorkspaceSelector _workspaceSelector;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _manager = manager;
        _workspaceSelector = (WorkspaceSelector) manager.lookup(WorkspaceSelector.ROLE);
    }
    
    public String getDescriptor(String key)
    {
        return _delegate.getDescriptor(key);
    }

    public String[] getDescriptorKeys()
    {
        return _delegate.getDescriptorKeys();
    }
    
    @Override
    public Value getDescriptorValue(String key)
    {
        return _delegate.getDescriptorValue(key);
    }
    
    @Override
    public Value[] getDescriptorValues(String key)
    {
        return _delegate.getDescriptorValues(key);
    }
    
    @Override
    public boolean isSingleValueDescriptor(String key)
    {
        return _delegate.isSingleValueDescriptor(key);
    }
    
    @Override
    public boolean isStandardDescriptor(String key)
    {
        return _delegate.isStandardDescriptor(key);
    }

    public Session login() throws LoginException, NoSuchWorkspaceException, RepositoryException
    {
        String workspace = _workspaceSelector.getWorkspace();
        return login(workspace);
    }

    public Session login(Credentials creds) throws LoginException, NoSuchWorkspaceException, RepositoryException
    {
        String workspace = _workspaceSelector.getWorkspace();
        return login(creds, workspace);
    }

    public Session login(Credentials creds, String workspace) throws LoginException, NoSuchWorkspaceException, RepositoryException
    {
        if (creds != null && !(creds instanceof SimpleCredentials))
        {
            throw new RepositoryException("Credentials must be instance of SimpleCredentials");
        }
        
        if (creds != null)
        {
            String login = ((SimpleCredentials) creds).getUserID();
        
            if (!login.equals("ametys"))
            {
                throw new LoginException("The authentication is only allowed with the ametys login.");
            }
        }
        
        return login(workspace);
    }
}
