/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.provider;

import java.util.List;
import java.util.Map;

import javax.jcr.RepositoryException;

import org.apache.jackrabbit.core.cluster.UpdateEventChannel;
import org.apache.jackrabbit.core.id.NodeId;
import org.apache.jackrabbit.core.id.PropertyId;
import org.apache.jackrabbit.core.persistence.PMContext;
import org.apache.jackrabbit.core.persistence.PersistenceManager;
import org.apache.jackrabbit.core.persistence.check.ConsistencyCheckListener;
import org.apache.jackrabbit.core.persistence.check.ConsistencyReport;
import org.apache.jackrabbit.core.persistence.pool.BundleDbPersistenceManager;
import org.apache.jackrabbit.core.persistence.pool.DerbyPersistenceManager;
import org.apache.jackrabbit.core.persistence.pool.MySqlPersistenceManager;
import org.apache.jackrabbit.core.persistence.pool.OraclePersistenceManager;
import org.apache.jackrabbit.core.persistence.pool.PostgreSQLPersistenceManager;
import org.apache.jackrabbit.core.persistence.util.NodeInfo;
import org.apache.jackrabbit.core.state.ChangeLog;
import org.apache.jackrabbit.core.state.ItemStateException;
import org.apache.jackrabbit.core.state.NoSuchItemStateException;
import org.apache.jackrabbit.core.state.NodeReferences;
import org.apache.jackrabbit.core.state.NodeState;
import org.apache.jackrabbit.core.state.PropertyState;
import org.apache.jackrabbit.core.util.db.ConnectionFactory;

import org.ametys.core.datasource.AbstractDataSourceManager.DataSourceDefinition;
import org.ametys.core.datasource.ConnectionHelper;
import org.ametys.runtime.config.Config;

/**
 * Ametys implementation of Jackrabbit's {@link PersistenceManager}.
 * This implementation is only a wrapper to an actual {@link BundleDbPersistenceManager}, depending on the config parameter.
 */
public class AmetysPersistenceManager extends BundleDbPersistenceManager
{
    /**
     * Config parameter's name to use default Jackrabbit behaviour.
     */
    public static final String CONFIG_USE_DEFAULT = "org.ametys.plugins.repository.default-datasource";
    
    /**
     * Config parameter's name referencing the datasource to use.
     */
    public static final String CONFIG_PM_DATASOURCE = "org.ametys.plugins.repository.datasource";
    
    private BundleDbPersistenceManager _wrappedPM;
    private boolean _useDefault;
    private String _workspace;
    
    /**
     * Constructor.
     */
    public AmetysPersistenceManager()
    {
        _useDefault = Config.getInstance().getValue(CONFIG_USE_DEFAULT);
        
        if (_useDefault)
        {
            _wrappedPM = new DerbyPersistenceManager();
            _wrappedPM.setSchemaObjectPrefix("");
            _wrappedPM.setExternalBLOBs(true);
        }
        else
        {
            String datasource = Config.getInstance().getValue(CONFIG_PM_DATASOURCE);
            
            DataSourceDefinition dsDefinition = ConnectionHelper.getDataSourceDefinition(datasource);
            
            Map<String, Object> parameters = dsDefinition.getParameters();
            String defUrl = (String) parameters.get("url");
            String defDriver = (String) parameters.get("driver");
            String defUser = (String) parameters.get("user");
            String defPassword = (String) parameters.get("password");
            
            String dbType = ConnectionHelper.getDatabaseType(defUrl);
            switch (dbType)
            {
                case ConnectionHelper.DATABASE_DERBY:
                    _wrappedPM = new DerbyPersistenceManager();
                    break;
                case ConnectionHelper.DATABASE_MYSQL:
                    _wrappedPM = new MySqlPersistenceManager();
                    break;
                case ConnectionHelper.DATABASE_ORACLE:
                    _wrappedPM = new OraclePersistenceManager();
                    break;
                case ConnectionHelper.DATABASE_POSTGRES:
                    _wrappedPM = new PostgreSQLPersistenceManager();
                    break;
                default:
                    throw new IllegalArgumentException("Datasource URL " + defUrl + " is not supported for Jackrabbit PersistenceManagers.");
            }
            
            _wrappedPM.setUrl(defUrl);
            _wrappedPM.setDriver(defDriver);
            _wrappedPM.setUser(defUser);
            _wrappedPM.setPassword(defPassword);
        }
    }
    
    /**
     * Get the wrapped persistence manager
     * @return the persistence manager
     */
    public BundleDbPersistenceManager getWrappedPM()
    {
        return _wrappedPM;
    }

    /**
     * Indicates the current JCR workspace.
     * @param workspace the JCR workspace.
     */
    public void setWorkspace(String workspace)
    {
        _workspace = workspace;
    }
    
    // Unsupported methods
    
    @Override
    public void setUrl(String newUrl)
    {
        throw new UnsupportedOperationException("setUrl() method is not supported for AmetysPersistenceManager.");
    }
    
    @Override
    public void setUser(String unused)
    {
        throw new UnsupportedOperationException("setUser() method is not supported for AmetysPersistenceManager.");
    }
    
    @Override
    public void setPassword(String unused)
    {
        throw new UnsupportedOperationException("setPassword() method is not supported for AmetysPersistenceManager.");
    }
    
    @Override
    public void setDriver(String unused)
    {
        throw new UnsupportedOperationException("setDriver() method is not supported for AmetysPersistenceManager.");
    }
    
    // PersistenceManager methods

    @Override
    public void init(PMContext ctx) throws Exception
    {
        if (_useDefault)
        {
            String dbUrl = "jdbc:derby:" + ctx.getHomeDir().getCanonicalPath() + "/db;create=true";
            _wrappedPM.setUrl(dbUrl);
        }
        else
        {
            String prefix = _workspace  == null ? "version_" : _workspace + "_";
            _wrappedPM.setSchemaObjectPrefix(prefix);
        }
        
        _wrappedPM.init(ctx);
    }

    @Override
    public void close() throws Exception
    {
        _wrappedPM.close();
    }

    @Override
    public NodeState createNew(NodeId id)
    {
        return _wrappedPM.createNew(id);
    }

    @Override
    public PropertyState createNew(PropertyId id)
    {
        return _wrappedPM.createNew(id);
    }

    @Override
    public NodeState load(NodeId id) throws NoSuchItemStateException, ItemStateException
    {
        return _wrappedPM.load(id);
    }

    @Override
    public PropertyState load(PropertyId id) throws NoSuchItemStateException, ItemStateException
    {
        return _wrappedPM.load(id);
    }

    @Override
    public NodeReferences loadReferencesTo(NodeId id) throws NoSuchItemStateException, ItemStateException
    {
        return _wrappedPM.loadReferencesTo(id);
    }

    @Override
    public boolean exists(NodeId id) throws ItemStateException
    {
        return _wrappedPM.exists(id);
    }

    @Override
    public boolean exists(PropertyId id) throws ItemStateException
    {
        return _wrappedPM.exists(id);
    }

    @Override
    public boolean existsReferencesTo(NodeId targetId) throws ItemStateException
    {
        return _wrappedPM.existsReferencesTo(targetId);
    }

    @Override
    public void store(ChangeLog changeLog) throws ItemStateException
    {
        _wrappedPM.store(changeLog);
    }
    
    @Override
    public ConsistencyReport check(String[] uuids, boolean recursive, boolean fix, String lostNFoundId, ConsistencyCheckListener listener) throws RepositoryException
    {
        return _wrappedPM.check(uuids, recursive, fix, lostNFoundId, listener);
    }

    @Override
    public void checkConsistency(String[] uuids, boolean recursive, boolean fix)
    {
        _wrappedPM.checkConsistency(uuids, recursive, fix);
    }
    
    // BundleDBPersistenceManager methods
    
    @Override
    public synchronized List<NodeId> getAllNodeIds(NodeId bigger, int maxCount) throws ItemStateException, RepositoryException
    {
        return _wrappedPM.getAllNodeIds(bigger, maxCount);
    }
    
    @Override
    public synchronized void destroy(NodeReferences refs) throws ItemStateException
    {
        _wrappedPM.destroy(refs);
    }
    
    @Override
    public synchronized Map<NodeId, NodeInfo> getAllNodeInfos(NodeId bigger, int maxCount) throws ItemStateException
    {
        return _wrappedPM.getAllNodeInfos(bigger, maxCount);
    }
    
    @Override
    public void setBlockOnConnectionLoss(String block)
    {
        _wrappedPM.setBlockOnConnectionLoss(block);
    }
    
    @Override
    public void setBundleCacheSize(String bundleCacheSize)
    {
        _wrappedPM.setBundleCacheSize(bundleCacheSize);
    }
    
    @Override
    public void setConnectionFactory(ConnectionFactory connectionFactory)
    {
        _wrappedPM.setConnectionFactory(connectionFactory);
    }
    
    @Override
    public void setConsistencyCheck(String check)
    {
        _wrappedPM.setConsistencyCheck(check);
    }
    
    @Override
    public void setConsistencyFix(String fix)
    {
        _wrappedPM.setConsistencyFix(fix);
    }
    
    @Override
    public void setErrorHandling(String errHandling)
    {
        _wrappedPM.setErrorHandling(errHandling);
    }
    
    @Override
    public void setEventChannel(UpdateEventChannel eventChannel)
    {
        _wrappedPM.setEventChannel(eventChannel);
    }
    
    @Override
    public void setExternalBLOBs(boolean externalBlobs)
    {
        _wrappedPM.setExternalBLOBs(externalBlobs);
    }
    
    @Override
    public void setMinBlobSize(String minBlobSize)
    {
        _wrappedPM.setMinBlobSize(minBlobSize);
    }
    
    @Override
    public void setSchemaObjectPrefix(String schemaPrefix)
    {
        _wrappedPM.setSchemaObjectPrefix(schemaPrefix);
    }
    
    @Override
    public String toString()
    {
        return _wrappedPM.toString();
    }
}
