/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.provider;

import java.util.concurrent.atomic.AtomicInteger;

import javax.jcr.AccessDeniedException;
import javax.jcr.Credentials;
import javax.jcr.LoginException;
import javax.jcr.NoSuchWorkspaceException;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.security.auth.Subject;

import org.apache.jackrabbit.api.stats.RepositoryStatistics;
import org.apache.jackrabbit.core.RepositoryContext;
import org.apache.jackrabbit.core.RepositoryImpl;
import org.apache.jackrabbit.core.SessionImpl;
import org.apache.jackrabbit.core.config.RepositoryConfig;
import org.apache.jackrabbit.core.config.WorkspaceConfig;
import org.apache.jackrabbit.core.security.authentication.AuthContext;

/**
 * JCR Repository implementation, based on Jackrabbit and allowing to change the logout behaviour for Sessions
 */
public class AmetysRepository extends RepositoryImpl
{
    private LogoutManager _logoutManager;
    private AtomicInteger _sessionCount = new AtomicInteger();
    private boolean _isShuttingDown;
    
    /**
     * Creates a Repository.
     * @param config the repository configuration.
     * @throws RepositoryException if an error occurs.
     */
    public AmetysRepository(RepositoryConfig config) throws RepositoryException
    {
        super(config);
    }
    
    /** 
     * Get the repository context
     * @return the repository context
     */
    public RepositoryContext getContext()
    {
        return context;
    }
    
    /**
     * Get the JCR repository statistics
     * @return a {@link RepositoryStatistics} representing all the statistics available
     */
    public RepositoryStatistics getRepositoryStatistics()
    {
        return context.getRepositoryStatistics();
    }
    
    @Override
    protected SessionImpl createSessionInstance(AuthContext loginContext, WorkspaceConfig wspConfig) throws AccessDeniedException, RepositoryException
    {
        return new AmetysSession(context, loginContext, wspConfig);
    }
    
    @Override
    protected SessionImpl createSessionInstance(Subject subject, WorkspaceConfig wspConfig) throws AccessDeniedException, RepositoryException
    {
        return new AmetysSession(context, subject, wspConfig);
    }
    
    void setLogoutManager(LogoutManager logoutManager)
    {
        _logoutManager = logoutManager;
    }
    
    LogoutManager getLogoutManager()
    {
        return _logoutManager;
    }
    
    int getSessionCount()
    {
        return _sessionCount.get();
    }
    
    @Override
    protected void onSessionCreated(SessionImpl session)
    {
        super.onSessionCreated(session);
        
        if (session instanceof AmetysSession)
        {
            _sessionCount.incrementAndGet();
        }
    }
    
    @Override
    public void loggedOut(SessionImpl session)
    {
        super.loggedOut(session);
        
        if (session instanceof AmetysSession)
        {
            _sessionCount.decrementAndGet();
        }
    }
    
    @Override
    protected synchronized void doShutdown() 
    {
        _isShuttingDown = true;
        super.doShutdown();
    }
    
    @Override
    public Session login(Credentials credentials, String workspaceName) throws LoginException, NoSuchWorkspaceException, RepositoryException
    {
        if (_isShuttingDown)
        {
            throw new RepositoryException("Cannot login as the repository is shutting down.");
        }
        return super.login(credentials, workspaceName);
    }
}
