/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.ametys.plugins.repository.provider;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.time.Instant;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.HashMap;
import java.util.Map;

import javax.jcr.LoginException;
import javax.jcr.NoSuchWorkspaceException;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.SimpleCredentials;

import org.apache.avalon.framework.CascadingRuntimeException;
import org.apache.avalon.framework.activity.Disposable;
import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.Constants;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Context;
import org.apache.cocoon.environment.Request;
import org.apache.commons.io.IOUtils;
import org.apache.excalibur.source.ModifiableSource;
import org.apache.excalibur.source.ModifiableTraversableSource;
import org.apache.excalibur.source.MoveableSource;
import org.apache.excalibur.source.Source;
import org.apache.excalibur.source.SourceResolver;
import org.apache.excalibur.source.TraversableSource;
import org.apache.jackrabbit.api.stats.RepositoryStatistics;
import org.apache.jackrabbit.core.cache.CacheManager;
import org.apache.jackrabbit.core.config.ConfigurationException;
import org.apache.jackrabbit.core.config.RepositoryConfig;
import org.apache.jackrabbit.stats.RepositoryStatisticsImpl;

import org.ametys.core.datasource.ConnectionHelper;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.util.AmetysHomeHelper;

/**
 * JackrabbitRepository is a JCR repository component based on Jackrabbit
 */
public class JackrabbitRepository extends AbstractRepository implements LogoutManager, Initializable, Contextualizable, Disposable, Component
{
    private static final String __REPOSITORY_NODETYPES_PATH = "ametys-home://data/repository/repository/nodetypes";

    private org.apache.avalon.framework.context.Context _avalonContext;
    private Context _context;
    
    private Session _adminSession;

    private SourceResolver _resolver;
    
    /**
     * Must implements ModifiableTraversableSource, MoveableSource
     */
    private Source _customNodeTypesBackupSource;

    @Override
    public void contextualize(org.apache.avalon.framework.context.Context context) throws ContextException
    {
        _avalonContext = context;
        _context = (Context) context.get(Constants.CONTEXT_ENVIRONMENT_CONTEXT);
    }
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        // Make sure the ConnectionHelper is initialized first, so the SQLDataSourceManager can be used in AmetysPersistenceManager
        smanager.lookup(ConnectionHelper.ROLE);
        _resolver = (SourceResolver) smanager.lookup(SourceResolver.ROLE);
    }
    
    
    public void initialize() throws Exception
    {
        _backupCustomNodetypes();
        RepositoryConfig repositoryConfig = createRepositoryConfig();
        AmetysRepository repo = new AmetysRepository(repositoryConfig);
        
        _delegate = repo;
        ((AmetysRepository) _delegate).setLogoutManager(this);
        
        Long cacheSize = Config.getInstance().getValue("org.ametys.plugins.repository.cache");  
        if (cacheSize == null || cacheSize <= 0)
        {
            cacheSize = 16777216L; // default value;
        }
        
        CacheManager cacheManager = repo.getCacheManager();
        cacheManager.setMaxMemory(cacheSize);
        cacheManager.setMaxMemoryPerCache(cacheSize / 4);
        cacheManager.setMinMemoryPerCache(cacheSize / 128);
        
        // Register the delegate repository in the context for the repository workspace.
        _context.setAttribute(CONTEXT_REPOSITORY_KEY, _delegate);
        _context.setAttribute(CONTEXT_CREDENTIALS_KEY, new SimpleCredentials("ametys", new char[]{}));
        _context.setAttribute(CONTEXT_IS_JNDI_KEY, false);
        
        _adminSession = login("default");
        
        if (getLogger().isDebugEnabled())
        {
            getLogger().debug("JCR Repository running");
        }
    }

    public void dispose()
    {
        AmetysRepository repo = (AmetysRepository) _delegate;
        repo.setLogoutManager(null);
        repo.shutdown();
    }
    
    /**
     * Get the JCR repository statistics
     * @return a {@link RepositoryStatisticsImpl} representing all the statistics available
     */
    public RepositoryStatistics getRepositoryStatistics()
    {
        AmetysRepository repo = (AmetysRepository) _delegate;
        return repo.getRepositoryStatistics();
    }
    
    /**
     * Returns the admin session
     * @return the admin session
     * @throws RepositoryException if an error occurs.
     */
    public Session getAdminSession() throws RepositoryException
    {
        _adminSession.refresh(false);
        return _adminSession;
    }
    
    
    /**
     * Returns the all JCR workspaces
     * @return the available workspaces
     * @throws RepositoryException if an error occurred
     */
    public String[] getWorkspaces() throws RepositoryException
    {
        return _adminSession.getWorkspace().getAccessibleWorkspaceNames();
    }


    /**
     * Create the repository configuration from the configuration file
     * @return The repository configuration
     * @throws ConfigurationException if an error occurred
     */
    RepositoryConfig createRepositoryConfig() throws ConfigurationException
    {
        String config = _context.getRealPath("/WEB-INF/param/repository.xml");
        
        File homeFile = new File(AmetysHomeHelper.getAmetysHomeData(), "repository");
        
        if (getLogger().isDebugEnabled())
        {
            getLogger().debug("Creating JCR Repository config at: " + homeFile.getAbsolutePath());
        }
        
        return RepositoryConfig.create(config, homeFile.getAbsolutePath());
    }

    int getSessionCount()
    {
        return ((AmetysRepository) _delegate).getSessionCount();
    }
    
    public Session login(String workspace) throws LoginException, NoSuchWorkspaceException, RepositoryException
    {
        try
        {
            if (getLogger().isDebugEnabled())
            {
                getLogger().debug("Getting JCR Session");
            }
            
            try
            {
                Request request = ContextHelper.getRequest(_avalonContext);
                
                @SuppressWarnings("unchecked")
                Map<String, Session> sessions = (Map<String, Session>) request.getAttribute(RepositoryConstants.JCR_SESSION_REQUEST_ATTRIBUTE);
                
                if (sessions == null)
                {
                    sessions = new HashMap<>();
                    request.setAttribute(RepositoryConstants.JCR_SESSION_REQUEST_ATTRIBUTE, sessions);
                }
                
                Session session = sessions.get(workspace);
                
                if (session == null || !session.isLive())
                {
                    session = _delegate.login(new SimpleCredentials("ametys", new char[]{}), workspace);
                    sessions.put(workspace, session);
                }
                
                return session;
            }
            catch (CascadingRuntimeException e)
            {
                if (e.getCause() instanceof ContextException)
                {
                    // Unable to get request. Must be in another thread or at init time.
                    Session session = _delegate.login(new SimpleCredentials("ametys", new char[]{}), workspace);
                    return session;
                }
                else
                {
                    throw e;
                }
            }
        }
        catch (RepositoryException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            throw new RuntimeException("Unable to get Session", e);
        }
    }
    
    public void logout(Session session)
    {
        if (!(session instanceof AmetysSession))
        {
            throw new IllegalArgumentException("JCR Session should be an instance of AmetysSession");
        }
        
        AmetysSession ametysSession = (AmetysSession) session;
        
        if (getLogger().isDebugEnabled())
        {
            getLogger().debug("Logging out AmetysSession");
        }
        
        try
        {
            // the following statement will fail if we are not processing a request.
            ContextHelper.getRequest(_avalonContext);
            
            // does nothing as the session will be actually logged out at the end of the request.
            if (getLogger().isDebugEnabled())
            {
                getLogger().debug("AmetysSession logout delayed until the end of the HTTP request.");
            }
            
        }
        catch (Exception e)
        {
            // unable to get request. Must be in another thread or at init time.
            if (getLogger().isDebugEnabled())
            {
                getLogger().debug("Not in a request. AmetysSession will be actually logged out.");
            }
            
            ametysSession.forceLogout();
        }
    }

    /**
     * Create a backup of custom_nodetypes.xml and make a backup with the curent time in the filename
     * The backup file name is stored to be compared ONCE with the re-created custom_nodetypes.xml by calling {@link JackrabbitRepository#compareCustomNodetypes()}
     */
    protected void _backupCustomNodetypes()
    {
        Source source = null;
        Source destination = null;
        try
        {
            source = _resolver.resolveURI(__REPOSITORY_NODETYPES_PATH + "/custom_nodetypes.xml");
            if (source.exists() && source instanceof ModifiableSource)
            {
                ModifiableSource fsource = (ModifiableSource) source;
                
                DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyyMMdd-HHmmss").withZone(ZoneId.systemDefault());
                String timestamp = formatter.format(Instant.now());
                
                destination = _resolver.resolveURI(__REPOSITORY_NODETYPES_PATH + "/custom_nodetypes-" + timestamp + ".xml");
                if (destination.exists())
                {
                    fsource.delete();
                    getLogger().warn("Impossible to backup custom_nodetypes.xml, custom_nodetypes-" + timestamp + ".xml already exists.");
                }
                else
                {
                    _customNodeTypesBackupSource = destination;
                    getLogger().info("Backup custom_nodetypes.xml as custom_nodetypes-" + timestamp + ".xml");
                    ((MoveableSource) fsource).moveTo(destination);
                }
                
                if (fsource.exists())
                {
                    getLogger().error("Impossible to delete custom_nodetypes.xml after creating a backup.");
                }
            }
        }
        catch (IOException e)
        {
            getLogger().error("An error occurred while backuping the custom_nodetypes.xml file", e);
        }
        finally
        {
            _resolver.release(source);
            _resolver.release(destination);
        }
    }
    
    /**
     * Compare custom_nodetypes.xml with custom_nodetypes-[timestamp].xml and delete the backup if they are the same
     */
    public void compareCustomNodetypes()
    {
        if (_customNodeTypesBackupSource != null && _customNodeTypesBackupSource.exists())
        {
            Source source = null;
            try
            {
                source = _resolver.resolveURI(__REPOSITORY_NODETYPES_PATH + "/custom_nodetypes.xml");
                
                if (source.exists() && source instanceof ModifiableTraversableSource)
                {
                    if (_compareSources((TraversableSource) source, (TraversableSource) _customNodeTypesBackupSource))
                    {
                        getLogger().info(((TraversableSource) _customNodeTypesBackupSource).getName() + " will be deleted, no changes found.");
                        ((ModifiableTraversableSource) _customNodeTypesBackupSource).delete();

                        
                        if (_customNodeTypesBackupSource.exists())
                        {
                            getLogger().error("Impossible to delete custom_nodetypes.xml after creating a backup.");
                        }
                    }
                    else
                    {
                        getLogger().info(((TraversableSource) _customNodeTypesBackupSource).getName() + " will be kept, changes found with last version.");
                    }
                }
                else
                {
                    getLogger().warn("Impossible to compare custom_nodetypes.xml, it seems to be unavailable.");
                }
            }
            catch (IOException e)
            {
                getLogger().error("Impossible to compare custom_nodetypes.xml with it's backup '" + _customNodeTypesBackupSource.getURI() + "'", e);
            }
            finally
            {
                _resolver.release(source);
            }
        }
        else
        {
            getLogger().debug("There is no backup of custom_nodetypes.xml to compare");
        }
        
        _resolver.release(_customNodeTypesBackupSource);
        _customNodeTypesBackupSource = null;
    }
    
    /**
     * Compare 2 sources
     * @param source1 source to compare
     * @param source2 source to compare
     * @return true if the content of both sources is equal
     * @throws IOException something went wrong reading the sources
     */
    protected boolean _compareSources(TraversableSource source1, TraversableSource source2) throws IOException
    {
        if (source1 == null && source2 == null)
        {
            return true;
        }
        if (source1 == null || source2 == null)
        {
            return false;
        }
        final boolean source1Exists = source1.exists();
        if (source1Exists != source2.exists())
        {
            return false;
        }

        if (!source1Exists)
        {
            // two not existing files are equal
            return true;
        }
        


        if (source1.isCollection() || source2.isCollection())
        {
            // don't want to compare directory contents
            throw new IOException("Can't compare collections, only content source");
        }

        if (source1.getContentLength() != source2.getContentLength())
        {
            // lengths differ, cannot be equal
            return false;
        }

        if (source1.getURI().equals(source2.getURI()))
        {
            // same source
            return true;
        }

        try (InputStream input1 = source1.getInputStream();
             InputStream input2 = source2.getInputStream())
        {
            return IOUtils.contentEquals(input1, input2);
        }
    }
}
