/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.query;

import org.apache.commons.lang3.StringUtils;

import org.ametys.plugins.repository.query.expression.Expression;

/**
 * Helper for creating JCR XPath queries.<br>
 * Created XPath queries are like : <code>//element(*, ametys:object)[<i>&lt;predicates&lt;&gt;</i>]</code>
 */
public final class QueryHelper
{
    private QueryHelper()
    {
        // empty constructor
    }
    
    /**
     * Creates the XPath query corresponding to specified {@link Expression} and {@link SortCriteria}.
     * @param namePattern a pattern for the name of the resulting nodes. May be null, which is equivalent to '*'
     * @param nodetype the base nodetype for the query. May be null, which is equivalent to ametys:object
     * @param expression the query predicates
     * @param sortCriteria criteria for sorting results
     * @return the created XPath query
     */
    public static String getXPathQuery(String namePattern, String nodetype, Expression expression, SortCriteria sortCriteria)
    {
        String predicats = null;
        
        if (expression != null)
        {
            predicats = StringUtils.trimToNull(expression.build());
        }
        
        String type = nodetype == null ? "ametys:object" : nodetype;
        String name = namePattern == null ? "*" : namePattern;
        
        String xpathQuery = "//element(" + name + ", " + type + ")" + (predicats != null ? "[" + predicats + "]" : "") + ((sortCriteria != null) ? (" " + sortCriteria.build()) : "");
        return xpathQuery;
    }

    /**
     * Creates the XPath query corresponding to specified {@link Expression}.
     * @param namePattern a pattern for the name of the resulting nodes. May be null, which is equivalent to '*'
     * @param nodetype the base nodetype for the query. May be null, which is equivalent to 'ametys:object'
     * @param expression the query predicates
     * @return the created XPath query
     */
    public static String getXPathQuery(String namePattern, String nodetype, Expression expression)
    {
        return getXPathQuery(namePattern, nodetype, expression, null);
    }
}
