/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.query;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.ametys.plugins.repository.RepositoryConstants;

/**
 * A sort criteria allows to sort query results.<br>
 * Warning : methods of this class support a boolean "normalize" which suppose that the patch at https://issues.apache.org/jira/browse/JCR-3443 has been applied.
 */
public class SortCriteria
{
    private List<SortCriterion> _criteria = new ArrayList<>();

    /**
     * Add a sort criteria to the criteria list, using a metadata name. <br>
     * Order of adding is important. First added has more weight during sort.
     * @param metadataPath The name of the metadata to sort, or the path to the metadata if composite
     * @param ascending The order for sorting results
     * @param normalize <code>true</code> to normalize string properties (remove accents and lower case) 
     */
    public void addCriterion(String metadataPath, boolean ascending, boolean normalize)
    {
        _criteria.add(new SortCriterion(metadataPath, null, ascending, normalize));
    }

    /**
     * Add a sort criteria to the criteria list, using a JCR property name. <br>
     *  Order of adding is important. First added has more weight during sort.
     * @param jcrProperty The name of the JCR property to sort
     * @param ascending The order for sorting results
     * @param normalize <code>true</code> to normalize string properties (remove accents and lower case) 
     */
    public void addJCRPropertyCriterion(String jcrProperty, boolean ascending, boolean normalize)
    {
        _criteria.add(new SortCriterion(null, jcrProperty, ascending, normalize));
    }
    
    /**
     * Get the criteria.
     * @return an unmodifiable view of the criteria list.
     */
    public List<SortCriterion> getCriteria()
    {
        return Collections.unmodifiableList(_criteria);
    }
    
    /**
     * Build the criteria.
     * @return The formal view of the criteria.
     */
    public String build()
    {
        if (_criteria.isEmpty())
        {
            return "";
        }

        StringBuffer exp = new StringBuffer("order by ");
        for (int i = 0; i < _criteria.size(); i++)
        {
            if (i != 0)
            {
                exp.append(", ");
            }
            
            SortCriterion criterion = _criteria.get(i);
            
            exp.append(criterion.build());
        }
        
        return exp.toString();
    }
    
    /**
     * A sort criterion.
     */
    public class SortCriterion
    {
        
        /** The metadata path. */
        protected String _metadataPath;
        
        /** The JCR property. */
        protected String _jcrProperty;
        
        /** True if the sort is made in ascending order, false otherwise. */
        protected boolean _ascending;
        
        /** True to sort on normalized versions of the properties, false otherwise. */
        protected boolean _normalize;
        
        /**
         * Build a sort criterion.
         * @param metadataPath the metadata path (can be null if a JCR property is provided).
         * @param jcrProperty the JCR property (can be null if a metadata path is provided).
         * @param ascending true to sort in ascending order, false otherwise.
         * @param normalize true to sort on normalized properties, false otherwise.
         */
        public SortCriterion(String metadataPath, String jcrProperty, boolean ascending, boolean normalize)
        {
            this._metadataPath = metadataPath;
            this._jcrProperty = jcrProperty;
            this._ascending = ascending;
            this._normalize = normalize;
        }
        
        /**
         * Get the metadataPath.
         * @return the metadataPath
         */
        public String getMetadataPath()
        {
            return _metadataPath;
        }
        
        /**
         * Set the metadataPath.
         * @param metadataPath the metadataPath to set
         */
        public void setMetadataPath(String metadataPath)
        {
            this._metadataPath = metadataPath;
        }
        
        /**
         * Get the jcrProperty.
         * @return the jcrProperty
         */
        public String getJcrProperty()
        {
            return _jcrProperty;
        }
        
        /**
         * Set the jcrProperty.
         * @param jcrProperty the jcrProperty to set
         */
        public void setJcrProperty(String jcrProperty)
        {
            this._jcrProperty = jcrProperty;
        }
        
        /**
         * Test if the results are to be sorted in ascending order.
         * @return true to sort in ascending order, false otherwise.
         */
        public boolean isAscending()
        {
            return _ascending;
        }
        
        /**
         * Set the criterion to sort in ascending order.
         * @param ascending true to sort in ascending order, false otherwise.
         */
        public void setAscending(boolean ascending)
        {
            this._ascending = ascending;
        }
        
        /**
         * Test if the results are to be sorted in normalized form.
         * @return true to sort in normalized form, false otherwise.
         */
        public boolean isNormalizedSort()
        {
            return _normalize;
        }
        
        /**
         * Set the criterion to sort in normalized form.
         * @param normalize true to sort in normalized form, false otherwise.
         */
        public void setNormalize(boolean normalize)
        {
            this._normalize = normalize;
        }
        
        /**
         * Build an XPath order string representing the criterion.
         * @return an XPath order string.
         */
        public String build()
        {
            StringBuilder buff = new StringBuilder();
            
            String ascending = _ascending ? "ascending" : "descending";
            
            if (_metadataPath != null)
            {
                String metadata = "";
                String[] path = _metadataPath.split("/");
                for (int j = 0; j < path.length; j++)
                {
                    if (j == (path.length - 1))
                    {
                        metadata += "@" + RepositoryConstants.NAMESPACE_PREFIX + ':' + path[j];
                    }
                    else
                    {
                        metadata += RepositoryConstants.NAMESPACE_PREFIX + ':' + path[j] + "/";
                    }
                }
                if (_normalize)
                {
                    buff.append("rep:normalize(" + metadata + ") " + ascending);
                }
                else
                {
                    buff.append(metadata + " " + ascending);
                }
            }
            else
            {
                if (_normalize)
                {
                    buff.append("rep:normalize(@" + _jcrProperty + ") " + ascending);
                }
                else
                {
                    buff.append("@" + _jcrProperty + " " + ascending);
                }
            }
            
            return buff.toString();
        }
        
    }
    
}
