/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.query.expression;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Objects;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;

/**
 * Constructs an {@link Expression} corresponding to a logical operator between several {@link Expression}.
 */
public abstract class AbstractLogicalExpression extends ArrayList<Expression> implements Expression
{
    /**
     * Creates the logical expression.
     * @param exprs the operands
     */
    public AbstractLogicalExpression(Collection<Expression> exprs)
    {
        super(exprs);
    }
    
    /**
     * Creates the logical expression.
     * @param exprs the operands
     */
    public AbstractLogicalExpression(Expression ... exprs)
    {
        super(Arrays.asList(exprs));
    }
    
    public String build()
    {
        String logicalOperator = getLogicalOperator().toString();
        
        String joinExpression = this.stream()
            .filter(Objects::nonNull)
            .map(Expression::build)
            .filter(StringUtils::isNotBlank)
            .collect(Collectors.joining(logicalOperator));

        if (joinExpression.contains(logicalOperator))
        {
            joinExpression = "(" + joinExpression + ")";
        }
        
        return joinExpression;
    }
    
    /**
     * Get the logicial operator to apply between operations, can contains spaces to separate the operator from the expressions.
     * @return the operator to apply between each expressions
     */
    protected abstract LogicalOperator getLogicalOperator();
}
