/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.query.expression;

/**
 * Constructs an {@link Expression} corresponding to the boolean comparison with a metadata.
 */
public class BooleanExpression implements Expression
{
    private boolean _value;
    private MetadataExpression _metadata;
    private Operator _operator;

    /**
     * Creates the comparison Expression.
     * @param metadata the metadata name
     * @param value the boolean value
     */
    public BooleanExpression(String metadata, boolean value)
    {
        this(metadata, Operator.EQ, value);
    }
    
    /**
     * Creates the comparison Expression.
     * @param metadata the metadata name
     * @param operator the operator to make the comparison
     * @param value the boolean value
     */
    public BooleanExpression(String metadata, Operator operator, boolean value)
    {
        _value = value;
        _operator = operator;
        _metadata = new MetadataExpression(metadata);
    }
    
    /**
     * Creates the comparison Expression.
     * @param metadata the metadata name
     * @param value the boolean value
     * @param context the expression context
     */
    public BooleanExpression(String metadata, boolean value, ExpressionContext context)
    {
        this(metadata, Operator.EQ, value, context);
    }
    
    /**
     * Creates the comparison Expression.
     * @param metadata the metadata name
     * @param operator the operator to make the comparison
     * @param value the boolean value
     * @param context the expression context
     */
    public BooleanExpression(String metadata, Operator operator, boolean value, ExpressionContext context)
    {
        _value = value;
        _operator = operator;
        _metadata = new MetadataExpression(metadata, context);
    }


    public String build()
    {
        return _metadata.build() + " " + _operator + " " + Boolean.toString(_value) + "()";
    }
}
