/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.query.expression;

import java.time.LocalDate;
import java.time.ZonedDateTime;
import java.util.Date;

import org.apache.commons.lang3.time.FastDateFormat;

import org.ametys.core.util.DateUtils;

/**
 * Constructs an {@link Expression} corresponding to the Date comparison with a metadata.
 */
public class DateExpression implements Expression
{
    private Date _value;
    private MetadataExpression _metadata;
    private Operator _operator;

    /**
     * Creates the comparison Expression.
     * @param metadata the metadata name
     * @param operator the operator to make the comparison
     * @param value the Date value
     */
    public DateExpression(String metadata, Operator operator, Date value)
    {
        _value = value;
        _operator = operator;
        _metadata = new MetadataExpression(metadata);
    }
    
    /**
     * Creates the comparison Expression.
     * @param metadata the metadata name
     * @param operator the operator to make the comparison
     * @param value the Date value
     * @param context the expression context
     */
    public DateExpression(String metadata, Operator operator, Date value, ExpressionContext context)
    {
        _value = value;
        _operator = operator;
        _metadata = new MetadataExpression(metadata, context);
    }
    
    /**
     * Creates the comparison Expression
     * @param metadata the metadata name
     * @param operator the operator to make the comparison
     * @param value the LocalDate value
     */
    public DateExpression(String metadata, Operator operator, LocalDate value)
    {
        this(metadata, operator, DateUtils.asDate(value));
    }

    /**
     * Creates the comparison Expression
     * @param metadata the metadata name
     * @param operator the operator to make the comparison
     * @param value the LocalDate value
     * @param context the expression context
     */
    public DateExpression(String metadata, Operator operator, LocalDate value, ExpressionContext context)
    {
        this(metadata, operator, DateUtils.asDate(value), context);
    }

    /**
     * Creates the comparison Expression
     * @param metadata the metadata name
     * @param operator the operator to make the comparison
     * @param value the ZonedDateTime value
     */
    public DateExpression(String metadata, Operator operator, ZonedDateTime value)
    {
        this(metadata, operator, DateUtils.asDate(value));
    }

    /**
     * Creates the comparison Expression
     * @param metadata the metadata name
     * @param operator the operator to make the comparison
     * @param value the ZonedDateTime value
     * @param context the expression context
     */
    public DateExpression(String metadata, Operator operator, ZonedDateTime value, ExpressionContext context)
    {
        this(metadata, operator, DateUtils.asDate(value), context);
    }

    public String build()
    {
        return _metadata.build() + " " + _operator + " xs:dateTime('" + FastDateFormat.getInstance("yyyy-MM-dd'T'HH:mm:ss.SSSZZ").format(_value) + "')";
    }
}
