/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.query.expression;

/**
 * Object that gives some context for expressions
 */
public class ExpressionContext
{
    private boolean _unversioned;
    private boolean _internal;
    private boolean _caseInsensitive;
    
    /**
     * Creates a new instance of a {@link ExpressionContext}
     */
    protected ExpressionContext()
    {
        // Empty private constructor
    }
    
    /**
     * Creates a new instance of a {@link ExpressionContext}
     * @return the created instance
     */
    public static ExpressionContext newInstance()
    {
        return new ExpressionContext();
    }
    
    /**
     * Determines if the data under the expression is internal
     * @return <code>true</code> if the data is internal, <code>false</code> otherwise
     */
    public boolean unversioned()
    {
        return _unversioned;
    }
    
    /**
     * Set to <code>true</code> to create an expression for unversioned data (default to <code>false</code>) 
     * @param unversioned <code>true</code> to create an expression for unversioned data, <code>false</code> otherwise
     * @return the current {@link ExpressionContext}
     */
    public ExpressionContext withUnversioned(boolean unversioned)
    {
        _unversioned = unversioned;
        return this;
    }
    
    /**
     * Determines if the data under the expression is internal
     * @return <code>true</code> if the data is internal, <code>false</code> otherwise
     */
    public boolean internal()
    {
        return _internal;
    }
    
    /**
     * Set to <code>true</code> to create an expression for internal data (default to <code>false</code>) 
     * @param internal <code>true</code> to create an expression for internal data, <code>false</code> otherwise
     * @return the current {@link ExpressionContext}
     */
    public ExpressionContext withInternal(boolean internal)
    {
        _internal = internal;
        return this;
    }
    
    /**
     * Determines if the expression is not case sensitive
     * @return <code>true</code> if the expression is not case sensitive, <code>false</code> otherwise
     */
    public boolean caseInsensitive()
    {
        return _caseInsensitive;
    }
    
    /**
     * Set to <code>true</code> to create a case insensitive expression (default to <code>false</code>) 
     * @param caseInsensitive <code>true</code> to create a case insensitive expression, <code>false</code> otherwise
     * @return the current {@link ExpressionContext}
     */
    public ExpressionContext withCaseInsensitive(boolean caseInsensitive)
    {
        _caseInsensitive = caseInsensitive;
        return this;
    }
}
