/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.query.expression;

/**
 * Constructs an {@link Expression} corresponding to the String comparison with a metadata.
 */
public class StringExpression implements Expression
{
    private String _value;
    private MetadataExpression _metadata;
    private Operator _operator;
    private boolean _caseInsensitive; 
    
    /**
     * Creates the comparison Expression.
     * @param metadata the metadata name
     * @param operator the operator to make the comparison
     * @param value the String value
     */
    public StringExpression(String metadata, Operator operator, String value)
    {
        _value = value;
        _operator = operator;
        _metadata = new MetadataExpression(metadata);
        _caseInsensitive = false;
    }
    
    /**
     * Creates the comparison Expression.
     * @param metadata the metadata name
     * @param operator the operator to make the comparison
     * @param value the String value
     * @param context the expression context
     */
    public StringExpression(String metadata, Operator operator, String value, ExpressionContext context)
    {
        _value = value;
        _operator = operator;
        _metadata = new MetadataExpression(metadata, context);
        _caseInsensitive = context.caseInsensitive();
    }
    
    public String build()
    {
        if (_operator.equals(Operator.WD))
        {
            if (_caseInsensitive)
            {
                return "jcr:like(fn:lower-case(" + _metadata.build() + "), '%" + _escapeValue(_value.toLowerCase()) + "%')";
            }
            else
            {
                return "jcr:like(" + _metadata.build() + ", '%" + _escapeValue(_value) + "%')";
            }
        }
        else
        {
            if (_caseInsensitive)
            {
                return "fn:lower-case(" + _metadata.build() + ") " + _operator + " '" + _value.toLowerCase().replaceAll("'", "''") + "'";
            }
            else
            {
                return _metadata.build() + " " + _operator + " '" + _value.replaceAll("'", "''") + "'";
            }
        }
    }
    
    private String _escapeValue(String value)
    {
        // First escape ' - " \ _ and %, Then escape ' into '' for XQuery compliance
        return value.replaceAll("(['\\-_\"\\\\%])", "\\\\$1").replaceAll("'", "''");
    }
}
