/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.query.expression;

import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.repository.RepositoryConstants;

/**
 * Constructs an {@link Expression} corresponding to a user comparison store as a composite with login and id of population.
 */
public class UserExpression implements Expression
{
    private UserIdentity _user;
    private Operator _operator;
    private String _metadata;
    private boolean _unversioned;
    private boolean _multiple;
    
    /**
     * Creates the comparison Expression.
     * @param metadata the metadata path
     * @param operator the operator to make the comparison
     * @param value the user identity as login#population.
     */
    public UserExpression(String metadata, Operator operator, String value)
    {
        this(metadata, operator, UserIdentity.stringToUserIdentity(value), false);
    }
    
    /**
     * Creates the comparison Expression.
     * @param metadata the metadata path
     * @param operator the operator to make the comparison
     * @param value the user identity as login#population.
     * @param multiple true if the attribute is multiple
     */
    public UserExpression(String metadata, Operator operator, String value, boolean multiple)
    {
        this(metadata, operator, UserIdentity.stringToUserIdentity(value), multiple, false);
    }
    
    /**
     * Creates the comparison Expression.
     * @param metadata the metadata path
     * @param operator the operator to make the comparison
     * @param value the user identity as login#population.
     * @param multiple true if the attribute is multiple
     * @param unversioned true if the metadata is unversioned, false otherwise.
     */
    public UserExpression(String metadata, Operator operator, String value, boolean multiple, boolean unversioned)
    {
        this(metadata, operator, UserIdentity.stringToUserIdentity(value), multiple, unversioned);
    }
    
    /**
     * Creates the comparison Expression.
     * @param metadata the metadata path
     * @param operator the operator to make the comparison
     * @param userIdentity the user identity (login and population)
     */
    public UserExpression(String metadata, Operator operator, UserIdentity userIdentity)
    {
        this(metadata, operator, userIdentity, false, false);
    }
    
    /**
     * Creates the comparison Expression.
     * @param metadata the metadata path
     * @param operator the operator to make the comparison
     * @param userIdentity the user identity (login and population)
     * @param multiple true if the attribute is multiple
     */
    public UserExpression(String metadata, Operator operator, UserIdentity userIdentity, boolean multiple)
    {
        this(metadata, operator, userIdentity, multiple, false);
    }
    
    /**
     * Creates the comparison Expression.
     * @param metadata the metadata path
     * @param operator the operator to make the comparison
     * @param userIdentity the user identity (login and population)
     * @param multiple true if the attribute is multiple
     * @param unversioned true if the metadata is unversioned, false otherwise.
     */
    public UserExpression(String metadata, Operator operator, UserIdentity userIdentity, boolean multiple, boolean unversioned)
    {
        _user = userIdentity;
        _metadata = metadata;
        _unversioned = unversioned;
        _operator = operator;
        _multiple = multiple;
    }
    
    @Override
    public String build()
    {
        Expression expr = new AndExpression(new UserLoginExpression(_metadata, _operator, _user.getLogin(), _multiple, _unversioned), new UserPopulationExpression(_metadata, _operator, _user.getPopulationId(), _multiple, _unversioned));
        return expr.build();
    }
    
    /**
     * Internal class to create a expression based on a user's login comparison
     */
    class UserLoginExpression implements Expression
    {
        private String _uleValue;
        private Operator _uleOperator;
        
        private String _uleMetadata;
        private String _ulePath;
        private boolean _uleUnversioned;
        private boolean _uleMultiple;
        
        /**
         * Construction an expression to test a user login
         * @param metadata The path to metadata
         * @param operator The operator
         * @param value The login value
         * @param multiple true if the attribute is multiple
         * @param unversioned unversioned true if the metadata is unversioned, false otherwise.
         */
        public UserLoginExpression(String metadata, Operator operator, String value, boolean multiple, boolean unversioned)
        {
            _uleValue = value;
            _uleOperator = operator;
            
            int index = metadata.lastIndexOf('/');
            boolean isComposite = index != -1;
            _ulePath = isComposite ? metadata.substring(0, index) : null;
            _uleMetadata = isComposite ? metadata.substring(index + 1) : metadata;
            _uleUnversioned = unversioned;
            _uleMultiple = multiple;
        }
        
        @Override
        public String build()
        {
            StringBuilder buff = new StringBuilder();
            
            if (_uleUnversioned)
            {
                buff.append(RepositoryConstants.NAMESPACE_PREFIX_INTERNAL).append(":unversioned/");
            }
            
            if (_ulePath != null)
            {
                String[] parts = _ulePath.split("/");
                for (String part : parts)
                {
                    if ("*".equals(part))
                    {
                        buff.append(part).append('/');
                    }
                    else
                    {
                        buff.append(RepositoryConstants.NAMESPACE_PREFIX).append(':').append(part).append('/');
                    }
                    
                }
            }
            
            buff.append(RepositoryConstants.NAMESPACE_PREFIX).append(':').append(_uleMetadata).append('/');
            
            if (_uleMultiple)
            {
                buff.append("*/");
            }
            
            buff.append('@').append(RepositoryConstants.NAMESPACE_PREFIX).append(':').append("login");
            buff.append(" " + _uleOperator);
            buff.append(" '" + _uleValue.replaceAll("'", "''") + "'");
            
            return buff.toString();
        }
    }
    
    /**
     * Internal class to create a expression based on a user's population comparison
     */
    class UserPopulationExpression implements Expression
    {
        private String _upeValue;
        private Operator _upeOperator;
        
        private String _upeMetadata;
        private String _upePath;
        private boolean _upeUnversioned;
        private boolean _upeMultiple;
        
        /**
         * Construction an expression to test a user population
         * @param metadata The path to metadata
         * @param operator The operator
         * @param value The population's id
         * @param multiple true if the attribute is multiple
         * @param unversioned unversioned true if the metadata is unversioned, false otherwise.
         */
        public UserPopulationExpression(String metadata, Operator operator, String value, boolean multiple, boolean unversioned)
        {
            _upeValue = value;
            _upeOperator = operator;
            
            int index = metadata.lastIndexOf('/');
            boolean isComposite = index != -1;
            _upePath = isComposite ? metadata.substring(0, index) : null;
            _upeMetadata = isComposite ? metadata.substring(index + 1) : metadata;
            _upeUnversioned = unversioned;
            _upeMultiple = multiple;
        }
        
        @Override
        public String build()
        {
            StringBuilder buff = new StringBuilder();
            
            if (_upeUnversioned)
            {
                buff.append(RepositoryConstants.NAMESPACE_PREFIX_INTERNAL).append(":unversioned/");
            }
            
            if (_upePath != null)
            {
                String[] parts = _upePath.split("/");
                for (String part : parts)
                {
                    if ("*".equals(part))
                    {
                        buff.append(part).append('/');
                    }
                    else
                    {
                        buff.append(RepositoryConstants.NAMESPACE_PREFIX).append(':').append(part).append('/');
                    }
                    
                }
            }
            
            buff.append(RepositoryConstants.NAMESPACE_PREFIX).append(':').append(_upeMetadata).append('/');
            
            if (_upeMultiple)
            {
                buff.append("*/");
            }
            
            buff.append('@').append(RepositoryConstants.NAMESPACE_PREFIX).append(':').append("population");
            buff.append(" " + _upeOperator);
            buff.append(" '" + _upeValue + "'");
            
            return buff.toString();
        }
    }
    
    
}
