/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.script;

import java.util.HashMap;
import java.util.Map;

import javax.jcr.Credentials;
import javax.jcr.LoginException;
import javax.jcr.NoSuchWorkspaceException;
import javax.jcr.Repository;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.Value;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.plugins.core.ui.script.ScriptExecArguments;
import org.ametys.plugins.core.ui.script.StaticConfigurableScriptBinding;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repositoryapp.RepositoryProvider;

/**
 * Script binding that provides repository variables
 */
public class RepositoryScriptBinding extends StaticConfigurableScriptBinding
{
    private RepositoryProvider _repositoryProvider;
    private AmetysObjectResolver _ametysResolver;

    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _repositoryProvider = (RepositoryProvider) serviceManager.lookup(RepositoryProvider.ROLE);
        _ametysResolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
    }
    
    @Override
    public Map<String, Object> getVariables(ScriptExecArguments execArgs)
    {
        Map<String, Object> variables = new HashMap<>();
        Repository repository = _repositoryProvider.getRepository();
        Credentials credentials = _repositoryProvider.getCredentials();
        Session session;
        
        try
        {
            session = repository.login(credentials, "default");
        }
        catch (Exception e)
        {
            getLogger().warn("Unable to provides the JavaScript console with repository variables.", e);
            return null;
        }

        variables.put("__internal_repository_repository", new RepositoryWrapper(repository));
        variables.put("__internal_repository_session", session);
        variables.put("__internal_repository_objectResolver", _ametysResolver);
        return variables;
    }

    @Override
    public void cleanVariables(Map<String, Object> variables)
    {
        if (variables.containsKey("__internal_repository_session"))
        {
            Object session = variables.get("__internal_repository_session");
            if (session != null && session instanceof Session)
            {
                ((Session) session).logout();
                variables.remove("__internal_repository_session");
            }
        }
    }
    
    private class RepositoryWrapper implements Repository
    {
        private Repository _repo;
        
        RepositoryWrapper(Repository repo)
        {
            _repo = repo;
        }

        @Override
        public String getDescriptor(String key)
        {
            return _repo.getDescriptor(key);
        }

        @Override
        public String[] getDescriptorKeys()
        {
            return _repo.getDescriptorKeys();
        }
        
        @Override
        public Value getDescriptorValue(String key)
        {
            return _repo.getDescriptorValue(key);
        }

        @Override
        public Value[] getDescriptorValues(String key)
        {
            return _repo.getDescriptorValues(key);
        }

        @Override
        public boolean isSingleValueDescriptor(String key)
        {
            return _repo.isSingleValueDescriptor(key);
        }
        
        @Override
        public boolean isStandardDescriptor(String key)
        {
            return _repo.isStandardDescriptor(key);
        }

        @Override
        public Session login() throws LoginException, RepositoryException
        {
            return _repo.login();
        }

        @Override
        public Session login(Credentials credentials) throws LoginException, RepositoryException
        {
            return _repo.login(credentials);
        }

        @Override
        public Session login(String workspaceName) throws LoginException, NoSuchWorkspaceException, RepositoryException
        {
            return _repo.login(workspaceName);
        }

        @Override
        public Session login(Credentials credentials, String workspaceName) throws LoginException, NoSuchWorkspaceException, RepositoryException
        {
            return _repo.login(credentials, workspaceName);
        }
    }
}
