/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.script;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.RepositoryException;

import org.ametys.plugins.core.ui.script.ScriptHandler;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectIterator;

/**
 * {@link ScriptHandler} aware of {@link Node} and {@link AmetysObject}.
 */
public class RepositoryScriptHandler extends ScriptHandler
{
    @Override
    protected ResultProcessor getProcessor()
    {
        return new RepositoryResultProcessor();
    }
    
    static class RepositoryResultProcessor extends ResultProcessor
    {
        @Override
        protected Object process(Map<String, Object> results, Object scriptResult)
        {
            try
            {
                if (scriptResult instanceof Node) 
                {
                    // JCR Node
                    Node node = (Node) scriptResult;
                    return _process(node);
                } 
                else if (scriptResult instanceof NodeIterator) 
                {
                    // JCR NodeIterator
                    List<Object> nodePaths = new ArrayList<>();
                    
                    NodeIterator nodeIt = (NodeIterator) scriptResult;
                    while (nodeIt.hasNext())
                    {
                        nodePaths.add(_process(nodeIt.nextNode()));
                    }
                    
                    return nodePaths;
                }
                else if (scriptResult instanceof AmetysObject)
                {
                    // Ametys Object
                    AmetysObject object = (AmetysObject) scriptResult;
                    return _process(object);
                }
                else if (scriptResult instanceof AmetysObjectIterable<?>)
                {
                    AmetysObjectIterator<?> iterator = ((AmetysObjectIterable<?>) scriptResult).iterator();
                    List<Object> objects = new ArrayList<>();
                    
                    while (iterator.hasNext())
                    {
                        objects.add(_process(iterator.next()));
                    }
                    
                    return objects;
                }
            }
            catch (RepositoryException e)
            {
                throw new RuntimeException(e);
            }

            return super.process(results, scriptResult);
        }
        
        private Object _process(Node node) throws RepositoryException
        {
            Map<String, String> props = new HashMap<>();
            props.put("type", "node");
            props.put("path", node.getPath());
            props.put("workspace", node.getSession().getWorkspace().getName());
            
            return props;
        }
        
        private Object _process(AmetysObject object)
        {
            Map<String, String> props = new HashMap<>();
            props.put("type", "ametys-object");
            props.put("path", object.getPath());
            props.put("id", object.getId());
            
            return props;
        }
    }
}
