/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.script;

import java.lang.reflect.Array;
import java.util.List;

import javax.jcr.Node;
import javax.jcr.Property;
import javax.jcr.PropertyType;
import javax.jcr.RepositoryException;
import javax.jcr.Value;

import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;

import org.ametys.plugins.repository.data.holder.DataHolder;
import org.ametys.plugins.repository.data.holder.ModifiableModelAwareDataHolder;
import org.ametys.plugins.repository.data.holder.ModifiableModelLessDataHolder;
import org.ametys.runtime.model.exception.BadItemTypeException;
import org.ametys.runtime.model.type.ElementType;
import org.ametys.runtime.model.type.ModelItemType;

/**
 * Helper methods to manipulate some objects in the script tool. Arrays and lists are not managed the same way depending of JS engine used in the JVM.
 */
public final class RepositoryScriptHelper
{
    private RepositoryScriptHelper()
    {
        // Utility class
    }
    
    /**
     * Helper to set ambiguous object values from a node property
     * @param node the node
     * @param name the property name
     * @param values the property values as String[] or Value[]
     * @throws RepositoryException if an error occurred
     */
    public static void setProperty(Node node, String name, Object values) throws RepositoryException
    {
        Object javaValues = values;
        if (values instanceof List)
        {
            List<?> list = (List<?>) values;
            if (list.isEmpty())
            {
                javaValues = new Value[0];
            }
            else if (list.get(0) instanceof String)
            {
                javaValues = list.toArray(new String[list.size()]);
            }
            else if (list.get(0) instanceof Value)
            {
                javaValues = list.toArray(new Value[list.size()]);
            }
        }
        _setProperty(node, name, javaValues);
    }
    
    private static void _setProperty(Node node, String name, Object values) throws RepositoryException
    {
        if (values instanceof String[])
        {
            node.setProperty(name, (String[]) values);
        }
        else if (values instanceof Value[])
        {
            node.setProperty(name, (Value[]) values);
        }
        else if (values instanceof String)
        {
            node.setProperty(name, (String) values);
        }
        else if (values instanceof Boolean)
        {
            node.setProperty(name, (Boolean) values);
        }
        else if (values instanceof Long)
        {
            node.setProperty(name, (Long) values);
        }
        else if (values instanceof Integer)
        {
            node.setProperty(name, ((Integer) values).longValue());
        }
        else if (values instanceof Double)
        {
            node.setProperty(name, (Double) values);
        }
        else if (values instanceof Float)
        {
            node.setProperty(name, ((Float) values).doubleValue());
        }
        else
        {
            Class< ? > clazz = values.getClass();
            throw new IllegalArgumentException("argument values (" + values + ") is of unsupported type by RepositoryScriptHelper#setProperty: '" + clazz + "'");
        }
    }
    
    /**
     * Helper to convert a single-valued property to a multi-valued property.
     * This helper checks that property exists and that it is not already multiple.
     * @param node the node holding the property
     * @param propertyName the property's name
     * @return true if changes was made
     * @throws RepositoryException if an error occurred
     */
    public static boolean convertSingleToMultipleProperty (Node node, String propertyName) throws RepositoryException
    {
        boolean needSave = false;
        if (node.hasProperty(propertyName))
        {
            Property property = node.getProperty(propertyName);
            if (!property.getDefinition().isMultiple())
            {
                Value value = property.getValue();
                if (value.getType() == PropertyType.STRING)
                {
                    String valueAsStr = value.getString();
                    property.remove();
                    
                    if (!StringUtils.isEmpty(valueAsStr))
                    {
                        String[] strArray = value.getString().split(",");
                        node.setProperty(propertyName, strArray);
                    }
                    else
                    {
                        node.setProperty(propertyName, new String[0]);
                    }
                }
                else
                {
                    Value[] values = ArrayUtils.toArray(value);
                    node.setProperty(propertyName, values);
                }
                
                needSave = true;
            }
        }
        
        return needSave;
    }
    
    /**
     * Set the value of a {@link ModifiableModelLessDataHolder}.
     * If a multiple value is provided as a {@link List} instead of an array, this method will try to convert it into
     * an array based on the available {@link ElementType} of the {@link DataHolder}.
     * 
     * @param dataHolder the data holder to update
     * @param name the attribute name to update
     * @param value the value
     * @param type the type of the data
     */
    @SuppressWarnings("unchecked")
    public static void setModelLessValue(ModifiableModelLessDataHolder dataHolder, String name, Object value, String type)
    {
        if (value instanceof List list)
        {
            ModelItemType modelItemType = dataHolder.getModelItemTypeExtensionPoint().getExtension(type);
            if (modelItemType == null)
            {
                throw new BadItemTypeException("Type '" + type + "'does not exist for the given DataHolder (" + dataHolder.getRepositoryData().toString() + "). Impossible to use setValue with it.");
            }
            else if (modelItemType instanceof ElementType elementType)
            {
                Class managedClass = elementType.getManagedClass();
                Object[] array =  (Object[]) Array.newInstance(managedClass, list.size());
                dataHolder.setValue(name, list.toArray(array), type);
            }
            else
            {
                throw new BadItemTypeException("Type '" + type + "'is not associated with an ElementType. Impossible to use setValue with it.");
            }
        }
        else
        {
            dataHolder.setValue(name, value, type);
        }
    }
    
    /**
     * Set the value of a {@link ModifiableModelAwareDataHolder}.
     * If a multiple value is provided as a {@link List} instead of an array, this method will try to convert it into
     * an array based on the managed class of the type of data at path name.
     * 
     * @param dataHolder the data holder to update
     * @param name the attribute name to update
     * @param value the value
     */
    @SuppressWarnings("unchecked")
    public static void setModelAwareValue(ModifiableModelAwareDataHolder dataHolder, String name, Object value)
    {
        if (value instanceof List list)
        {
            ModelItemType itemType = dataHolder.getType(name);
            if (itemType instanceof ElementType elementType)
            {
                Class managedClass = elementType.getManagedClass();
                Object[] array =  (Object[]) Array.newInstance(managedClass, list.size());
                dataHolder.setValue(name, list.toArray(array));
            }
            else
            {
                throw new BadItemTypeException("Item at path '" + name + "' is not associated with an ElementType. Impossible to use setValue with it.");
            }
        }
        else
        {
            dataHolder.setValue(name, value);
        }
    }
    
}
