/*
 *  Copyright 2012 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.tag;

import java.util.Collections;
import java.util.HashSet;
import java.util.Set;

import javax.jcr.Node;
import javax.jcr.PathNotFoundException;
import javax.jcr.RepositoryException;
import javax.jcr.Value;
import javax.jcr.ValueFactory;

import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.jcr.DefaultAmetysObject;
import org.ametys.plugins.repository.jcr.JCRAmetysObject;

/**
 * Helper class which provides methods to manage tags on {@link JCRAmetysObject}s.
 */
public final class TaggableAmetysObjectHelper
{
    /** Constant for tags metadata. */
    public static final String DEFAULT_METADATA_TAGS = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":tags";
    
    private TaggableAmetysObjectHelper()
    {
        // Hide the default constructor.
    }
    
    /**
     * Retrieve the tags of a {@link JCRAmetysObject}.
     * @param object the {@link JCRAmetysObject} to tag.
     * @return the object tags.
     * @throws AmetysRepositoryException if failed to tag Ametys object
     */
    public static Set<String> getTags(JCRAmetysObject object) throws AmetysRepositoryException
    {
        return getTags(object, DEFAULT_METADATA_TAGS);
    }
    
    /**
     * Retrieve the tags of a {@link JCRAmetysObject}.
     * @param object the {@link JCRAmetysObject} to tag.
     * @param metadataName The name of the metadata holding the tags
     * @return the object tags.
     * @throws AmetysRepositoryException if failed to tag Ametys object
     */
    public static Set<String> getTags(JCRAmetysObject object, String metadataName) throws AmetysRepositoryException
    {        
        try
        {
            Node node = getNode(object);
            Value[] values = node.getProperty(metadataName).getValues();
            Set<String> tags = new HashSet<>(values.length);
            
            for (Value value : values)
            {
                tags.add(value.getString());
            }
            
            return tags;
        }
        catch (PathNotFoundException e)
        {
            return Collections.emptySet();
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get tags property " + metadataName, e);
        }
    }
        
    /**
     * Tag a {@link JCRAmetysObject}.
     * @param object the {@link JCRAmetysObject} to tag.
     * @param tag the tag to put on the object.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public static void tag(JCRAmetysObject object, String tag) throws AmetysRepositoryException
    {
        tag(object, tag, DEFAULT_METADATA_TAGS);
    }
    
    /**
     * Tag a {@link JCRAmetysObject}.
     * @param object the {@link JCRAmetysObject} to tag.
     * @param tag the tag to put on the object.
     * @param metadataName The name of the metadata holding the tags
     * @throws AmetysRepositoryException if an error occurs.
     */
    public static void tag(JCRAmetysObject object, String tag, String metadataName) throws AmetysRepositoryException
    {
        try
        {
            Node node = getNode(object);
            Value[] values = new Value[0];
            
            if (node.hasProperty(metadataName))
            {
                values = node.getProperty(metadataName).getValues();
            }
            
            Set<String> tags = new HashSet<>(values.length + 1);
            
            for (Value value : values)
            {
                tags.add(value.getString());
            }
            
            if (tags.add(tag))
            {
                ValueFactory valueFactory = node.getSession().getValueFactory();
                Set<Value> newValues = new HashSet<>(tags.size());
                
                for (String currentTag : tags)
                {
                    newValues.add(valueFactory.createValue(currentTag));
                }
                
                node.setProperty(metadataName, newValues.toArray(new Value[newValues.size()]));
            }
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to set tags property", e);
        }
    }
    
    /**
     * Remove a tag on a {@link JCRAmetysObject}.
     * @param object the {@link JCRAmetysObject} to tag.
     * @param tag the tag to remove on the object.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public static void untag(JCRAmetysObject object, String tag) throws AmetysRepositoryException
    {
        untag(object, tag, DEFAULT_METADATA_TAGS);
    }
    
    /**
     * Remove a tag on a {@link JCRAmetysObject}.
     * @param object the {@link JCRAmetysObject} to tag.
     * @param tag the tag to remove on the object.
     * @param metadataName The name of the metadata holding the tags
     * @throws AmetysRepositoryException if an error occurs.
     */
    public static void untag(JCRAmetysObject object, String tag, String metadataName) throws AmetysRepositoryException
    {
        try
        {
            Node node = getNode(object);
            Value[] values = new Value[0];
            
            if (node.hasProperty(metadataName))
            {
                values = node.getProperty(metadataName).getValues();
            }
            
            Set<String> tags = new HashSet<>(values.length + 1);
            
            for (Value value : values)
            {
                tags.add(value.getString());
            }
            
            if (tags.remove(tag))
            {
                ValueFactory valueFactory = node.getSession().getValueFactory();
                Set<Value> newValues = new HashSet<>(tags.size());
                
                for (String currentTag : tags)
                {
                    newValues.add(valueFactory.createValue(currentTag));
                }
                
                node.setProperty(metadataName, newValues.toArray(new Value[newValues.size()]));
            }
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to set tags property", e);
        }
    }
    
    /**
     * Get an object's base node.
     * @param object the object which node to get.
     * @return the object's base node.
     */
    protected static Node getNode(JCRAmetysObject object)
    {
        Node node = null;
        
        if (object instanceof DefaultAmetysObject)
        {
            node = ((DefaultAmetysObject) object).getBaseNode();
        }
        else
        {
            node = object.getNode();
        }
        
        return node;
    }
    
}
